/***
Demonstrates SPI communication.

You can short MOSI to MISO for testing.

MOSI    FIO2
MISO    FIO3
CLK     FIO0
CS      FIO1

If you short MISO to MOSI, then you will read back the same bytes that you
write.  If you short MISO to GND, then you will read back zeros.  If you
short MISO to VS or leave it unconnected, you will read back 255s.

 ***/
import java.util.Random;
import com.sun.jna.ptr.IntByReference;
import com.labjack.LJM;
import com.labjack.LJMException;

public class SPI {

	public static void main(final String[] args) {

		try {
			IntByReference handleRef = new IntByReference(0);
			int handle = 0;
			IntByReference errAddrRef = new IntByReference(-1);


			//Open first found LabJack
			LJM.openS("ANY", "ANY", "ANY", handleRef);
			//LJM.open(LJM.Constants.dtANY, LJM.Constants.ctANY, "ANY", handleRef);
			handle = handleRef.getValue();

			LJMUtilities.printDeviceInfo(handle);

			
			//CS is FIO1
			LJM.eWriteName(handle, "SPI_CS_DIONUM", 1);

			//CLK is FIO0
			LJM.eWriteName(handle, "SPI_CLK_DIONUM", 0);

			//MISO is FIO3
			LJM.eWriteName(handle, "SPI_MISO_DIONUM", 3);

			//MOSI is FIO2
			LJM.eWriteName(handle, "SPI_MOSI_DIONUM", 2);

			//Modes:
			//0 = A: CPHA=0, CPOL=0 
			//    Data clocked on the rising edge
			//    Data changed on the falling edge
			//    Final clock state low
			//    Initial clock state low
			//1 = B: CPHA=0, CPOL=1
			//    Data clocked on the falling edge
			//    Data changed on the rising edge
			//    Final clock state low
			//    Initial clock state low
			//2 = C: CPHA=1, CPOL=0 
			//    Data clocked on the falling edge
			//    Data changed on the rising edge
			//    Final clock state high
			//    Initial clock state high
			//3 = D: CPHA=1, CPOL=1 
			//    Data clocked on the rising edge
			//    Data changed on the falling edge
			//    Final clock state high
			//    Initial clock state high

			//Selecting Mode: A - CPHA=1, CPOL=1.
			LJM.eWriteName(handle, "SPI_MODE", 0);

			//Speed Throttle:
			//Frequency = 1000000000 / (175*(65536-SpeedThrottle) + 1020)
			//Valid speed throttle values are 1 to 65536 where 0 = 65536.
			//Note: The above equation and its frequency range were tested for
			//firmware 1.0009 and may change in the future.

			//Configuring Max. Speed (~ 1 MHz)
			LJM.eWriteName(handle, "SPI_SPEED_THROTTLE", 0);

			//Options
			//bit 0:
			//    0 = Active low clock select enabled
			//    1 = Active low clock select disabled.
			//bit 1:
			//    0 = DIO directions are automatically changed
			//    1 = DIO directions are not automatically changed.
			//bits 2-3: Reserved
			//bits 4-7: Number of bits in the last byte. 0 = 8.
			//bits 8-15: Reserved

			//Enabling active low clock select pin
			LJM.eWriteName(handle, "SPI_OPTIONS", 0);

			//Read back and display the SPI settings
			String[] aNames = {"SPI_CS_DIONUM", "SPI_CLK_DIONUM",
					"SPI_MISO_DIONUM", "SPI_MOSI_DIONUM",
					"SPI_MODE", "SPI_SPEED_THROTTLE",
					"SPI_OPTIONS"};
			double[] aValues = new double[aNames.length];
			LJM.eReadNames(handle, aNames.length, aNames, aValues, errAddrRef);

			System.out.println("SPI Configuration:");
			for(int i = 0; i < aNames.length; i++) {
				System.out.println("  " + aNames[i] + " = " + aValues[i]);
			}


			//Write/Read 4 bytes
			int numBytes = 4;
			LJM.eWriteName(handle, "SPI_NUM_BYTES", numBytes);


			//Setup write bytes
			double[] dataWrite = new double[numBytes];
			Random rand = new Random();
			for(int i = 0; i < numBytes; i++) {
				dataWrite[i] = rand.nextInt(255);
			}

			aNames = new String[1];
			int[] aWrites = new int[1];
			int[] aNumValues = new int[1];

			//Write the bytes
			aNames[0] = "SPI_DATA_WRITE";
			aWrites[0] = LJM.Constants.WRITE;
			aNumValues[0] = numBytes;
			LJM.eNames(handle, 1, aNames, aWrites, aNumValues, dataWrite, errAddrRef);

			//Display the bytes written
			System.out.println("");
			for(int i = 0; i < numBytes; i++) {
				System.out.println("dataWrite[" + i + "] = "
						+ String.format("%.0f", dataWrite[i]));
			}


			//Read the bytes
			double[] dataRead = new double[numBytes];
			aNames[0] = "SPI_DATA_READ";
			aWrites[0] = LJM.Constants.READ;
			aNumValues[0] = numBytes;
			LJM.eNames(handle, 1, aNames, aWrites, aNumValues, dataRead, errAddrRef);

			//Display the bytes read
			System.out.println("");
			for(int i = 0; i < numBytes; i++) {
				System.out.println("dataRead[" + i + "] = "
						+ String.format("%.0f", dataRead[i]));
			}


			//Close handle
			LJM.close(handle);
		}
		catch (LJMException le) {
			le.printStackTrace();
			LJM.closeAll();
		}
	}
}