/***
Demonstrates how to stream a range of sequential analog inputs using the eStream
functions. Useful when streaming many analog inputs. AIN channel scan list is
FIRST_AIN_CHANNEL to FIRST_AIN_CHANNEL + NUMBER_OF_AINS - 1.

 ***/
import java.io.DataInputStream;
import com.sun.jna.ptr.IntByReference;
import com.sun.jna.ptr.DoubleByReference;
import com.labjack.LJM;
import com.labjack.LJMException;

public class StreamSequentialAIN {

	public static void main(final String[] args) {

		try {
			final int FIRST_AIN_CHANNEL = 0;  //AIN0
			final int NUMBER_OF_AINS = 16;
			
			IntByReference handleRef = new IntByReference(0);
			int handle = 0;

		    //If arguments are passed to the application, the first argument
			//specfies how many times to loop. If an argument is not passed,
			//will loop until the Enter key is pressed.
			long loopAmount = 0;
			boolean infLoop = true;
			if(args.length > 0) {
				try {
					loopAmount = new Double(args[0]).longValue();
					infLoop = false;
				}
				catch(Exception ex) {
					throw new Exception("Invalid first argument \"" + args[0] +
							"\". This specifies how many times to loop and " +
							"needs to be a number.");
				}
			}
			
			//Open first found LabJack
			LJM.openS("ANY", "ANY", "ANY", handleRef);
			//LJM.open(LJM.Constants.dtANY, LJM.Constants.ctANY, "ANY", handleRef);
			handle = handleRef.getValue();

			LJMUtilities.printDeviceInfo(handle);

			//Stream Configuration
			int scansPerRead = 1000; //# scans returned by eStreamRead call
			//Scan list names to stream. AIN(FIRST_AIN_CHANNEL) to AIN(NUMBER_OF_AINS-1).
			String[] aScanListNames = new String[NUMBER_OF_AINS];
			for(int i = 0; i < aScanListNames.length; i++) {
				aScanListNames[i] = "AIN" + String.valueOf(FIRST_AIN_CHANNEL+i);
			}
			int numAddresses = aScanListNames.length;
			int[] aTypes = new int[numAddresses]; //Dummy
			int[] aScanList = new int[numAddresses]; //Scan list addresses to stream. eStreamStart uses Modbus addresses.
			LJM.namesToAddresses(numAddresses, aScanListNames, aScanList, aTypes);
			double scanRate = 1000; //Scans per second

			try {
				//Configure the analog inputs' negative channel, range, settling time and
				//resolution.
				//Note when streaming, negative channels and ranges can be configured for
				//individual analog inputs, but the stream has only one settling time and
				//resolution.
				String[] aNames = new String[] { "AIN_ALL_NEGATIVE_CH", "AIN_ALL_RANGE",
					"STREAM_SETTLING_US", "STREAM_RESOLUTION_INDEX" };
				double[] aValues = new double[] { LJM.Constants.GND, 10.0, 0, 0 };  //single-ended, +/-10V, 0 (default), 0 (default)
				IntByReference errAddrRef = new IntByReference(-1);
				LJM.eWriteNames(handle, aNames.length, aNames, aValues, errAddrRef);

				System.out.println("\nStarting stream."
						+ " Press Enter to stop streaming.");
				DataInputStream ins = new DataInputStream(System.in);
				Thread.sleep(1000); //Delay so user's can read message

				//Configure and start Stream
				DoubleByReference scanRateRef = new DoubleByReference(scanRate);
				LJM.eStreamStart(handle, scansPerRead, numAddresses, aScanList, scanRateRef);
				scanRate = scanRateRef.getValue();

				long loop = 0;
				long totScans = 0;
				double[] aData = new double[scansPerRead*numAddresses]; //# of samples per eStreamRead is scansPerRead * numAddresses
				long skippedTotal = 0;
				int skippedCur = 0;
				int deviceScanBacklog = 0;
				int ljmScanBacklog = 0;
				IntByReference deviceScanBacklogRef = new IntByReference(0);
				IntByReference ljmScanBacklogRef = new IntByReference(0);

				System.out.println("Starting read loop.");
				long stTime = System.nanoTime();

				while(ins.available() <= 0) {
					LJM.eStreamRead(handle, aData, deviceScanBacklogRef, ljmScanBacklogRef);
					deviceScanBacklog = deviceScanBacklogRef.getValue();
					ljmScanBacklog = ljmScanBacklogRef.getValue();

					totScans += scansPerRead;

					//Count the skipped samples which are indicated by -9999
					//values. Missed samples occur after a device's stream
					//buffer overflows and are reported after auto-recover mode
					//ends.
					skippedCur = 0;
					for(int i = 0; i < aData.length; i++) {
						if(aData[i] == -9999.00)
							skippedCur++;
					}
					skippedTotal += skippedCur;
					loop++;
					System.out.println("\neStreamRead " + loop);
					System.out.print("  First scan out of "
							+ scansPerRead + ": ");
					for(int j = 0; j < numAddresses; j++) {
						System.out.print(aScanListNames[j] + " = "
								+ String.format("%.4f", aData[j]) + ", ");
					}
					System.out.println("\n  numSkippedScans: "
							+ skippedCur/numAddresses + ", deviceScanBacklog: "
							+ deviceScanBacklog + ", ljmScanBacklog: "
							+ ljmScanBacklog);
					
					//Stop after the user specified number of loops.
					if(!infLoop && loop >= loopAmount) {
						break;
					}
				}

				long enTime = System.nanoTime();

				System.out.println("\nTotal scans: " + totScans);
				System.out.println("Skipped scans: "
						+ skippedTotal/numAddresses);
				double time = (enTime - stTime)/1000000000.0; //in seconds
				System.out.println("Time taken: " + String.format("%.3f", time)
						+ " seconds");
				System.out.println("LJM Scan Rate: " + scanRate
						+ " scans/second");
				System.out.println("Timed Scan Rate: "
						+ String.format("%.3f", (totScans/time))
						+ " scans/second");
				System.out.println("Sample Rate: "
						+ String.format("%.3f", (totScans*numAddresses/time))
						+ " samples/second");
			}
			catch(LJMException le) {
				le.printStackTrace();
			}
			catch(Exception e) {
				e.printStackTrace();
			}

			System.out.println("\nStop Stream");
			LJM.eStreamStop(handle);

			//Close handle
			LJM.close(handle);
		}
		catch (LJMException le) {
			le.printStackTrace();
			LJM.closeAll();
		}
		catch(Exception e) {
			e.printStackTrace();
			LJM.closeAll();
		}
	}
}
