/***
Demonstrates reading 2 analog inputs (AINs) in a loop from a LabJack.

 ***/
import java.io.DataInputStream;
import com.sun.jna.ptr.IntByReference;
import com.labjack.LJM;
import com.labjack.LJMException;

public class DualAINLoop {
	
	public static void main(final String[] args) {
		try {
			IntByReference handleRef = new IntByReference(0);
			int handle = 0;
		    
			//If arguments are passed to the application, the first argument
			//specfies how many times to loop. If an argument is not passed,
			//will loop until the Enter key is pressed.
			int loopAmount = 0;
			boolean infLoop = true;
			String loopStr = "\nStarting read loop. Press Enter to stop.";
			if(args.length > 0) {
				try {
					loopAmount = new Double(args[0]).intValue();
					infLoop = false;
					loopStr = "\nStarting " + loopAmount + " read loops.";
				}
				catch(Exception ex) {
					throw new Exception("Invalid first argument \"" + args[0] +
							"\". This specifies how many times to loop and " +
							"needs to be a number.");
				}
			}
			
			//Open first found LabJack
			LJM.openS("ANY", "ANY", "ANY", handleRef);
			//LJM.open(LJM.Constants.dtANY, LJM.Constants.ctANY, "ANY", handleRef);
			handle = handleRef.getValue();

			LJMUtilities.printDeviceInfo(handle);

			//Setup and call eWriteNames to configure AINs.
			//Setting AIN0-1 Negative Channel to 199 (Single-ended), Range to
			//+-10 V, Resolution index to 0 (default: index 8 or 9 for Pro) and
			//Settling to 0 (automatic)
			String[] aNames = {"AIN0_NEGATIVE_CH", "AIN0_RANGE",
					"AIN0_RESOLUTION_INDEX", "AIN1_NEGATIVE_CH",
					"AIN1_RANGE", "AIN1_RESOLUTION_INDEX"};
			double[] aValues = {199, 10,
					0, 199,
					10, 0};
			int numFrames = aNames.length;
			IntByReference errAddr = new IntByReference(-1);
			LJM.eWriteNames(handle, numFrames, aNames, aValues, errAddr);

			System.out.println("\nSet configuration:");
			for(int i = 0; i < numFrames; i++) {
				System.out.println("  " + aNames[i] + " : " + aValues[i]);
			}

			//Setup and call eReadNames to read AINs.
			aNames = new String[] {"AIN0", "AIN1"};
			aValues = new double[] {0, 0};
			numFrames = aNames.length;
			
			System.out.println(loopStr);
			DataInputStream ins = new DataInputStream(System.in);
			int it = 0;
			while(ins.available() <= 0) {
				LJM.eReadNames(handle, numFrames, aNames, aValues, errAddr);
				System.out.print("\n");
				for(int i = 0; i < numFrames; i++) {
					System.out.print(aNames[i] + " = "
							+ String.format("%.4f", aValues[i]) + " V");
					if(i < numFrames - 1) {
						System.out.print(", ");
					}
				}
				System.out.print("\n");

				//Stop after the user specified number of loops.
				it++;
				if(!infLoop && it >= loopAmount) {
					break;
				}

				Thread.sleep(1000); //Wait 1 second
			}

			//Close handle
			LJM.close(handle);
		}
		catch(LJMException le) {
			le.printStackTrace();
			LJM.closeAll();
		}
		catch(Exception e) {
			e.printStackTrace();
			LJM.closeAll();
		}
	}
}
