/***
Demonstrates setting up stream-in and stream-out together, then reading
stream-in values.

Connect a wire from AIN0 to DAC0 to see the effect of stream-out on stream-in
channel 0.

 ***/
import java.io.DataInputStream;
import java.util.Arrays;

import com.sun.jna.ptr.IntByReference;
import com.sun.jna.ptr.DoubleByReference;
import com.labjack.LJM;
import com.labjack.LJMException;

public class OStreamRun {

	public static void main(final String[] args) {

		try {
			final int MAX_REQUESTS = 20;  //The number of eStreamRead calls that will be performed.

			IntByReference handleRef = new IntByReference(0);
			int handle = 0;

			//Open first found LabJack
			LJM.openS("ANY", "ANY", "ANY", handleRef);
			//LJM.open(LJM.Constants.dtANY, LJM.Constants.ctANY, "ANY", handleRef);
			handle = handleRef.getValue();

			LJMUtilities.printDeviceInfo(handle);

			//Setup Stream Out
			String[] outNames = new String[] {"DAC0"};
			int numAddressesOut = outNames.length;
			int[] outAddresses = new int[numAddressesOut];
			int[] aTypes = new int[numAddressesOut]; //Dummy
			LJM.namesToAddresses(numAddressesOut, outNames, outAddresses, aTypes);

			//Allocate memory for the stream-out buffer
			LJM.eWriteName(handle, "STREAM_OUT0_TARGET", outAddresses[0]);
			LJM.eWriteName(handle, "STREAM_OUT0_BUFFER_SIZE", 512);
			LJM.eWriteName(handle, "STREAM_OUT0_ENABLE", 1);

			//Write values to the stream-out buffer
			LJM.eWriteName(handle, "STREAM_OUT0_LOOP_SIZE", 6);
			LJM.eWriteName(handle, "STREAM_OUT0_BUFFER_F32", 0.0);  //0.0 V
			LJM.eWriteName(handle, "STREAM_OUT0_BUFFER_F32", 1.0);  //1.0 V
			LJM.eWriteName(handle, "STREAM_OUT0_BUFFER_F32", 2.0);  //2.0 V
			LJM.eWriteName(handle, "STREAM_OUT0_BUFFER_F32", 3.0);  //3.0 V
			LJM.eWriteName(handle, "STREAM_OUT0_BUFFER_F32", 4.0);  //4.0 V
			LJM.eWriteName(handle, "STREAM_OUT0_BUFFER_F32", 5.0);  //5.0 V

			LJM.eWriteName(handle, "STREAM_OUT0_SET_LOOP", 1);

			DoubleByReference valueRef = new DoubleByReference(0);
			LJM.eReadName(handle, "STREAM_OUT0_BUFFER_STATUS", valueRef);
			System.out.println("\nSTREAM_OUT0_BUFFER_STATUS = " + valueRef.getValue());
			
			//Stream Configuration
			double scanRate = 2000; //Scans per second
			int scansPerRead = 60; //# scans returned by eStreamRead call

			String[] aScanListNames = {"AIN0", "AIN1"}; //Input scan list names to stream.
			int numAddressesIn = aScanListNames.length;
			aTypes = new int[numAddressesIn]; //Dummy
			int[] aScanList = new int[numAddressesIn+numAddressesOut]; //Scan list addresses to stream. eStreamStart uses Modbus addresses.
			LJM.namesToAddresses(numAddressesIn, aScanListNames, aScanList, aTypes); //Get the scan list's inputs
			
			//Add the scan list outputs to the end of the scan list.
			//STREAM_OUT0 = 4800, STREAM_OUT1 = 4801, ...
			aScanList[numAddressesIn] = 4800; //STREAM_OUT0
			//If we had more STREAM_OUTs
			//aScanList[numAddressesIn+1] = 4801;  //STREAM_OUT1
			//etc.
			
			try {
				//Configure the analog inputs' negative channel, range, settling time and
				//resolution.
				//Note when streaming, negative channels and ranges can be configured for
				//individual analog inputs, but the stream has only one settling time and
				//resolution.
				String[] aNames = new String[] { "AIN_ALL_NEGATIVE_CH", "AIN_ALL_RANGE",
					"STREAM_SETTLING_US", "STREAM_RESOLUTION_INDEX" };
				double[] aValues = new double[] { LJM.Constants.GND, 10.0, 0, 0 };  //single-ended, +/-10V, 0 (default), 0 (default)
				IntByReference errAddrRef = new IntByReference(-1);
				LJM.eWriteNames(handle, aNames.length, aNames, aValues, errAddrRef);

				DataInputStream ins = new DataInputStream(System.in);
				System.out.println("Scan List = " + Arrays.toString(aScanList));

				//Configure and start Stream
				DoubleByReference scanRateRef = new DoubleByReference(scanRate);
				LJM.eStreamStart(handle, scansPerRead, aScanList.length, aScanList, scanRateRef);
				scanRate = scanRateRef.getValue();
				System.out.println("Stream started with a scan rate of "
						+ scanRate + " Hz.");
				System.out.println("Performing " + MAX_REQUESTS + " stream reads.");
				
				long totScans = 0;
				double[] aData = new double[scansPerRead*numAddressesIn]; //# of samples per eStreamRead is scansPerRead * numAddressesIn
				long skippedTotal = 0;
				int skippedCur = 0;
				int deviceScanBacklog = 0;
				int ljmScanBacklog = 0;
				IntByReference deviceScanBacklogRef = new IntByReference(0);
				IntByReference ljmScanBacklogRef = new IntByReference(0);

				long stTime = System.nanoTime();

				for(int loop = 0; loop < MAX_REQUESTS; loop++) {
					LJM.eStreamRead(handle, aData, deviceScanBacklogRef, ljmScanBacklogRef);
					deviceScanBacklog = deviceScanBacklogRef.getValue();
					ljmScanBacklog = ljmScanBacklogRef.getValue();

					totScans += scansPerRead;

					//Count the skipped samples which are indicated by -9999
					//values. Missed samples occur after a device's stream
					//buffer overflows and are reported after auto-recover mode
					//ends.
					skippedCur = 0;
					for(int i = 0; i < aData.length; i++) {
						if(aData[i] == -9999.00)
							skippedCur++;
					}
					skippedTotal += skippedCur;
					System.out.println("\neStreamRead #" + (loop+1));
					for(int j = 0; j < scansPerRead; j++) {
						for(int k = 0; k < numAddressesIn; k++) {
							System.out.print("  " + aScanListNames[k] + " = "
									+ String.format("%.4f", aData[j*numAddressesIn + k]) + ",");
						}
						System.out.println("");
					}
					System.out.println("  Skipped Scans = "
							+ skippedCur/numAddressesIn + ", Scan Backlogs: Device = "
							+ deviceScanBacklog + ", LJM = " + ljmScanBacklog);
				}

				long enTime = System.nanoTime();

				System.out.println("\nTotal scans: " + totScans);
				System.out.println("Skipped scans: "
						+ skippedTotal/numAddressesIn);
				double time = (enTime - stTime)/1000000000.0; //in seconds
				System.out.println("Time taken: " + String.format("%.3f", time)
						+ " seconds");
				System.out.println("LJM Scan Rate: " + scanRate
						+ " scans/second");
				System.out.println("Timed Scan Rate: "
						+ String.format("%.3f", (totScans/time))
						+ " scans/second");
				System.out.println("Sample Rate: "
						+ String.format("%.3f", (totScans*numAddressesIn/time))
						+ " samples/second");
			}
			catch(LJMException le) {
				le.printStackTrace();
			}
			catch(Exception e) {
				e.printStackTrace();
			}

			System.out.println("\nStop Stream");
			LJM.eStreamStop(handle);

			//Close handle
			LJM.close(handle);
		}
		catch (LJMException le) {
			le.printStackTrace();
			LJM.closeAll();
		}
		catch(Exception e) {
			e.printStackTrace();
			LJM.closeAll();
		}
	}
}
