/**
 * This file is part of LabJackJava.  Copyright 2003-2004, Teravation.  All rights reserved. 
 * 
 * LabJackJava is free software; you can redistribute it and/or modify it under the terms of the GNU
 * General Public License as published by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 * 
 * LabJackJava is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; 
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with LabJackJava;
 * if not, write to:
 *    Free Software Foundation, Inc.
 *    59 Temple Place, Suite 330
 *    Boston, MA  02111-1307  USA
 * 
 * In addition, as a special exception, Teravation gives permission to link the code of this program with
 * the LabJack C driver (or with modified versions of the LabJack C driver that use the same license),
 * and distribute linked combinations including the two.  You must obey the GNU General Public License in all
 * respects for all of the code used other than the LabJack C driver.  If you modify this file, you may 
 * extend this exception to your version of the file, but you are not obligated to do so.  If you do not wish to
 * do so, delete this exception statement from your version.
 * 
 */

/**
 * This is the underlying JNI code for the LabJackDriver class.  It is dependent upon a platform-specific 
 * implementation of the LabJack C driver:
 *    Windows:  ljackuw.dll
 *    Mac OSX:  libljackmacx.dylib
 *    Linux:    liblabjack.so
 *
 * <p />Copyright &copy; 2003-2004 <a href="http://www.teravation.com">Teravation</a>. All rights reserved.
 * 
 * @created (12/30/2002 2:52:36 PM)
 * @version 4.0
 * @author Chris Reigrut 
 */
#include <jni.h>
#include "com_teravation_labjack_LabJackDriver.h"

#ifdef WINDOWS /* {{ */
#include <wtypes.h>
#endif /* }} */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

#ifdef WINDOWS /* {{ */
#include "drivers/windows/ljackuw.h"
#endif /* }} */
#ifdef LINUX /* {{ */
#include "ljackv112.h"
#endif /* }} */
#ifdef MACOSX /* {{ */
#include "drivers/macosx/ljackmacx.h"
#endif /* }} */

#define DEBUG true
#define NODEMO 0

#ifdef WINDOWS /* {{ */
#define LJACK_DLL "ljackuw"
HINSTANCE hLabJackDLL;            // Handle to DLL

// Function templates (prototypes) for ljackuw.dll
typedef long (CALLBACK *FT_LISTALL) (long *, long *, long *, long *, long (*)[20], long *, long *, long *);
typedef long (CALLBACK *FT_EANALOGIN) (long *, long, long, long, long *, float *);
typedef long (CALLBACK *FT_EANALOGOUT) (long *, long, float, float);
typedef long (CALLBACK *FT_EDIGITALIN) (long *, long, long, long, long *);
typedef long (CALLBACK *FT_EDIGITALOUT) (long *, long, long, long, long);
typedef long (CALLBACK *FT_ECOUNT) (long *, long, long, double *, double *);
typedef long (CALLBACK *FT_AOUPDATE) (long *, long, long, long, long *, long *, long, long, unsigned long *, float, float);
typedef long (CALLBACK *FT_AISAMPLE) (long *, long, long *, long, long, long, long *, long *, long, long *, float *);
typedef long (CALLBACK *FT_AIBURST)(long *, long, long, long, long, long, long *, long *, float *, long, long, long, long, long, float (*)[4], long *, long *, long);
typedef long (CALLBACK *FT_SHT1X) (long *, long, long, long, long, float *, float *, float *);
typedef long (CALLBACK *FT_NOTHREAD) (long *, long);
typedef long (CALLBACK *FT_PULSEOUT) (long *, long, long, long, long, long, long, long, long);
typedef long (CALLBACK *FT_PULSEOUTSTART) (long *, long, long, long, long, long, long, long, long);
typedef long (CALLBACK *FT_PULSEOUTFINISH) (long *, long, long);
typedef long (CALLBACK *FT_REENUM) (long *);
typedef long (CALLBACK *FT_RESET) (long *);
typedef float (CALLBACK *FT_GETFIRMWAREVERSION) (long *);
typedef float (CALLBACK *FT_GETDRIVERVERSION) ();
typedef long (CALLBACK *FT_LOCALID) (long *, long);
typedef long (CALLBACK *FT_AISTREAMSTART) (long *, long, long, long, long, long, long *, long *, float *, long, long, long);
typedef long (CALLBACK *FT_AISTREAMREAD) (long, long, long, float (*)[4], long *, long *, long *, long *);
typedef long (CALLBACK *FT_AISTREAMCLEAR) (long);

FT_LISTALL fpListAll;                         // Function pointer
FT_EANALOGIN fpEAnalogIn;                     // Function pointer
FT_EANALOGOUT fpEAnalogOut;                   // Function pointer
FT_EDIGITALIN fpEDigitalIn;                   // Function pointer
FT_EDIGITALOUT fpEDigitalOut;                 // Function pointer
FT_ECOUNT fpECount;                           // Function pointer
FT_AOUPDATE fpAOUpdate;                       // Function pointer
FT_AISAMPLE fpAISample;                       // Function pointer
FT_AIBURST fpAIBurst;                         // Function pointer
FT_SHT1X fpSHT1X;                             // Function pointer
FT_NOTHREAD fpNoThread;                       // Function pointer
FT_PULSEOUT fpPulseOut;                       // Function pointer
FT_PULSEOUTSTART fpPulseOutStart;             // Function pointer
FT_PULSEOUTFINISH fpPulseOutFinish;           // Function pointer
FT_REENUM fpReEnum;                           // Function pointer
FT_RESET fpReset;                             // Function pointer
FT_GETFIRMWAREVERSION fpGetFirmwareVersion;   // Function pointer
FT_GETDRIVERVERSION fpGetDriverVersion;       // Function pointer
FT_LOCALID fpLocalID;                         // Function pointer
FT_AISTREAMSTART fpAIStreamStart;             // Function pointer
FT_AISTREAMREAD fpAIStreamRead;               // Function pointer
FT_AISTREAMCLEAR fpAIStreamClear;             // Function pointer
#endif /* }} */
  
#define EXCEPTION_CLASS_NAME "com/teravation/labjack/LabJackException"
jclass jclsLabJackException;
jmethodID jmidLabJackExceptionInit;

#define SCAN_CLASS_NAME "com/teravation/labjack/LabJackScan"
jclass jclsLabJackScan;
jmethodID jmidLabJackScanSetCounts;
jmethodID jmidLabJackScanSetDigitalIOValues;
jmethodID jmidLabJackScanSetAnalogInputValues;
jmethodID jmidLabJackScanSetCounterValues;

// Cached object rederences
jobject jobjSystemOut;
jobject jobjSystemErr;

// Cached method IDs
jmethodID jmidPrintln;
jmethodID jmidSetCounter;
jmethodID jmidSetTempC;
jmethodID jmidSetTempF;
jmethodID jmidSetHumidity;

// Scan information to persist between calls
long scanNumChannels;
long scanChannels[4];
long scanGains[4];
long scanReadCount;

#define MSG_SIZE 1024

/*****************************************************************************/
/* systemOutPrintln                                                          */
/*                                                                           */
/* Performs the same function as System.out.println(), but takes a C printf  */
/* format string and optional associated variables.                          */
/*****************************************************************************/
void systemOutPrintln(JNIEnv *env, char* fmt, ...) {
   char msgbuf[MSG_SIZE + 1];
   jstring jmsg = 0;
   va_list argp;

   va_start(argp, fmt);
#ifdef WINDOWS /* {{ */
   _vsnprintf(msgbuf, MSG_SIZE, fmt, argp);   
#else /* }{ */
   vsnprintf(msgbuf, MSG_SIZE, fmt, argp);   
#endif /* }} */
   va_end(argp);

   jmsg = (*env)->NewStringUTF(env, msgbuf);
   if (jmsg) {
      (*env)->CallVoidMethod(env, jobjSystemOut, jmidPrintln, jmsg);
   }
}

/*****************************************************************************/
/* systemErrPrintln                                                          */
/*                                                                           */
/* Performs the same function as System.err.println(), but takes a C printf  */
/* format string and optional associated variables.                          */
/*****************************************************************************/
void systemErrPrintln(JNIEnv *env, char* fmt, ...) {
   char msgbuf[MSG_SIZE + 1];
   jstring jmsg = 0;
   va_list argp;

   va_start(argp, fmt);
#ifdef WINDOWS /* {{ */
   _vsnprintf(msgbuf, MSG_SIZE, fmt, argp);   
#else /* }{ */
   vsnprintf(msgbuf, MSG_SIZE, fmt, argp);   
#endif /* }} */
   va_end(argp);

   jmsg = (*env)->NewStringUTF(env, msgbuf);
   if (jmsg) {
      (*env)->CallVoidMethod(env, jobjSystemErr, jmidPrintln, jmsg);
   }
}

/*****************************************************************************/
/* createException                                                           */
/*                                                                           */
/* Creates an exception object with the specified errorcode.                 */
/*****************************************************************************/
jobject createException(JNIEnv *env, long errorCode) {
   jobject exception = NULL;

   exception = (*env)->NewObject(env, jclsLabJackException, jmidLabJackExceptionInit, errorCode);

   return exception;
}

/*****************************************************************************/
/* setupScanInfo                                                             */
/*                                                                           */
/* This stores information about the scan in the global variables in order   */
/* to keep state between calls                                               */
/*****************************************************************************/
JNIEXPORT void setupScanInfo(JNIEnv *env, jlong numChannels, jlongArray channels, jlongArray gains, jlong readCount) {
   long channel;
   jlong copyOfChannels[4];
   jlong copyOfGains[4];

   scanNumChannels = (long) numChannels;
   scanReadCount = (long) readCount;
   (*env)->GetLongArrayRegion(env, channels, (jsize) 0, (jsize) numChannels, copyOfChannels);
   (*env)->GetLongArrayRegion(env, gains, (jsize) 0, (jsize) numChannels, copyOfGains);
   for (channel=0; channel < 4; channel++) {
      if (channel < numChannels) {
         scanChannels[channel] = (long) copyOfChannels[channel];
      } else {
         scanChannels[channel] = 0;
      }       
      if (channel < numChannels) {
         scanGains[channel] = (long) copyOfGains[channel];
      } else {
         scanGains[channel] = 0;
      }       
   }
}

/*****************************************************************************/
/* setScan                                                                   */
/*                                                                           */
/* Store all of the returned scan data into the LabJackScan object           */
/*****************************************************************************/
JNIEXPORT void setScan(JNIEnv *env, jobject scanObject, long numScans, long backlog, float ai[4096][4], long io[4096], long overVoltage) {
   int scan = 0;
   int channel = 0;
   long trueNumChannels = 0;

   jfloatArray jAIData;
   jlongArray jIOData;
   jlongArray jCounterData;
   jfloat aiData[4096];
   jlong ioData[4096];
   jlong counterData[4096];
   
   // Set the number of scans returned
   (*env)->CallVoidMethod(env, scanObject, jmidLabJackScanSetCounts, numScans, backlog);

   // Set the IO data
   for (scan=0; scan < numScans; scan++) {
      ioData[scan] = io[scan];
   }
   jIOData = (*env)->NewLongArray(env, (long) numScans);
   (*env)->SetLongArrayRegion(env, jIOData, 0, (long) numScans, ioData);
   (*env)->CallVoidMethod(env, scanObject, jmidLabJackScanSetDigitalIOValues, jIOData);

   if (scanReadCount > 0) {
      // Set the counter data
      for (scan=0; scan < numScans; scan++) {
         counterData[scan] = ((long) ai[scan][3] * 16777216)+ ((long) ai[scan][2] * 4096) + (long) ai[scan][1];
      }
      jCounterData = (*env)->NewLongArray(env, (long) numScans);
      (*env)->SetLongArrayRegion(env, jCounterData, 0, (long) numScans, counterData);   
      (*env)->CallVoidMethod(env, scanObject, jmidLabJackScanSetCounterValues, jCounterData);
      
      trueNumChannels = 1;
   } else {
      trueNumChannels = scanNumChannels;
   }
      
   // Set each analog input's data
   for (channel=0; channel < trueNumChannels; channel++) {
      for (scan=0; scan < numScans; scan++) {
         if (overVoltage) {
            aiData[scan] = -1.0f;
         } else {
            aiData[scan] = ai[scan][channel];
         }
      }
      jAIData = (*env)->NewFloatArray(env, (long) numScans);
      (*env)->SetFloatArrayRegion(env, jAIData, 0, (long) numScans, aiData);   
      (*env)->CallVoidMethod(env, scanObject, jmidLabJackScanSetAnalogInputValues, scanChannels[channel], jAIData);
   }
}

/*****************************************************************************/
/* JNI_OnLoad                                                                */
/*****************************************************************************/
JNIEXPORT jint JNICALL JNI_OnLoad(JavaVM *vm, void *reserved) {
   JNIEnv *env;
   jclass jclsException = 0;
   jclass jclsScan = 0;
   jobject jobjLocalSystemOut = 0;
   jobject jobjLocalSystemErr = 0;
   jclass clsSystem = 0;
   jclass clsPrintStream = 0;
   jfieldID fidSystemOut = 0;
   jfieldID fidSystemErr = 0;

   if ((*vm)->GetEnv(vm, (void **)&env, JNI_VERSION_1_2)) {
      return JNI_ERR; /* JNI version not supported */
   }

   // Get reference to java.lang.System
   clsSystem = (*env)->FindClass(env, "java/lang/System");
   if (!clsSystem) {
      return JNI_ERR;
   }

   // Get references to System.out and System.err
   fidSystemOut = (*env)->GetStaticFieldID(env, clsSystem, "out", "Ljava/io/PrintStream;");
   fidSystemErr = (*env)->GetStaticFieldID(env, clsSystem, "err", "Ljava/io/PrintStream;");
   if (!fidSystemOut | !fidSystemErr) {
      return JNI_ERR;
   }

   // Cache a reference to System.out
   jobjLocalSystemOut = (*env)->GetStaticObjectField(env, clsSystem, fidSystemOut);
   if (!jobjLocalSystemOut) {
      return JNI_ERR;
   }
   jobjSystemOut = (*env)->NewGlobalRef(env, jobjLocalSystemOut);

   // Cache a reference to System.err
   jobjLocalSystemErr = (*env)->GetStaticObjectField(env, clsSystem, fidSystemErr);
   if (!jobjLocalSystemErr) {
      return JNI_ERR;
   }
   jobjSystemErr = (*env)->NewGlobalRef(env, jobjLocalSystemErr);

   // Get a reference to java.io.PrintStream
   clsPrintStream = (*env)->FindClass(env, "java/io/PrintStream");
   if (!clsPrintStream) {
      return JNI_ERR;
   }
    
   // Get a reference to the PrintStream println() method 
   jmidPrintln = (*env)->GetMethodID(env, clsPrintStream, "println", "(Ljava/lang/String;)V");
   if (!jmidPrintln) {
      return JNI_ERR;
   }

   // Get reference to LabJackException
   jclsException = (*env)->FindClass(env, EXCEPTION_CLASS_NAME);
   if (!jclsException) {
      systemErrPrintln(env, "Unable to find exception class %s\n", EXCEPTION_CLASS_NAME);
      return JNI_ERR;
   }

   // Cache the method id for the LabJackException constructor
   jclsLabJackException = (*env)->NewGlobalRef(env, jclsException);
   jmidLabJackExceptionInit = (*env)->GetMethodID(env, jclsLabJackException, "<init>", "(I)V");
   if (!jmidLabJackExceptionInit) {
      systemErrPrintln(env, "Unable to find %s constructor\n", EXCEPTION_CLASS_NAME);
      return JNI_ERR;
   }

   // Get reference to LabJackScan
   jclsScan = (*env)->FindClass(env, SCAN_CLASS_NAME);
   if (!jclsScan) {
      systemErrPrintln(env, "Unable to find scan class %s\n", SCAN_CLASS_NAME);
      return JNI_ERR;
   }

   // Cache the method id for the LabJackScan setters
   jclsLabJackScan = (*env)->NewGlobalRef(env, jclsScan);
   jmidLabJackScanSetCounts = (*env)->GetMethodID(env, jclsLabJackScan, "setCounts", "(II)V");
   if (!jmidLabJackScanSetCounts) {
      systemErrPrintln(env, "Unable to find %s method (setCounts)\n", SCAN_CLASS_NAME);
      return JNI_ERR;
   }
   jmidLabJackScanSetDigitalIOValues = (*env)->GetMethodID(env, jclsLabJackScan, "setDigitalIOValues", "([J)V");
   if (!jmidLabJackScanSetDigitalIOValues) {
      systemErrPrintln(env, "Unable to find %s method (setDigitalIOValues)\n", SCAN_CLASS_NAME);
      return JNI_ERR;
   }   
   jmidLabJackScanSetAnalogInputValues = (*env)->GetMethodID(env, jclsLabJackScan, "setAnalogInputValues", "(I[F)V");
   if (!jmidLabJackScanSetAnalogInputValues) {
      systemErrPrintln(env, "Unable to find %s method (setAnalogInputValues)\n", SCAN_CLASS_NAME);
      return JNI_ERR;
   }
   jmidLabJackScanSetCounterValues = (*env)->GetMethodID(env, jclsLabJackScan, "setCounterValues", "([J)V");
   if (!jmidLabJackScanSetCounterValues) {
      systemErrPrintln(env, "Unable to find %s method (setCounterValues)\n", SCAN_CLASS_NAME);
      return JNI_ERR;
   }

#ifdef WINDOWS /* {{ */
   // Load ljackuw.dll (which is linked dynamically at run-time in order to provide the most flexibility).
   // All other OSes link at run-time by default.
   hLabJackDLL = LoadLibrary(LJACK_DLL);
   if (hLabJackDLL == NULL) {
      systemErrPrintln(env, "Unable to load DLL %s\n", LJACK_DLL);
      return JNI_ERR;
   }

   // Cache the pointers to the required DLL functions
   fpListAll = (FT_LISTALL)GetProcAddress(hLabJackDLL, "ListAll");
   fpEAnalogIn = (FT_EANALOGIN)GetProcAddress(hLabJackDLL, "EAnalogIn");
   fpEAnalogOut = (FT_EANALOGOUT)GetProcAddress(hLabJackDLL, "EAnalogOut");
   fpEDigitalIn = (FT_EDIGITALIN)GetProcAddress(hLabJackDLL, "EDigitalIn");
   fpEDigitalOut = (FT_EDIGITALOUT)GetProcAddress(hLabJackDLL, "EDigitalOut");
   fpECount = (FT_ECOUNT)GetProcAddress(hLabJackDLL, "ECount");
   fpAISample = (FT_AISAMPLE)GetProcAddress(hLabJackDLL, "AISample");
   fpAIBurst = (FT_AIBURST)GetProcAddress(hLabJackDLL, "AIBurst");
   fpAOUpdate = (FT_AOUPDATE)GetProcAddress(hLabJackDLL, "AOUpdate");
   fpNoThread = (FT_NOTHREAD)GetProcAddress(hLabJackDLL, "NoThread");
   fpSHT1X = (FT_SHT1X)GetProcAddress(hLabJackDLL, "SHT1X");
   fpPulseOut = (FT_PULSEOUT)GetProcAddress(hLabJackDLL, "PulseOut");
   fpPulseOutStart = (FT_PULSEOUTSTART)GetProcAddress(hLabJackDLL, "PulseOutStart");
   fpPulseOutFinish = (FT_PULSEOUTFINISH)GetProcAddress(hLabJackDLL, "PulseOutFinish");
   fpReEnum = (FT_REENUM)GetProcAddress(hLabJackDLL, "ReEnum");
   fpReset = (FT_RESET)GetProcAddress(hLabJackDLL, "Reset");
   fpGetFirmwareVersion = (FT_GETFIRMWAREVERSION)GetProcAddress(hLabJackDLL, "GetFirmwareVersion");
   fpGetDriverVersion = (FT_GETDRIVERVERSION)GetProcAddress(hLabJackDLL, "GetDriverVersion");
   fpLocalID = (FT_LOCALID)GetProcAddress(hLabJackDLL, "LocalID");
   fpAIStreamStart = (FT_AISTREAMSTART)GetProcAddress(hLabJackDLL, "AIStreamStart");
   fpAIStreamRead = (FT_AISTREAMREAD)GetProcAddress(hLabJackDLL, "AIStreamRead");
   fpAIStreamClear = (FT_AISTREAMCLEAR)GetProcAddress(hLabJackDLL, "AIStreamClear");

   if (!(fpListAll && fpEAnalogIn && fpEAnalogOut && fpEDigitalIn && fpEDigitalOut && fpECount && fpAISample && fpAOUpdate && fpNoThread && fpPulseOut && fpPulseOutStart && fpPulseOutFinish && fpReEnum && fpReset && fpGetFirmwareVersion && fpGetDriverVersion && fpLocalID && fpAIStreamStart && fpAIStreamRead && fpAIStreamClear)) {
      systemErrPrintln(env, "Unable to load function from DLL %s\n", LJACK_DLL);
      return JNI_ERR;
   }
#endif /* }} */

   return JNI_VERSION_1_2;
}

/*****************************************************************************/
/* JNI_OnUnload                                                              */
/*****************************************************************************/
JNIEXPORT void JNICALL JNI_OnUnload(JavaVM *vm, void *reserved) {
   JNIEnv *env;

#ifdef WINDOWS /* {{ */
   FreeLibrary(hLabJackDLL);
#endif /* }} */

   // Free the cached object references
   if (!(*vm)->GetEnv(vm, (void **)&env, JNI_VERSION_1_2)) {
      (*env)->DeleteGlobalRef(env, jclsLabJackException);
      (*env)->DeleteGlobalRef(env, jclsLabJackScan);
      (*env)->DeleteGlobalRef(env, jobjSystemOut);
      (*env)->DeleteGlobalRef(env, jobjSystemErr);
   }
}

/*****************************************************************************/
/* apiGetDriverVersion                                                       */
/*                                                                           */
/* Calls the GetDriverVersion method of the DLL                              */
/*****************************************************************************/
JNIEXPORT jfloat JNICALL Java_com_teravation_labjack_LabJackDriver_apiGetDriverVersion(JNIEnv *env, jclass cls) {
   float fReturnVal = 0.0f;

#ifdef WINDOWS /* {{ */
   fReturnVal = fpGetDriverVersion();
#else /* }{ */
   fReturnVal = GetDriverVersion();
#endif /* }} */

   return fReturnVal;
}

/*****************************************************************************/
/* apiListAll                                                                */
/*                                                                           */
/* Calls the GetAll method of the DLL, and returns an array of Java LabJack  */
/* objects, one object for each device found.                                */
/*****************************************************************************/
JNIEXPORT jobjectArray JNICALL Java_com_teravation_labjack_LabJackDriver_apiListAll(JNIEnv *env, jclass cls) {
   long uReturnVal = 0;

   long productIdList[128];
   long serialNumList[128];
   long localIdList[128];
   long powerList[128];
   long calMatrix[128][20];
   long numberFound = 0;
   long labjackNumber = 0;
#ifdef LINUX /* {{ */
   long idnum = -1;
   long resetCounter = 0;
   double count = 0;
   double ms = 0;
#else /* }{ */
   long fcddMaxSize;
   long hvcMaxSize;
#endif /* }} */
 
   long localId;

   jmethodID mid;
   jobjectArray labjacks = NULL;
   jobject labjack = NULL;

   // Reset arrays
   int i,j;
   for (i=0; i < 128; i++) {
      productIdList[i] = 0;
      serialNumList[i] = 0;
      localIdList[i] = 0;
      powerList[i] = 0;
      for (j=0; j < 20; j++) {
         calMatrix[i][j] = 0;
      }
   }

   // Cache the method ID of setCounter for later use
   jmidSetCounter = (*env)->GetMethodID(env, cls, "setCounter", "(I)V");
   jmidSetTempC = (*env)->GetMethodID(env, cls, "setTempC", "(F)V");
   jmidSetTempF = (*env)->GetMethodID(env, cls, "setTempF", "(F)V");
   jmidSetHumidity = (*env)->GetMethodID(env, cls, "setHumidity", "(F)V");

#ifdef WINDOWS /* {{ */
   // Call the function
   uReturnVal = fpListAll(productIdList, serialNumList, localIdList, powerList, calMatrix, &numberFound, &fcddMaxSize, &hvcMaxSize);
#endif /* }} */
#ifdef MACOSX /* {{ */
   // Call the function
   uReturnVal = ListAll(productIdList, serialNumList, localIdList, powerList, calMatrix, &numberFound, &fcddMaxSize, &hvcMaxSize);
#endif /* }} */
#ifdef LINUX /* {{ */
   // Fake it for drivers that don't support ListAll
   for (i = 0; i < 128; i++) {
      productIdList[i] = 9999;
   }

   // Try to run ECount to see if there is a LabJack plugged in
   uReturnVal = ECount(&idnum, NODEMO, (long) resetCounter, &count, &ms);
   if (uReturnVal == 0) {
      numberFound = 1;
      serialNumList[0] = -1;
      localIdList[0] = idnum;
      productIdList[0] = -1;
   }   
   uReturnVal = 0;
#endif /* }} */

   if (uReturnVal > 0) {
      (*env)->Throw(env, createException(env,uReturnVal));
   } else {
      // Create an array of Java LabJack objects
      mid = (*env)->GetMethodID(env, cls, "<init>", "(III)V");
      labjacks = (*env)->NewObjectArray(env, numberFound, cls, NULL);
      // Process each LabJack device found
      if (numberFound > 0) {
         for (i=0; i < 128; i++) {
            if (productIdList[i] != 9999 ) {
               localId = localIdList[i];
               // Create a new LabJack object
               labjack = (*env)->NewObject(env, cls, mid, localId, serialNumList[i], productIdList[i]);
               // Add it to the array
               (*env)->SetObjectArrayElement(env, labjacks, labjackNumber++, labjack);
               if (--numberFound == 0) {
                  break;
               }
            }
         }
      }
   }
   return labjacks;
}

/*****************************************************************************/
/* apiEAnalogIn                                                              */
/*                                                                           */
/* Calls the EAnalogIn method of the DLL, and returns a float that contains  */
/* the analog voltage (or -1.0f on an overvoltage).                          */
/*****************************************************************************/
JNIEXPORT jfloat JNICALL Java_com_teravation_labjack_LabJackDriver_apiEAnalogIn(JNIEnv *env, jobject obj, jint idnum, jint channel, jint gain) {
   unsigned uReturnVal = 0;
   long overVoltage = 0;
   float voltage = 0.0f;

#ifdef WINDOWS /* {{ */
   uReturnVal = fpEAnalogIn(&idnum, NODEMO, (long) channel, (long) gain, &overVoltage, &voltage);
#else /* }{ */
   uReturnVal = EAnalogIn(&idnum, NODEMO, (long) channel, (long) gain, &overVoltage, &voltage);
#endif /* }} */

   if (uReturnVal > 0) {
      (*env)->Throw(env, createException(env,uReturnVal));
   } else {
      if (overVoltage) {
         voltage = -1.0f;
      }
   }
   return (jfloat) voltage;
}

/*****************************************************************************/
/* apiEAnalogOut                                                             */
/*                                                                           */
/* Calls the EAnalogOut method of the DLL.                                   */
/*****************************************************************************/
JNIEXPORT void JNICALL Java_com_teravation_labjack_LabJackDriver_apiEAnalogOut(JNIEnv *env, jobject obj, jint idnum, jfloat analogOut0, jfloat analogOut1) {
   unsigned uReturnVal = 0;

#ifdef WINDOWS /* {{ */
   uReturnVal = fpEAnalogOut(&idnum, NODEMO, analogOut0, analogOut1);
#else /* }{ */
   uReturnVal = EAnalogOut(&idnum, NODEMO, analogOut0, analogOut1);
#endif /* }} */

   if (uReturnVal > 0) {
      (*env)->Throw(env, createException(env,uReturnVal));
   }
}

/*****************************************************************************/
/* apiECount                                                                 */
/*                                                                           */
/* Calls the ECount method of the DLL, and returns a long that contains      */
/* the value of the counter.                                                 */
/*****************************************************************************/
JNIEXPORT jlong JNICALL Java_com_teravation_labjack_LabJackDriver_apiECount(JNIEnv *env, jobject obj, jint idnum, jboolean resetCounter) {
   unsigned uReturnVal = 0;
   double count = 0;
   double ms = 0;

#ifdef WINDOWS /* {{ */
   uReturnVal = fpECount(&idnum, NODEMO, (long) resetCounter, &count, &ms);
#else /* }{ */
   uReturnVal = ECount(&idnum, NODEMO, (long) resetCounter, &count, &ms);
#endif /* }} */

   if (uReturnVal > 0) {
      (*env)->Throw(env, createException(env,uReturnVal));
   }

   return (jlong) count;
}

/*****************************************************************************/
/* apiEDigitalIn                                                             */
/*                                                                           */
/* Calls the EDigitalIn method of the DLL, and returns a long that contain   */
/* the value of the digital line (either 1 or 0).                            */
/*****************************************************************************/
JNIEXPORT jboolean JNICALL Java_com_teravation_labjack_LabJackDriver_apiEDigitalIn(JNIEnv *env, jobject obj, jint idnum, jint channel, jboolean readD) {
   unsigned uReturnVal = 0;
   long value = 0;

#ifdef WINDOWS /* {{ */
   uReturnVal = fpEDigitalIn(&idnum, NODEMO, (long) channel, (long) readD, &value);
#else /* }{ */
   uReturnVal = EDigitalIn(&idnum, NODEMO, (long) channel, (long) readD, &value);
#endif /* }} */

   if (uReturnVal > 0) {
      (*env)->Throw(env, createException(env,uReturnVal));
   }
 
   return (jboolean) value;
}

/*****************************************************************************/
/* apiEDigitalOut                                                            */
/*                                                                           */
/* Calls the EDigitalOut method of the DLL.                                  */
/*****************************************************************************/
JNIEXPORT void JNICALL Java_com_teravation_labjack_LabJackDriver_apiEDigitalOut(JNIEnv *env, jobject obj, jint idnum, jint channel, jboolean writeD, jboolean value) {
   unsigned uReturnVal = 0;

#ifdef WINDOWS /* {{ */
   uReturnVal = fpEDigitalOut(&idnum, NODEMO, (long) channel, (long) writeD, (long) value);
#else /* }{ */
   uReturnVal = EDigitalOut(&idnum, NODEMO, (long) channel, (long) writeD, (long) value);
#endif /* }} */

   if (uReturnVal > 0) {
      (*env)->Throw(env, createException(env,uReturnVal));
   }
}

/*****************************************************************************/
/* apiAOUpdate                                                               */
/*                                                                           */
/* Calls the AOUpdate method of the DLL, and returns a long that contains    */
/* the status of ALL of the digital lines (both D and IO).  Additionally,    */
/* since we can only return one object (because this is, in effect, a Java   */
/* method) we will call the setCounter method on the LabJack object to       */
/* update the counter.                                                       */
/*****************************************************************************/
JNIEXPORT jlong JNICALL Java_com_teravation_labjack_LabJackDriver_apiAOUpdate(JNIEnv *env, jobject obj, jint idnum, jlong trisD, jlong trisIO, jlong stateD, jlong stateIO, jlong updateDigital, jlong resetCounter, jfloat analogOut0, jfloat analogOut1) {
   unsigned uReturnVal = 0;
   unsigned long count = 0;

   long localStateD = (long) stateD;
   long localStateIO = (long) stateIO;

#ifdef WINDOWS /* {{ */
   uReturnVal = fpAOUpdate(&idnum, NODEMO, (long) trisD, (long) trisIO, &localStateD, &localStateIO, (long) updateDigital, (long) resetCounter, &count, (float) analogOut0, (float) analogOut1);
#else /* }{ */
   uReturnVal = AOUpdate(&idnum, NODEMO, (long) trisD, (long) trisIO, &localStateD, &localStateIO, (long) updateDigital, (long) resetCounter, &count, (float) analogOut0, (float) analogOut1);
#endif /* }} */

   if (uReturnVal > 0) {
      (*env)->Throw(env, createException(env,uReturnVal));
   } else {
      // Call the setCounter method
      (*env)->CallVoidMethod(env, obj, jmidSetCounter, count);
   }
   return (jlong) ((localStateD << 4) | localStateIO);
}

/*****************************************************************************/
/* apiAISample                                                               */
/*                                                                           */
/* Calls the AISample method of the DLL.  Returns an array of floats         */
/* containing the analog voltages (or -1.0f on an overvoltage.               */
/*****************************************************************************/
JNIEXPORT jfloatArray JNICALL Java_com_teravation_labjack_LabJackDriver_apiAISample(JNIEnv *env, jobject obj, jint idnum, jlong stateIO, jlong updateIO, jlong ledOn, jlong numChannels, jlongArray channels, jlongArray gains, jlong disableCal) {
   unsigned uReturnVal = 0;
   int channel;

   long overVoltage = 0;
   float voltages[4];
   long localStateIO = (long) stateIO;
   jfloatArray jVoltages;

   for (channel=0; channel < 4; channel++) {
      voltages[channel] = 0.0f;
   }

   setupScanInfo(env, numChannels, channels, gains, (jlong) 0);
   
#ifdef WINDOWS /* {{ */
   uReturnVal = fpAISample(&idnum, NODEMO, &localStateIO, (long) updateIO, (long) ledOn, scanNumChannels, scanChannels, scanGains, (long) disableCal, &overVoltage, voltages);
#else /* }{ */
   uReturnVal = AISample(&idnum, NODEMO, &localStateIO, (long) updateIO, (long) ledOn, scanNumChannels, scanChannels, scanGains, (long) disableCal, &overVoltage, voltages);
#endif /* }} */

   if (uReturnVal > 0) {
      (*env)->Throw(env, createException(env,uReturnVal));
   } else {
      if (overVoltage) {
         for (channel=0; channel<4; channel++) {
            voltages[channel] = -1.0f;
         }
      }

      jVoltages = (*env)->NewFloatArray(env, 4);
      (*env)->SetFloatArrayRegion(env, jVoltages, 0, 4, voltages);
   }
   return jVoltages;
}


/*****************************************************************************/
/* apiAIBurst                                                                */
/*                                                                           */
/* Calls the AIBurst method of the DLL.                                      */
/*****************************************************************************/
JNIEXPORT jfloat JNICALL Java_com_teravation_labjack_LabJackDriver_apiAIBurst(JNIEnv *env, jobject obj, jint idnum, jlong stateIOIn, jlong updateIO, jlong ledOn, jlong numChannels, jlongArray channels, jlongArray gains, jfloat scanRate, jlong disableCal, jlong triggerIO, jlong triggerState, jlong numScans, jlong timeout, jlong transferMode, jobject scanObject) {
   unsigned uReturnVal = 0;
   int channel = 0;
   int scan = 0;
   
   long overVoltage = 0;
   float voltages[4096][4];
   long stateIOOut[4096];
   float localScanRate = (float) scanRate;

   // Clear out the stateIOOut and voltages arrays
   for (scan = 0; scan < 4096; scan++) {
     stateIOOut[scan] = 0;
     for (channel=0; channel < 4; channel++) {
        voltages[scan][channel] = 0.0f;
     }
   }
     
   setupScanInfo(env, numChannels, channels, gains, (jlong) 0);
     
#ifdef WINDOWS /* {{ */
   uReturnVal = fpAIBurst(&idnum, NODEMO, (long) stateIOIn, (long) updateIO, (long) ledOn, scanNumChannels, scanChannels, scanGains, &localScanRate, (long) disableCal, (long) triggerIO, (long) triggerState, (long) numScans, (long) timeout, voltages, stateIOOut, &overVoltage, (long) transferMode);
#else /* }{ */
   uReturnVal = AIBurst(&idnum, NODEMO, (long) stateIOIn, (long) updateIO, (long) ledOn, scanNumChannels, scanChannels, scanGains, &localScanRate, (long) disableCal, (long) triggerIO, (long) triggerState, (long) numScans, (long) timeout, voltages, stateIOOut, &overVoltage, (long) transferMode);
#endif /* }} */

   if (uReturnVal > 0) {
      (*env)->Throw(env, createException(env,uReturnVal));
   } else {
      setScan(env, scanObject, (long) numScans, 0, voltages, stateIOOut, overVoltage); 
   }
   
   return (jlong) localScanRate;
}


/*****************************************************************************/
/* apiSHT1X                                                                  */
/*                                                                           */
/* Calls the SHT1X method of the DLL.  Since we can only return one object   */
/* (because this is, in effect, a Java method) we opt to return nothing,     */
/* and just call the setTempC, setTempF, and setHumidity methods to update   */
/* the LabJack.                                                              */
/*****************************************************************************/
JNIEXPORT void JNICALL Java_com_teravation_labjack_LabJackDriver_apiSHT1X(JNIEnv *env, jobject obj, jint idnum) {

   unsigned uReturnVal = 0;
   float tempC = 0.0f;
   float tempF = 0.0f;
   float rh = 0.0f;

#ifdef WINDOWS /* {{ */
   if (!fpSHT1X) {
      (*env)->Throw(env, createException(env, 1001));
      return;
   }
   uReturnVal = fpSHT1X(&idnum, NODEMO, 0, 0, 0, &tempC, &tempF, &rh);
#else /* }{ */
   uReturnVal = SHT1X(&idnum, NODEMO, 0, 0, 0, &tempC, &tempF, &rh);
#endif /* }} */

   if (uReturnVal > 0) {
      (*env)->Throw(env, createException(env,uReturnVal));
   } else {
     // Call setter methods on the LabJack object 
     (*env)->CallVoidMethod(env, obj, jmidSetTempC, tempC);
     (*env)->CallVoidMethod(env, obj, jmidSetTempF, tempF);
     (*env)->CallVoidMethod(env, obj, jmidSetHumidity, rh);
   }
}

/*****************************************************************************/
/* apiPulseOut                                                               */
/*                                                                           */
/* Calls the PulseOut method of the DLL                                      */
/*****************************************************************************/
JNIEXPORT void JNICALL Java_com_teravation_labjack_LabJackDriver_apiPulseOut(JNIEnv *env, jobject obj, jint idnum, jint lowFirst, jint bitSelect, jint numPulses, jint timeB1, jint timeC1, jint timeB2, jint timeC2) {
   unsigned uReturnVal = 0;

#ifdef WINDOWS /* {{ */
   uReturnVal = fpPulseOut(&idnum, NODEMO, (long) lowFirst, (long) bitSelect, (long) numPulses, (long) timeB1, (long) timeC1, (long) timeB2, (long) timeC2);
#else /* }{ */
   uReturnVal = PulseOut(&idnum, NODEMO, (long) lowFirst, (long) bitSelect, (long) numPulses, (long) timeB1, (long) timeC1, (long) timeB2, (long) timeC2);
#endif /* }} */

   if (uReturnVal > 0) {
      (*env)->Throw(env, createException(env,uReturnVal));
   }
}

/*****************************************************************************/
/* apiPulseOutStart                                                          */
/*                                                                           */
/* Calls the PulseOutStart method of the DLL                                 */
/*****************************************************************************/
JNIEXPORT void JNICALL Java_com_teravation_labjack_LabJackDriver_apiPulseOutStart(JNIEnv *env, jobject obj, jint idnum, jint lowFirst, jint bitSelect, jint numPulses, jint timeB1, jint timeC1, jint timeB2, jint timeC2) {
   unsigned uReturnVal = 0;

#ifdef WINDOWS /* {{ */
   uReturnVal = fpPulseOutStart(&idnum, NODEMO, (long) lowFirst, (long) bitSelect, (long) numPulses, (long) timeB1, (long) timeC1, (long) timeB2, (long) timeC2);
#else /* }{ */
   uReturnVal = PulseOutStart(&idnum, NODEMO, (long) lowFirst, (long) bitSelect, (long) numPulses, (long) timeB1, (long) timeC1, (long) timeB2, (long) timeC2);
#endif /* }} */

   if (uReturnVal > 0) {
      (*env)->Throw(env, createException(env,uReturnVal));
   }
}

/*****************************************************************************/
/* apiPulseOutFinish                                                         */
/*                                                                           */
/* Calls the PulseOutFinish method of the DLL                                */
/*****************************************************************************/
JNIEXPORT void JNICALL Java_com_teravation_labjack_LabJackDriver_apiPulseOutFinish(JNIEnv *env, jobject obj, jint idnum, jint timeoutMS) {
   unsigned uReturnVal = 0;

#ifdef WINDOWS /* {{ */
   uReturnVal = fpPulseOutFinish(&idnum, NODEMO, (long) timeoutMS);
#else /* }{ */
   uReturnVal = PulseOutFinish(&idnum, NODEMO, (long) timeoutMS);
#endif /* }} */

   if (uReturnVal > 0) {
      (*env)->Throw(env, createException(env,uReturnVal));
   }
}

/*****************************************************************************/
/* apiReEnum                                                                 */
/*                                                                           */
/* Calls the ReEnum method of the DLL                                        */
/*****************************************************************************/
JNIEXPORT void JNICALL Java_com_teravation_labjack_LabJackDriver_apiReEnum(JNIEnv *env, jobject obj, jint idnum) {
   unsigned uReturnVal = 0;

#ifdef WINDOWS /* {{ */
   uReturnVal = fpReEnum(&idnum);
#else /* }{ */
   uReturnVal = ReEnum(&idnum);
#endif /* }} */

   if (uReturnVal > 0) {
      (*env)->Throw(env, createException(env,uReturnVal));
   }
}

/*****************************************************************************/
/* apiReset                                                                  */
/*                                                                           */
/* Calls the Reset method of the DLL                                         */
/*****************************************************************************/
JNIEXPORT void JNICALL Java_com_teravation_labjack_LabJackDriver_apiReset(JNIEnv *env, jobject obj, jint idnum) {
   unsigned uReturnVal = 0;

#ifdef WINDOWS /* {{ */
   uReturnVal = fpReset(&idnum);
#else /* }{ */
   uReturnVal = Reset(&idnum);
#endif /* }} */

   if (uReturnVal > 0) {
      (*env)->Throw(env, createException(env,uReturnVal));
   }
}

/*****************************************************************************/
/* apiGetFirmwareVersion                                                     */
/*                                                                           */
/* Calls the GetFirmwareVersion method of the DLL                            */
/*****************************************************************************/
JNIEXPORT jfloat JNICALL Java_com_teravation_labjack_LabJackDriver_apiGetFirmwareVersion(JNIEnv *env, jobject obj, jint idnum) {
   float fReturnVal = 0.0f;

#ifdef WINDOWS /* {{ */
   fReturnVal = fpGetFirmwareVersion(&idnum);
#else /* }{ */
   fReturnVal = GetFirmwareVersion(&idnum);
#endif /* }} */

   return fReturnVal;
}

/*****************************************************************************/
/* apiLocalID                                                                */
/*                                                                           */
/* Calls the LocalID method of the DLL                                       */
/*****************************************************************************/
JNIEXPORT void JNICALL Java_com_teravation_labjack_LabJackDriver_apiLocalID(JNIEnv *env, jobject obj, jint idnum, jint localID) {
   unsigned uReturnVal = 0.0f;

#ifdef WINDOWS /* {{ */
   uReturnVal = fpLocalID(&idnum, localID);
#else /* }{ */
   uReturnVal = LocalID(&idnum, localID);
#endif /* }} */

   if (uReturnVal > 0) {
      (*env)->Throw(env, createException(env,uReturnVal));
   }
}

/*****************************************************************************/
/* apiAIStreamStart                                                          */
/*                                                                           */
/* Calls the AISample method of the DLL.  Returns an array of floats         */
/* containing the analog voltages (or -1.0f on an overvoltage.               */
/*****************************************************************************/
JNIEXPORT jfloat JNICALL Java_com_teravation_labjack_LabJackDriver_apiAIStreamStart(JNIEnv *env, jobject obj, jint idnum, jlong stateIO, jlong updateIO, jlong ledOn, jlong numChannels, jlongArray channels, jlongArray gains, jfloat scanRate, jlong disableCal, jlong readCount) {
   unsigned uReturnVal = 0;
   
   float localScanRate = (float) scanRate;

   setupScanInfo(env, numChannels, channels, gains, readCount);
      
#ifdef WINDOWS /* {{ */
   uReturnVal = fpAIStreamStart(&idnum, NODEMO, (long) stateIO, (long) updateIO, (long) ledOn, scanNumChannels, scanChannels, scanGains, &localScanRate, (long) disableCal, 0, (long) readCount);
#else /* }{ */
   uReturnVal = AIStreamStart(&idnum, NODEMO, (long) stateIO, (long) updateIO, (long) ledOn, scanNumChannels, scanChannels, scanGains, &localScanRate, (long) disableCal, 0, (long) readCount);
#endif /* }} */

   if (uReturnVal > 0) {
      (*env)->Throw(env, createException(env,uReturnVal));
   }

   return localScanRate;
}

/*****************************************************************************/
/* apiAIStreamClear                                                          */
/*                                                                           */
/* Calls the LocalID method of the DLL                                       */
/*****************************************************************************/
JNIEXPORT void JNICALL Java_com_teravation_labjack_LabJackDriver_apiAIStreamClear(JNIEnv *env, jobject obj, jint idnum) {
   unsigned uReturnVal = 0.0f;

#ifdef WINDOWS /* {{ */
   uReturnVal = fpAIStreamClear(idnum);
#else /* }{ */
   uReturnVal = AIStreamClear(idnum);
#endif /* }} */

   if (uReturnVal > 0) {
      (*env)->Throw(env, createException(env,uReturnVal));
   }
}

/*****************************************************************************/
/* apiAIStreamRead                                                           */
/*                                                                           */
/* Calls the AIStreamRead method of the DLL.                                 */
/*****************************************************************************/
JNIEXPORT void JNICALL Java_com_teravation_labjack_LabJackDriver_apiAIStreamRead(JNIEnv *env, jobject obj, jint idnum, jlong numScans, jlong timeout, jobject scanObject) {
   unsigned uReturnVal = 0;
   int scan = 0;
   int channel = 0;
   
   float voltages[4096][4];
   long stateIOOut[4096];
   long reserved = 0;
   long scanBacklog = 0;
   long overVoltage = 0;
   
   // Clear out the stateIOOut and voltages arrays
   for (scan = 0; scan < 4096; scan++) {
     stateIOOut[scan] = 0;
     for (channel=0; channel < 4; channel++) {
        voltages[scan][channel] = 0.0f;
     }
   }
   
#ifdef WINDOWS /* {{ */
   uReturnVal = fpAIStreamRead(idnum, (long) numScans, (long) timeout, voltages, stateIOOut, &reserved, &scanBacklog, &overVoltage);
#else /* }{ */
   uReturnVal = AIStreamRead(idnum, (long) numScans, (long) timeout, voltages, stateIOOut, &reserved, &scanBacklog, &overVoltage);
#endif /* }} */

   if (uReturnVal > 0) {
      (*env)->Throw(env, createException(env,uReturnVal));
   } else {
      setScan(env, scanObject, (long) numScans, scanBacklog, voltages, stateIOOut, overVoltage); 
   }
}

