package com.teravation.labjack;
/**
 * This file is part of LabJackJava.  Copyright 2003-2004, Teravation.  All rights reserved. 
 * 
 * LabJackJava is free software; you can redistribute it and/or modify it under the terms of the GNU
 * General Public License as published by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 * 
 * LabJackJava is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; 
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with LabJackJava;
 * if not, write to:
 *    Free Software Foundation, Inc.
 *    59 Temple Place, Suite 330
 *    Boston, MA  02111-1307  USA
 * 
 * In addition, as a special exception, Teravation gives permission to link the code of this program with
 * the LabJack C driver (or with modified versions of the LabJack C driver that use the same license),
 * and distribute linked combinations including the two.  You must obey the GNU General Public License in all
 * respects for all of the code used other than the LabJack C driver.  If you modify this file, you may 
 * extend this exception to your version of the file, but you are not obligated to do so.  If you do not wish to
 * do so, delete this exception statement from your version.
 * 
 */

/**
 * This class holds the data for the scan data read from Burst mode.  It is intended that it will also be used
 * to hold scan information from Stream mode in the future.  The interface is very similar to the LabJack 
 * interface, but each method takes the scan number as an additional parameter.
 *
 * <p />Copyright &copy; 2003-2004 <a href="http://www.teravation.com">Teravation</a>. All rights reserved.
 * 
 * @created (12/30/2002 2:25:49 PM)
 * @version 4.0
 * @author Chris Reigrut 
 */
public class LabJackScan {
	private int available = 0;
	private int backlog = 0;
	private float[][] analogInputValues = new float[LabJackConstants.ANALOG_INPUT_CHANNELS][];
	private long[] counterValues = null;
	private long[] digitalIOValues = null;
	
	/**
	 * LabJackScan constructor.
	 */
	LabJackScan() {
		super();
	}
	
	/**
	 * Set the number of values read during the scan.
	 * NOTE:  This method should only be used by the underlying native driver! 
	 * 
	 * @created (1/10/2003 2:38:10 PM)
	 * @author: 
	 * 
	 * @param scans int
	 */
	void setAvailable(int scans) {
		available = scans;
	}

	/**
	 * Set the number of values still remaining after this scan.
	 * NOTE:  This method should only be used by the underlying native driver! 
	 * 
	 * @created  (3/28/2004 10:38:10 PM)
	 * @author: 
	 * 
	 * @param scans int
	 */
	void setBacklog(int scans) {
		backlog = scans;
	}

	/**
	 * Set the counts (available and backlog)
	 * NOTE:  This method should only be used by the underlying native driver! 
	 * 
	 * @created  (3/28/2004 10:38:10 PM)
	 * @author: 
	 * 
	 * @param available int
	 * @param backlog int
	 */
	void setCounts(int available, int backlog) {
		this.available = available;
		this.backlog = backlog;
	}

	/**
	 * Set the analog input values in the scan.
	 * NOTE:  This method should only be used by the underlying native driver! 
	 * 
	 * Creation date: (1/10/2003 2:38:10 PM)
	 * @author: 
	 * 
	 * @param channel int
	 * @param aiData float[]
	 */
	
	void setAnalogInputValues(int channel, float[] aiData) {
		this.analogInputValues[channel] = aiData;
	}	
	
	/**
	 * Set the counter values in the scan.
	 * NOTE:  This method should only be used by the underlying native driver! 
	 * 
	 * Creation date: (1/10/2003 2:38:10 PM)
	 * @author: 
	 * 
	 * @param counterValues long[]
	 */
	
	void setCounterValues(long[] counterData) {
		this.counterValues = counterData;
	}

	/**
	 * Set the digital IO values in the scan.
	 * NOTE:  This method should only be used by the underlying native driver! 
	 * 
	 * Creation date: (1/10/2003 2:38:10 PM)
	 * @author: Chris Reigrut
	 * 
	 * @param ioData long[]
	 */
	
	void setDigitalIOValues(long[] ioData) {
		this.digitalIOValues = ioData;
	}

	/**
	 * Returns the last updated voltage of the specified analog input for the specified scan, or -1.0f on an overvoltage.
	 * 
	 * @created (4/14/2003 6:23:29 PM)
	 * 
	 * @param channel the analog input channel to read.
	 * @param scan int the number of the scan to read.
	 * @return the voltage of the specified analog input.
	 * @exception com.teravation.labjack.LabJackException on any error
	 */
	public float getAI(int channel, int scan) throws LabJackException {
		if (channel < 0 || channel >= LabJackConstants.ANALOG_SINGLE_INPUT_CHANNELS) {
			throw new LabJackException(21);
		}
		
		float[] aiData = this.analogInputValues[channel];
		if (aiData == null) {
			throw new LabJackException(16);
		}
		if (scan < 0 || scan > LabJackConstants.MAX_SCANS || scan > aiData.length) {
			throw new LabJackException(32);
		}
		return aiData[scan];
	}
	/**
	 * Returns the last updated voltage of the specified differential analog input for the specified scan,
	 * or -1.0f on an overvoltage.  The differential channels are measured as follows:
	 * <ul>
	 * <li>Differential channel 0 is measured between AI0 and AI1.</li>
	 * <li>Differential channel 1 is measured between AI2 and AI3.</li>
	 * <li>Differential channel 2 is measured between AI4 and AI5.</li>
	 * <li>Differential channel 3 is measured between AI6 and AI7.</li>
	 * </ul>
	 * 
	 * @created (4/14/2003 6:23:59 PM)
	 * 
	 * @param channel the differential analog input channel to read.
	 * @param scan int the number of the scan to read.
	 * @return the voltage of the specified differential analog input.
	 * @exception com.teravation.labjack.LabJackException on any error
	 */
	public float getDifferentialAI(int channel, int scan) throws LabJackException {
		if (channel < 0 || channel >= LabJackConstants.ANALOG_SINGLE_INPUT_CHANNELS) {
			throw new LabJackException(21);
		}
		
		float[] aiData = this.analogInputValues[LabJackConstants.ANALOG_SINGLE_INPUT_CHANNELS + channel];
		if (aiData == null) {
			throw new LabJackException(16);
		}
		if (scan < 0 || scan > LabJackConstants.MAX_SCANS || scan > aiData.length) {
			throw new LabJackException(32);
		}
		return aiData[scan];
	}
	/**
	 * Returns the specified digital IO value from the appropriate bit array
	 * 
	 * @created (2/3/2003 5:08:49 PM)
	 * 
	 * @return boolean
	 * @param bit int
	 * @param scan int
	 */
	private boolean getDigitalIOValue(int bit, int scan) throws LabJackException {
		if (scan < 0 || scan > LabJackConstants.MAX_SCANS || scan > this.digitalIOValues.length) {
			throw new LabJackException(32);
		}
		return (this.digitalIOValues[scan] & (1 << bit)) > 0;
	}
	
	/**
	 * Returns the last updated/set value of the specified digital input (screw terminal) for the specified scan.
	 * 
	 * @created (4/14/2003 2:40:35 PM)
	 * 
	 * @param channel the digital input channel to read.
	 * @param scan int the number of the scan to read.
	 * @return the value of the specified digital input.
	 * @exception com.teravation.labjack.LabJackException on any underlying DLL error.
	 */
	public boolean getIO(int channel, int scan) throws LabJackException {
		boolean value = false;
		if (channel < 0 || channel >= LabJackConstants.IO_CHANNELS) {
			throw new LabJackException(21);
		}

		return this.getDigitalIOValue(channel, scan);
	}	
	
	/**
	 * Returns the last updated/set value of the counter for the specified scan.
	 * 
	 * @created  (3/28/2004 10:38:10 PM)
	 * 
	 * @param scan int the number of the scan to read.
	 * @return the value of the counter.
	 * @exception com.teravation.labjack.LabJackException on any underlying DLL error.
	 */
	public long getCounter(int scan) throws LabJackException {
		return this.counterValues[scan];
	}

	/**
	 * Returns the number of scans available in this object.
	 * 
	 * @created (4/14/2003 6:23:29 PM)
	 * 
	 * @return the number of scans available in this object
	 */
	public int getAvailable() {
		return available;
	}

	/**
	 * Returns the number of scans in the backlog.
	 * 
	 * @created (4/14/2003 6:23:29 PM)
	 * 
	 * @return the number of scans in the backlog
	 */
	public int getBacklog() {
		return backlog;
	}

}
