package com.teravation.labjack;
/**
 * This file is part of LabJackJava.  Copyright 2003-2004, Teravation.  All rights reserved. 
 * 
 * LabJackJava is free software; you can redistribute it and/or modify it under the terms of the GNU
 * General Public License as published by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 * 
 * LabJackJava is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; 
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with LabJackJava;
 * if not, write to:
 *    Free Software Foundation, Inc.
 *    59 Temple Place, Suite 330
 *    Boston, MA  02111-1307  USA
 * 
 * In addition, as a special exception, Teravation gives permission to link the code of this program with
 * the LabJack C driver (or with modified versions of the LabJack C driver that use the same license),
 * and distribute linked combinations including the two.  You must obey the GNU General Public License in all
 * respects for all of the code used other than the LabJack C driver.  If you modify this file, you may 
 * extend this exception to your version of the file, but you are not obligated to do so.  If you do not wish to
 * do so, delete this exception statement from your version.
 * 
 */

/**
 * This interface defines the methods to interface with a <a href="http://www.labjack.com">LabJack</a>
 * data acquisition & control device.  This interface will be implemented by the platform-dependent
 * concrete classes.
 * <p />The <code>LabJack</code> objects are virtual devices, generally independent from the device itself.
 * The <code>updateXXXXX</code> methods are used to synchronize the object with the device, reading and/or
 * writing the object information to/from the device itself.
 *
 * <p />Copyright &copy; 2003-2004 <a href="http://www.teravation.com">Teravation</a>. All rights reserved.
 * 
 * @created (12/30/2002 2:52:36 PM)
 * @version 4.0
 * @author Chris Reigrut 
 */
public interface LabJack {
	/**
	 * Returns the last updated voltage of the specified analog input, or -1.0f on an overvoltage.
	 * 
	 * @created (1/10/2003 6:23:29 PM)
	 * 
	 * @param channel the analog input channel to read.
	 * @return the voltage of the specified analog input.
	 * @exception com.teravation.labjack.LabJackException on any error
	 */
	float getAI(int channel) throws LabJackException;
	
	/**
	 * Returns the last set voltage of the specified analog output.
	 * 
	 * @created (2/7/2003 11:55:52 AM)
	 * 
	 * @param channel the analog output channel to read.
	 * @return the voltage of the specified analog output.
	 * @exception com.teravation.labjack.LabJackException on any error
	 */
	float getAO(int channel) throws LabJackException;

	/**
	 * Returns the last updated value of the counter.
	 * 
	 * @created (1/20/2003 12:46:40 PM)
	 * 
	 * @return the current counter value.
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	long getCounter() throws LabJackException;

	/**
	 * Returns the last updated/set value of the specified digital input (DB25).
	 * 
	 * @created (1/10/2003 2:40:35 PM)
	 * 
	 * @param channel the digital input channel to read.
	 * @return the value of the specified digital input.
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	boolean getD(int channel) throws LabJackException;

	/**
	 * Returns the last updated voltage of the specified differential analog input,
	 * or -1.0f on an overvoltage.  The differential channels are measured as follows:
	 * <ul>
	 * <li>Differential channel 0 is measured between AI0 and AI1.</li>
	 * <li>Differential channel 1 is measured between AI2 and AI3.</li>
	 * <li>Differential channel 2 is measured between AI4 and AI5.</li>
	 * <li>Differential channel 3 is measured between AI6 and AI7.</li>
	 * </ul>
	 * 
	 * @created (1/10/2003 6:23:59 PM)
	 * 
	 * @param channel the differential analog input channel to read.
	 * @return the voltage of the specified differential analog input.
	 * @exception com.teravation.labjack.LabJackException on any error
	 */
	public float getDifferentialAI(int channel) throws LabJackException;

	/**
	 * Returns the last set gain factor of the specified differential analog input.
	 * The gain values are specified as the constants:
	 * <ul>
	 * <li><code>GAIN_1x</code></li>
	 * <li><code>GAIN_2x</code></li>
	 * <li><code>GAIN_4x</code></li>
	 * <li><code>GAIN_5x</code></li>
	 * <li><code>GAIN_8x</code></li>
	 * <li><code>GAIN_10x</code></li>
	 * <li><code>GAIN_16x</code></li>
	 * <li><code>GAIN_20x</code></li>
	 * </ul>
	 * 
	 * @created (1/10/2003 6:23:59 PM)
	 * 
	 * @param channel the differential analog input channel for which to get the gain.
	 * @return the gain value of the specified differential analog input gain.
	 * @exception com.teravation.labjack.LabJackException on any error
	 */
	public int getDifferentialAIGain(int channel) throws LabJackException;

	/**
	 * Returns the last updated relative humidity percentage
	 *
	 * @created (2/6/2003 3:35:48 PM)
	 *
	 * @return the relative humidity percentage
	 */
	public float getHumidity();

	/**
	 * Returns the last updated/set value of the specified digital input (screw terminal).
	 * 
	 * @created (1/10/2003 2:40:35 PM)
	 * 
	 * @param channel the digital input channel to read.
	 * @return the value of the specified digital input.
	 * @exception com.teravation.labjack.LabJackException on any underlying DLL error.
	 */
	boolean getIO(int channel) throws LabJackException;

	/**
	 * Returns the local id of the LabJack.
	 * 
	 * @created (1/10/2003 11:58:21 AM)
	 * 
	 * @return the local id of the LabJack.
	 */
	public int getLocalId();

	/**
	 * Returns the product id of the LabJack.
	 * 
	 * @created (1/10/2003 11:58:21 AM)
	 * 
	 * @return the product id of the LabJack.
	 */
	public int getProductId();

	/**
	 * Returns the last updated LabJackScan object (from a Burst or Stream).
	 * 
	 * @created (1/10/2003 2:40:35 PM)
	 * 
	 * @return the last update scan.
	 */
	LabJackScan getScan();

	/**
	 * Returns the serial number of the LabJack.
	 * 
	 * @created (1/10/2003 11:58:21 AM)
	 * 
	 * @return the serial number of the LabJack.
	 */
	public int getSerialNumber();

	/**
	 * Returns the last updated temperature (in Celsius)
	 *
	 * @created (2/6/2003 3:35:48 PM)
	 *
	 * @return the temperature (in Celsius)
	 */
	public float getTempC();

	/**
	 * Returns the last updated temperature (in Fahrenheit)
	 *
	 * @created (2/6/2003 3:35:48 PM)
	 *
	 * @return the temperature (in Fahrenheit)
	 */
	public float getTempF();

	/**
	 * Returns whether the specified D line is set for output (from the bit array)
	 * 
	 * @created (2/3/2003 5:08:49 PM)
	 * 
	 * @return boolean
	 * @param bit int
	 */
	public boolean isDForOutput(int bit);

	/**
	 * Returns whether the specified IO line is set for output (from the bit array)
	 * 
	 * @created (2/3/2003 5:08:49 PM)
	 * 
	 * @return boolean
	 * @param bit int
	 */
	public boolean isIOForOutput(int bit);

	/**
	 * Sets the voltage of the specified analog output.  This will be applied to the device
	 * during the next appropriate <code>updateXXXXXX</code> call.
	 * 
	 * @created (1/10/2003 2:55:28 PM)
	 * 
	 * @param channel the analog output channel to set.
	 * @param value the voltage to be set.
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	void setAO(int channel, float value) throws LabJackException;

	/**
	 * Sets the state of the specified digital output (DB25).  This will be applied to the device
	 * during the next appropriate <code>updateXXXXXX</code> call.  If the channel is not currently
	 * set for output, a LabJackException will be thrown.
	 * 
	 * @created (1/10/2003 11:33:11 AM)
	 * 
	 * @param channel the digital output channel to set.
	 * @param value the state to set.
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	void setD(int channel, boolean value) throws LabJackException;

	/**
	 * Sets the specified digital channel (DB25) as an input channel (the default).
	 * 
	 * @created (1/10/2003 11:33:11 AM)
	 * 
	 * @param channel the digital channel to set as an input.
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void setDForInput(int channel) throws LabJackException;

	/**
	 * Sets the specified digital channel (DB25) as an output channel.
	 * 
	 * @created (1/10/2003 11:33:11 AM)
	 * 
	 * @param channel the digital output channel to set as an output.
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void setDForOutput(int channel) throws LabJackException;

	/**
	 * Sets the gain value for the specified differential analog input.
	 * The gain values are specified as the constants:
	 * <ul>
	 * <li>GAIN_1x</li>
	 * <li>GAIN_2x</li>
	 * <li>GAIN_4x</li>
	 * <li>GAIN_5x</li>
	 * <li>GAIN_8x</li>
	 * <li>GAIN_10x</li>
	 * <li>GAIN_16x</li>
	 * <li>GAIN_20x</li>
	 * </ul>
	 * 
	 * @created (1/10/2003 6:23:59 PM)
	 * 
	 * @param channel the differential analog input channel for which to set the gain.
	 * @param gain the gain value for the specified differential analog input.
	 * @exception com.teravation.labjack.LabJackException on any error
	 */
	public void setDifferentialAIGain(int channel, int gain) throws LabJackException;

	/**
	 * Sets the state of the specified digital output (screw terminal).  This will be applied to the device
	 * during the next appropriate <code>updateXXXXXX</code> call.  If the channel is not currently
	 * set for output, a LabJackException will be thrown.
	 * 
	 * @created (1/10/2003 11:33:11 AM)
	 * 
	 * @param channel the digital output channel to set.
	 * @param value the state to set.
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	void setIO(int channel, boolean value) throws LabJackException;

	/**
	 * Sets the specified digital channel (screw terminal) as an input channel (the default).
	 * 
	 * @created (1/10/2003 11:33:11 AM)
	 * 
	 * @param channel the digital channel to set as an input.
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void setIOForInput(int channel) throws LabJackException;

	/**
	 * Sets the specified digital channel (screw terminal) as an output channel.
	 * 
	 * @created (1/10/2003 11:33:11 AM)
	 * 
	 * @param channel the digital output channel to set as an output.
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void setIOForOutput(int channel) throws LabJackException;

	/**
	 * Updates the <code>LabJack</code> object with voltage read from the specified AI channel
	 * of the associated device.
	 * <br />On Windows, this call utilizes the <code>EAnalogIn</code> DLL call.
	 *
	 * @created (2/7/2003 12:12:37 PM)
	 *
	 * @param channel the analog input channel to update.
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updateAI(int channel) throws LabJackException;

	/**
	 * Updates the entire associated device with the values in the current <code>LabJack</code> object.
	 * Equivalent to calling <code>updateAllAOsAndDigitals</code>, <code>updateAllAIs</code>,
	 * <code>updateAllDifferentialAIs</code>, and <code>updateEnvironmentals</code>.
	 *
	 * @created (2/6/2003 12:05:18 PM)
	 *
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updateAll() throws LabJackException;

	/**
	 * Updates the entire associated device with the values in the current <code>LabJack</code> object.
	 * Equivalent to calling <code>updateAllAOsAndDigitals</code>, <code>updateAllAIs</code>,
	 * <code>updateAllDifferentialAIs</code>, and <code>updateEnvironmentals</code>.
	 *
	 * @created (2/6/2003 12:05:18 PM)
	 *
	 * @param resetCounter whether or not to reset the counter after reading it.
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updateAll(boolean resetCounter) throws LabJackException;

	/**
	 * Updates the <code>LabJack</code> object with voltages read from the AI channels 0-7
	 * of the associated device.
	 * <br />On Windows, this call utilizes the <code>AISample</code> DLL call.
	 *
	 * @created (2/6/2003 12:05:18 PM)
	 *
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updateAllAIs() throws LabJackException;

	/**
	 * Updates the associated device, setting the AO voltages and digital (both D and IO) output values
	 * stored in the <code>LabJack</code> object, as well as reading the digital input values and the 
	 * counter (but does not reset the counter).
	 * <br />On Windows, this call utilizes the <code>AOUpdate</code> DLL call.
	 *
	 * @created (2/6/2003 12:05:18 PM)
	 *
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updateAllAOsAndDigitals() throws LabJackException;

	/**
	 * Updates the associated device, setting the AO voltages and digital (both D and IO) output values
	 * stored in the <code>LabJack</code> object, as well as reading the digital input values and the 
	 * counter, and, if indicated, resets the counter after reading it.
	 * <br />On Windows, this call utilizes the <code>AOUpdate</code> DLL call.
	 *
	 * @created (2/6/2003 12:05:18 PM)
	 *
	 * @param resetCounter whether or not to reset the counter after reading it.
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updateAllAOsAndDigitals(boolean resetCounter) throws LabJackException;

	/**
	 * Updates the <code>LabJack</code> object with differential voltages read from the 
	 * associated device.
	 * <br />On Windows, this call utilizes the <code>AISample</code> DLL call.
	 *
	 * @created (2/6/2003 12:05:18 PM)
	 *
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updateAllDifferentialAIs() throws LabJackException;

	/**
	 * Updates the associated device, setting the specified AO voltage to the current value stored in the
	 * <code>LabJack</code> object.
	 * <br />On Windows, this call utilizes the <code>EAnalogOut</code> DLL call.
	 *
	 * @created (2/6/2003 12:05:18 PM)
	 *
	 * @param channel the analog output channel to update.
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updateAO(int channel) throws LabJackException;

	/**
	 * Updates the <code>LabJack</code> object with the counter value of the associated device (but does not
	 * reset the counter).
	 * <br />On Windows, this call utilizes the <code>ECount</code> DLL call.
	 *
	 * @created (2/6/2003 12:05:18 PM)
	 *
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updateCounter() throws LabJackException;

	/**
	 * Updates the <code>LabJack</code> object with counter value of the associated device, 
	 * and if indicated, resets the counter after reading it.
	 * <br />On Windows, this call utilizes the <code>ECount</code> DLL call.
	 *
	 * @created (2/6/2003 12:05:18 PM)
	 *
	 * @param resetCounter whether or not to reset the counter after reading it.
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updateCounter(boolean resetCounter) throws LabJackException;

	/**
	 * Updates the <code>LabJack</code> object with the value of the associated channel (DB25).  if the 
	 * channel is set as an input, the value is read from the device.  If the channel is set as an output,
	 * the value is written to the device.
	 * <br />On Windows, this call utilizes the <code>EDigitalOut</code> or <code>EDigitalIn</code> DLL call.
	 *
	 * @created (2/6/2003 12:05:18 PM)
	 *
	 * @param channel the digital IO channel to update.
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updateD(int channel) throws LabJackException;

	/**
	 * Updates the <code>LabJack</code> object with differential voltage read from the specified 
	 * differential channel of the associated device.
	 * <br />On Windows, this call utilizes the <code>EAnalogIn</code> DLL call.
	 *
	 * @created (2/7/2003 12:12:37 PM)
	 *
	 * @param channel the differential analog input channel to update.
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updateDifferentialAI(int channel) throws LabJackException;

	/**
	 * Updates the <code>LabJack</code> object with temperatures (Fahrenheit and Celsius) and relative
	 * humidity read from an attached temperature/humidity probe.  Requires LabJack firmware v1.1 or higher
	 * (otherwise, it will return zeros).
	 * <br />On Windows, this call utilizes the <code>SHT1X</code> DLL call.
	 *
	 * @created (2/6/2003 12:05:18 PM)
	 *
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updateEnvironmentals() throws LabJackException;

	/**
	 * Updates the <code>LabJack</code> object with voltages read from the AI channels 4-7
	 * of the associated device.
	 * <br />On Windows, this call utilizes the <code>AISample</code> DLL call.
	 *
	 * @created (2/6/2003 12:05:18 PM)
	 *
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updateHighAIs() throws LabJackException;

	/**
	 * Updates the <code>LabJack</code> object with the value of the associated channel (screw terminal).  if the 
	 * channel is set as an input, the value is read from the device.  If the channel is set as an output,
	 * the value is written to the device.
	 * <br />On Windows, this call utilizes the <code>EDigitalOut</code> or <code>EDigitalIn</code> DLL call.
	 *
	 * @created (2/6/2003 12:05:18 PM)
	 *
	 * @param channel the digital IO channel to update.
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updateIO(int channel) throws LabJackException;

	/**
	 * Updates the <code>LabJack</code> object with voltages read from the AI channels 0-3
	 * of the associated device.
	 * <br />On Windows, this call utilizes the <code>AISample</code> DLL call.
	 *
	 * @created (2/6/2003 12:05:18 PM)
	 *
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updateLowAIs() throws LabJackException;

	/**
	 * Updates the <code>LabJack</code> object with scans read from Burst mode.
	 * <br />On Windows, this call utilizes the <code>AIBurst</code> DLL call.
	 *
	 * @created (4/14/2003 12:05:18 PM)
	 *
	 * @param channels the digital IO channels to update.
	 * @param scanRate the rate to scan the channels.
	 * @param scans the total number of scans to record.
	 * @return the actual scan rate used
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public float updateScanFromBurst(long[] channels, float scanRate, int scans) throws LabJackException;

	/**
	 * Updates the <code>LabJack</code> object with scans read from Burst mode.
	 * <br />On Windows, this call utilizes the <code>AIBurst</code> DLL call.
	 *
	 * @created (4/14/2003 12:05:18 PM)
	 *
	 * @param channels the digital IO channels to update.
	 * @param scanRate the rate to scan the channels.
	 * @param scans the total number of scans to record.
	 * @param timeout the total number of scans to record.
	 * @param updateIO whether or not to update the IO channels.
	 * @param triggerIO the IO port to trigger on (false=IO0, true=IO1).
	 * @param triggerState whether or not to trigger on an IO port.
	 * @param transferMode unused (currently must be false).
	 * @param ledOn whether or not to turn on the device LED.
	 * @return the actual scan rate used
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public float updateScanFromBurst(long[] channels, float scanRate, int scans, int timeout, boolean updateIO, boolean triggerIO, boolean triggerState, long transferMode, boolean ledOn)
		throws LabJackException;

	/**
	 * Sets the local id.  This will be updated on the associated device on the next <code>updateLocalId</code> call.
	 * 
	 * @created (1/10/2003 11:33:11 AM)
	 * 
	 * @param value the state to set.
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	void setLocalId(int value) throws LabJackException;
	
	/**
	 * Updates the associated device, setting the local id to the value set in the <code>LabJack</code> object.
	 * <br />On Windows, this call utilizes the <code>LocalID</code> DLL call.
	 * <br /><em>NOTE:  The change will only take effect after a reset or re-enumeration of the physical device.</em>
	 *
	 * @created (3/6/2004 8:42:01 PM)
	 *
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updateLocalId() throws LabJackException;

	/**
	 * Resets the associated device.  This will typically take a few seconds to complete (even though the call 
	 * returns immediately).  
	 * <br />On Windows, this call utilizes the <code>Reset</code> DLL call.
	 *
	 * @created (3/6/2004 8:42:01 PM)
	 *
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void reset() throws LabJackException;

	/**
	 * Forces the associated device to re-enumerate.  This will typically take a few seconds to complete (even though the call 
	 * returns immediately).  
	 * <br />On Windows, this call utilizes the <code>ReEnum</code> DLL call.
	 *
	 * @created (3/6/2004 8:42:01 PM)
	 *
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void reenumerate() throws LabJackException;

	/**
	 * Updates the <code>LabJack</code> object with the firmware version from the associated device.
	 * <br />On Windows, this call utilizes the <code>GetFirmwareVersion</code> DLL call.
	 *
	 * @created (3/6/2004 8:42:01 PM)
	 *
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updateFirmwareVersion() throws LabJackException;

	/**
	 * Returns the firmware version of the LabJack.
	 * 
	 * @created (3/6/2004 8:42:01 PM)
	 * 
	 * @return the serial number of the LabJack.
	 */
	public float getFirmwareVersion() throws LabJackException;

	/**
	 * Sets the pulse width and spacing (in microseconds) for the device during subsequent calls to <code>updatePulseXXXX</code>.
	 * @see LabJack#setPulseWidth(float)
	 * @see LabJack#setPulseSpace(float)
	 * 
	 * @created (Mar 6, 2004 9:12:57 PM)
	 * 
	 * @param pulseWidth The pulse width (in microseconds)
	 * @param pulseSpace The pulse space (in microseconds)
	 */
	public void setPulse(float pulseWidth, float pulseSpace);
	
	/**
	 * Sets the pulse width (in microseconds) for the device during subsequent calls to <code>updatePulseXXXX</code>.
	 * <br /><em>NOTE:  This actually sets the pulse width to the nearest supported value, calculated by
	 * <br>1st half-cycle microseconds = ~17 + 0.83*C + 20.17*B*C
	 * <br />(where 1 <= B <= 255 and 1 <= C <= 255)
	 * <br />A subsequent call to getPulseWidth will return the actual pulse width.</em>
	 * 
	 * @created (Mar 6, 2004 9:12:57 PM)
	 * 
	 * @param pulseWidth The pulse width (in microseconds)
	 */
	public void setPulseWidth(float pulseWidth);

	/**
	 * Sets the space between pulses pulse (in microseconds) for the device during subsequent calls to <code>updatePulseXXXX</code>.
	 * <br /><em>NOTE:  This actually sets the pulse width to the nearest supported value, calculated by
	 * <br>1st half-cycle microseconds = ~12 + 0.83*C + 20.17*B*C
	 * <br />(where 1 <= B <= 255 and 1 <= C <= 255)
	 * <br />A subsequent call to getPulseSpace will return the actual pulse space.</em>
	 * 
	 * @created (Mar 6, 2004 9:12:57 PM)
	 * 
	 * @param pulseSpace The pulse space (in microseconds)
	 */
	public void setPulseSpace(float pulseSpace);

	/**
	 * Gets the pulse width (in microseconds).
	 * 
	 * @created (Mar 6, 2004 9:12:57 PM)
	 * 
	 * @return pulseWidth The pulse width (in microseconds)
	 */
	public float getPulseWidth();

	/**
	 * Gets the pulse spacing (in microseconds).
	 * 
	 * @created (Mar 6, 2004 9:12:57 PM)
	 * 
	 * @return pulseSpace The pulse space (in microseconds)
	 */
	public float getPulseSpace();

	/**
	 * Updates the associated device, causing it to generate square pulses on the specified D line.
	 * <br />On Windows, this call utilizes the <code>PulseOut</code> DLL call.
	 * <br /><em>NOTE:  This is the synchronous version, and will not return until all the pulses have been generated.</em>
	 * <br /><em>NOTE:  The selected D line must have been set as an output (on the hardware side) with prior calls to
	 * <code>setDForOutput</code> and <code>updateAllAOsAndDigitals</code>.</em>
	 *
	 * @created (Mar 6, 2004 9:12:57 PM)
	 *
	 * @param dChannel the line (D0-D7).
	 * @param numPulses the number of pulses to create.
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updatePulse(int dChannel, int numPulses) throws LabJackException;

	/**
	 * Updates the associated device, causing it to generate square pulses on the specified D line(s)
	 * <br />On Windows, this call utilizes the <code>PulseOut</code> DLL call.
	 * <br /><em>NOTE:  This is the synchronous version, and will not return until all the pulses have been generated.</em>
	 * <br /><em>NOTE:  The selected D lines must have been set as an output (on the hardware side) with prior calls to
	 * <code>setDForOutput</code> and <code>updateAllAOsAndDigitals</code>.</em>
	 *
	 * @created (Mar 6, 2004 9:12:57 PM)
	 *
	 * @param dChannels the lines (D0-D7).
	 * @param numPulses the number of pulses to create.
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updatePulse(boolean[] dChannels, int numPulses) throws LabJackException;

	/**
	 * Updates the associated device, causing it to generate square pulses on the specified D line(s)
	 * <br />On Windows, this call utilizes the <code>PulseOut</code> DLL call.
	 * <br /><em>NOTE:  This is the synchronous version, and will not return until all the pulses have been generated.</em>
	 * <br /><em>NOTE:  The selected D lines must have been set as an output (on the hardware side) with prior calls to
	 * <code>setDForOutput</code> and <code>updateAllAOsAndDigitals</code>.</em>
	 *
	 * @created (Mar 6, 2004 9:12:57 PM)
	 *
	 * @param dChannels the lines (D0-D7).
	 * @param numPulses the number of pulses to create.
	 * @param lowFirst whether to set the line high, then low (true) or low, then high (false).
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updatePulse(boolean[] dChannels, int numPulses, boolean lowFirst) throws LabJackException;

	/**
	 * Updates the associated device, causing it to generate square pulses on the specified D line.
	 * <br />On Windows, this call utilizes the <code>PulseOutStart</code> DLL call.
	 * <br /><em>NOTE:  This is the asynchronous version, and returns immediately.  In order to then wait for the call to complete,
	 * call <code>updatePulseAsynchFinish</code></em>
	 * <br /><em>NOTE:  The selected D line must have been set as an output (on the hardware side) with prior calls to
	 * <code>setDForOutput</code> and <code>updateAllAOsAndDigitals</code>.</em>
	 *
	 * @created (Mar 6, 2004 9:12:57 PM)
	 *
	 * @param dChannel the line (D0-D7).
	 * @param numPulses the number of pulses to create.
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updatePulseAsynch(int dChannel, int numPulses) throws LabJackException;

	/**
	 * Updates the associated device, causing it to generate square pulses on the specified D line(s)
	 * <br />On Windows, this call utilizes the <code>PulseOutStart</code> DLL call.
	 * <br /><em>NOTE:  This is the asynchronous version, and returns immediately.  In order to then wait for the call to complete,
	 * call <code>updatePulseAsynchFinish</code></em>
	 * <br /><em>NOTE:  The selected D lines must have been set as an output (on the hardware side) with prior calls to
	 * <code>setDForOutput</code> and <code>updateAllAOsAndDigitals</code>.</em>
	 *
	 * @created (Mar 6, 2004 9:12:57 PM)
	 *
	 * @param dChannels the lines (D0-D7).
	 * @param numPulses the number of pulses to create.
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updatePulseAsynch(boolean[] dChannels, int numPulses) throws LabJackException;

	/**
	 * Updates the associated device, causing it to generate square pulses on the specified D line(s)
	 * <br />On Windows, this call utilizes the <code>PulseOutStart</code> DLL call.
	 * <br /><em>NOTE:  This is the asynchronous version, and returns immediately.  In order to then wait for the call to complete,
	 * call <code>updatePulseAsynchFinish</code></em>
	 * <br /><em>NOTE:  The selected D lines must have been set as an output (on the hardware side) with prior calls to
	 * <code>setDForOutput</code> and <code>updateAllAOsAndDigitals</code>.</em>
	 *
	 * @created (Mar 6, 2004 9:12:57 PM)
	 *
	 * @param dChannels the lines (D0-D7).
	 * @param numPulses the number of pulses to create.
	 * @param lowFirst whether to set the line high, then low (true) or low, then high (false).
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updatePulseAsynch(boolean[] dChannels, int numPulses, boolean lowFirst) throws LabJackException;

	/**
	 * Updates the associated device, waiting until the last <code>updatePulseAsynch</code> has finished, or the timeout
	 * has expired.  If the <code>updatePulseAsynch</code> has already completed, this call can only time out.
	 * <br />On Windows, this call utilizes the <code>PulseOutFinish</code> DLL call.
	 *
	 * @created (Mar 6, 2004 9:12:57 PM)
	 *
	 * @param timeout the timeout (in ms).
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updatePulseAsynchFinish(int timeoutMS) throws LabJackException;

	/**
	 * Updates the associated device, putting it into Stream mode.
	 * <br />On Windows, this call utilizes the <code>AIStreamStart</code> DLL call.
	 *
	 * @created (Mar 28, 2004 10:12:57 PM)
	 *
	 * @param channels the channels to sample
	 * @param scanRate the rate at which to sample
	 * @param readCount whether or not to sample the counter (if so, only the first channel will actually be sampled)
	 * @return the actual scan rate that will be used
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public float updateStreamStart(long[] channels, float scanRate, boolean readCount) throws LabJackException;
	
	/**
	 * Updates the associated device, putting it into Stream mode.
	 * <br />On Windows, this call utilizes the <code>AIStreamStart</code> DLL call.
	 *
	 * @created (Mar 28, 2004 10:12:57 PM)
	 *
	 * @param channels the channels to sample
	 * @param scanRate the rate at which to sample
	 * @return the actual scan rate that will be used
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public float updateStreamStart(long[] channels, float scanRate) throws LabJackException;
	
	/**
	 * Updates the associated device, ending Stream mode.
	 * <br />On Windows, this call utilizes the <code>AIStreamClear</code> DLL call.
	 *
	 * @created (Mar 28, 2004 10:12:57 PM)
	 *
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updateStreamClear() throws LabJackException;
	
	/**
	 * Reads the designated number of scans from the device and updates the current 
	 * scan object with that data, or timeout after 10 seconds if the data is not
	 * available.
	 * <br />On Windows, this call utilizes the <code>AIStreamRead</code> DLL call.
	 *
	 * @created (Mar 28, 2004 10:12:57 PM)
	 *
	 * @param scans the number of scans to read
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updateScanFromStream(int scans) throws LabJackException;
	
	/**
	 * Reads the designated number of scans from the device and updates the current 
	 * scan object with that data, or timeout after the designated period.
	 * <br />On Windows, this call utilizes the <code>AIStreamRead</code> DLL call.
	 *
	 * @created (Mar 28, 2004 10:12:57 PM)
	 *
	 * @param scans the number of scans to read
	 * @param timeout the time (in seconds) to wait for the data before returning 
	 * @exception com.teravation.labjack.LabJackException on any error.
	 */
	public void updateScanFromStream(int scans, int timeout) throws LabJackException;
}
