{
  LJMUtilites.pas

  Provides some basic help functions.
}

unit LJMUtilities;

interface

uses
  SysUtils,
  LJMDelphi; { Path specified in project }

{ Converts passed integer to its string equivalent. Used by PrintDeviceInfo. }
function NumberToConnectionType(connectionType: Integer): String;
function NumberToDeviceType(deviceType: Integer): String;

{ Gets and displays the device's information based on the passed handle. }
procedure PrintDeviceInfoFromHandle(handle: Integer);

{ Displays the device's information based on the passed parameters. }
procedure PrintDeviceInfo(deviceType, connectionType, serialNumber, ipAddressInt, portOrPipe, packetMaxBytes: Integer);

{ Waits for user to hit the Enter key. }
procedure WaitForUser(message: String);

{ Displays error message and returns True if an error is detected, othewise }
{ returns False. }
function IsError(description: String; error: Integer; errorAddress: Integer = -1): Boolean;

{ Displays error message, closes all devices, waits for user to hit Enter Key }
{ and ends program if an error is detected. }
procedure ErrorHandler(description: String; error: Integer; errorAddress: Integer = -1);

implementation

function NumberToConnectionType(connectionType: Integer): String;
begin
  Case connectionType of
    LJM_ctANY : Result := 'LJM_ctANY';
    LJM_ctUSB : Result := 'LJM_ctUSB';
    LJM_ctTCP : Result := 'LJM_ctTCP';
    LJM_ctETHERNET : Result := 'LJM_ctETHERNET';
    LJM_ctWIFI : Result := 'LJM_ctWIFI';
  else Result := 'Unknown connection type';
  end;
end;

function NumberToDeviceType(deviceType: Integer): String;
begin
  Case deviceType of
    LJM_dtANY : Result := 'LJM_dtANY';
    LJM_dtUE9 : Result := 'LJM_dtUE9';
    LJM_dtU3 : Result := 'LJM_dtU3';
    LJM_dtU6 : Result := 'LJM_dtU6';
    LJM_dtT7 : Result := 'LJM_dtT7';
    LJM_dtSKYMOTE_BRIDGE : Result := 'LJM_dtSKYMOTE_BRIDGE';
    LJM_dtDIGIT : Result := 'LJM_dtDIGIT';
  else Result := 'Unknown device type';
  end;
end;

procedure PrintDeviceInfoFromHandle(handle: Integer);
var
  deviceType: Integer;
  connectionType: Integer;
  serialNumber: Integer;
  ipAddress: Integer;
  port: Integer;
  packetMaxBytes: Integer;
  err: Integer;
begin
  err := LJM_GetHandleInfo(handle, deviceType, connectionType, serialNumber, ipAddress, port, packetMaxBytes);
  ErrorHandler('PrintDeviceInfoFromHandle (LJM_GetHandleInfo)', err);

  PrintDeviceInfo(DeviceType, ConnectionType, SerialNumber, IPAddress, Port, packetMaxBytes);
end;

procedure PrintDeviceInfo(deviceType, connectionType, serialNumber, ipAddressInt, portOrPipe, packetMaxBytes: Integer);
var
  ipAddressString: Array[0..LJM_IPv4_STRING_SIZE-1] of AnsiChar;
begin
  { Print }
  Writeln('deviceType: ' + NumberToDeviceType(deviceType));
  Writeln('connectionType: ' + NumberToConnectionType(connectionType));
  Writeln('serialNumber: ' + IntToStr(serialNumber));
  LJM_NumberToIP(ipAddressInt, @ipAddressString);
  Writeln('IP address: ' + ipAddressString);

  if connectionType = LJM_ctUSB then
  begin
    Writeln('pipe: ' + IntToStr(portOrPipe));
  end
  else
  begin
    Writeln('port: ' + IntToStr(portOrPipe));
  end;

  Writeln('The maximum number of bytes you can send to or receive from this device in one packet is ' + IntToStr(packetMaxBytes) + ' bytes.');
  Writeln('');
end;

procedure WaitForUser(message: String);
begin
  Writeln('');
  Writeln(message);
  Readln;
end;

function IsError(description: String; error: Integer; errorAddress: Integer = -1): Boolean;
var
  errorString: Array[0..LJM_MAX_NAME_SIZE] of AnsiChar;
  msgType: String;
begin
  Result := False; { Default return }
  if (error >= LJME_WARNINGS_BEGIN) and (error <= LJME_WARNINGS_END) then
    msgType := 'warning'
  else if error <> LJME_NOERROR then
    msgType := 'error'
  else
    Exit;

  LJM_ErrorToString(error, @errorString);
  Writeln(description + ' ' + msgType + ': ' + errorString + '(' + IntToStr(error) + ')');
  if errorAddress > -1
  then Writeln('error address = ' + IntToStr(errorAddress));
  if msgType = 'error' then
    Result := True;
end;

procedure ErrorHandler(description: String; error: Integer; errorAddress: Integer = -1);
begin
  if IsError(description, error, errorAddress) then
  begin
    Writeln('Closing all devices and terminating application.');
    LJM_CloseAll();
    WaitForUser('Press Enter to exit.');
    Halt;
  end;
end;

end.
