{
  SPI.dpr

  Demonstrates SPI communication.

  You can short MOSI to MISO for testing.

  MOSI    FIO2
  MISO    FIO3
  CLK     FIO0
  CS      FIO1

  If you short MISO to MOSI, then you will read back the same bytes that you
  write.  If you short MISO to GND, then you will read back zeros.  If you
  short MISO to VS or leave it unconnected, you will read back 255s.
}

program SPI;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  LJMDelphi in '..\..\Wrapper\LJMDelphi.pas',
  LJMUtilities in '..\LJMUtilities.pas';

var
  i: Integer;
  handle: Integer;
  error: Integer = 0;
  errAddr: Integer = -1;

  { LJM_eReadNames and LJM_eNames variables. }
  numFrames: Integer;
  aNames: array of AnsiString;
  aWrites: array of Integer;
  aNumValues: array of Integer;
  aValues: array of Double;

  { Write/read bytes variables. }
  numBytes: Integer;
  dataWrite: array of Double;
  dataRead: array of Double;

  { Time variables for the random generator seed. }
  h: Word;
  m: Word;
  s: Word;
  ms: Word;
begin
  { Open first found LabJack. }
  error := LJM_Open(LJM_dtAny, LJM_ctAny, 'Any', handle);
  //error := LJM_OpenS('Any', 'Any', 'Any', handle);
  ErrorHandler('LJM_Open', error);

  { Display opened device's information. }
  PrintDeviceInfoFromHandle(handle);


  { CS is FIO1 }
  error := LJM_eWriteName(handle, 'SPI_CS_DIONUM', 1);
  ErrorHandler('LJM_WriteName', error);

  { CLK is FIO0 }
  error := LJM_eWriteName(handle, 'SPI_CLK_DIONUM', 0);
  ErrorHandler('LJM_WriteName', error);

  { MISO is FIO3 }
  error := LJM_eWriteName(handle, 'SPI_MISO_DIONUM', 3);
  ErrorHandler('LJM_WriteName', error);

  { MOSI is FIO2 }
  error := LJM_eWriteName(handle, 'SPI_MOSI_DIONUM', 2);
  ErrorHandler('LJM_WriteName', error);

  { Modes: }
  { 0 = A: CPHA=0, CPOL=0 }
  {     Data clocked on the rising edge }
  {     Data changed on the falling edge }
  {     Final clock state low }
  {     Initial clock state low }
  { 1 = B: CPHA=0, CPOL=1 }
  {     Data clocked on the falling edge }
  {     Data changed on the rising edge }
  {     Final clock state low }
  {     Initial clock state low }
  { 2 = C: CPHA=1, CPOL=0 }
  {     Data clocked on the falling edge }
  {    Data changed on the rising edge }
  {    Final clock state high }
  {    Initial clock state high }
  { 3 = D: CPHA=1, CPOL=1 }
  {     Data clocked on the rising edge }
  {     Data changed on the falling edge }
  {     Final clock state high }
  {    Initial clock state high }

  { Selecting Mode: A - CPHA=1, CPOL=1. }
  error := LJM_eWriteName(handle, 'SPI_MODE', 0);
  ErrorHandler('LJM_WriteName', error);

  { Speed Throttle: }
  { Frequency = 1000000000 / (175*(65536-SpeedThrottle) + 1020) }
  { Valid speed throttle values are 1 to 65536 where 0 = 65536. }
  { Note: The above equation and its frequency range were tested }
  { for firmware 1.0009 and may change in the future. }

  { Configuring Max. Speed (~ 1 MHz) }
  error := LJM_eWriteName(handle, 'SPI_SPEED_THROTTLE', 0);
  ErrorHandler('LJM_WriteName', error);

  { Options }
  { bit 0: }
  {     0 = Active low clock select enabled. }
  {     1 = Active low clock select disabled. }
  { bit 1: }
  {     0 = DIO directions are automatically changed }
  {     1 = DIO directions are not automatically changed. }
  { bits 2-3: Reserved }
  { bits 4-7: Number of bits in the last byte. 0 = 8. }
  { bits 8-15: Reserved }

  { Enabling active low clock select pin }
  error := LJM_eWriteName(handle, 'SPI_OPTIONS', 0);
  ErrorHandler('LJM_WriteName', error);

  { Read back and display the SPI settings }

  setLength(aNames, 7);
  setLength(aValues, 7);
  aNames[0] := 'SPI_CS_DIONUM';
  aNames[1] := 'SPI_CLK_DIONUM';
  aNames[2] := 'SPI_MISO_DIONUM';
  aNames[3] := 'SPI_MOSI_DIONUM';
  aNames[4] := 'SPI_MODE';
  aNames[5] := 'SPI_SPEED_THROTTLE';
  aNames[6] := 'SPI_OPTIONS';
  error := LJM_eReadNames(handle, 7, Pointer(aNames), Pointer(aValues), errAddr);
  ErrorHandler('LJM_ReadNames', error, errAddr);

  Writeln('SPI Configuration:');
  for i := 0 to 6 do
  begin
      Writeln('  ' + aNames[i] + ' = ' + FloatToStr(aValues[i]));
  end;


  { Write/Read 4 bytes }
  numBytes := 4;
  error := LJM_eWriteName(handle, 'SPI_NUM_BYTES', numBytes);
  ErrorHandler('LJM_WriteName', error);


  { Setup write bytes }
  setLength(dataWrite, numBytes);
  DecodeTime(now, h, m, s, ms);
  RandSeed := ms;
  for i := 0 to numBytes-1 do
  begin
      dataWrite[i] := Random(256);
  end;

  { Write the bytes }
  setLength(aNames, 1);
  setLength(aWrites, 1);
  setLength(aNumValues, 1);
  aNames[0] := 'SPI_DATA_WRITE';
  aWrites[0] := LJM_WRITE;
  aNumValues[0] := numBytes;
  error := LJM_eNames(handle, 1, Pointer(aNames), Pointer(aWrites), Pointer(aNumValues), Pointer(dataWrite), errAddr);
  ErrorHandler('LJM_WriteName', error, errAddr);

  { Display the bytes written }
  Writeln('');
  for i := 0 to numBytes-1 do
  begin
      Writeln('dataWrite[' + IntToStr(i) + '] = ' + FormatFloat('0.#', dataWrite[i]));
  end;


  { Read the bytes }
  setLength(dataRead, numBytes);
  aNames[0] := 'SPI_DATA_READ';
  aWrites[0] := LJM_READ;
  aNumValues[0] := numBytes;
  error := LJM_eNames(handle, 1, Pointer(aNames), Pointer(aWrites), Pointer(aNumValues), Pointer(dataRead), errAddr);
  ErrorHandler('LJM_WriteName', error, errAddr);

  { Display the bytes read }
  Writeln('');
  for i := 0 to numBytes-1 do
  begin
    Writeln('dataRead[' + IntToStr(i) + '] = ' + FormatFloat('0.#', dataRead[i]));
  end;


  { Close the handle. }
  error := LJM_Close(handle);
  ErrorHandler('LJM_Close', error);

  WaitForUser('Press Enter to exit.');
end.
