{
  StreamBasic.dpr

  Demonstrates how to stream using the eStream functions.
}

program StreamBasic;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  DateUtils,
  LJMDelphi in '..\..\Wrapper\LJMDelphi.pas',
  LJMUtilities in '..\LJMUtilities.pas';

const
  { Scan list names to stream }
  aScanListNames: array[0..3] of PAnsiChar = ('AIN0', 'AIN1', 'AIN2', 'AIN3');

  maxRequests: Integer = 50; { Number of LJM_eStreamRead calls that will be performed }
var
  i: Integer;
  j: Integer;
  handle: Integer;
  error: Integer = 0;
  errorAddress: Integer = -1;

  { Stream variables }
  numAddresses: Integer;
  aScanList: array of Integer;
  aDataTypes: array of Integer;
  scansPerRead: Integer;
  scanRate: Double;
  aData: array of Double;
  deviceScanBacklog: Integer = 0;
  ljmScanBacklog: Integer = 0;
  
  { eWriteNames parameters }
  aNames: array of AnsiString;
  aValues: array of Double;

  { Time variables }
  stTime: TDateTime = 0;
  endTime: TDateTime = 0;
  timeTaken: Double = 0; { In seconds }

  { Stream loop variables }
  totScans: Int64 = 0;
  curSkipSamp: Int64 = 0;
  totSkipSamp: Int64 = 0;
  ainStr: String;

begin
  { Open first found LabJack. }
  error := LJM_Open(LJM_dtAny, LJM_ctAny, 'Any', handle);
  //error := LJM_OpenS('Any', 'Any', 'Any', handle);
  ErrorHandler('LJM_Open', error);

  { Display opened device's information }
  PrintDeviceInfoFromHandle(handle);

  { Stream configuration }
  numAddresses := Length(aScanListNames);
  SetLength(aScanList, numAddresses);
  SetLength(aDataTypes, numAddresses);
  error := LJM_NamesToAddresses(numAddresses, @aScanListNames, Pointer(aScanList), Pointer(aDataTypes));
  ErrorHandler('LJM_NamesToAddresses', error);
  scanRate := 1000; { Scan rate = 1000 Hz }
  scansPerRead := Trunc(scanRate/2);
  SetLength(aData, scansPerRead*numAddresses);

  { Configure scan list for single ended readings }
  SetLength(aNames, numAddresses);
  SetLength(aValues, numAddresses);
  for i := 0 to numAddresses-1 do
  begin
    aNames[i] := aScanListNames[i] + '_NEGATIVE_CH'; { AINX_NEGAITVE_CH }
    aValues[i] := LJM_GND;
  end;
  error := LJM_eWriteNames(handle, numAddresses, Pointer(aNames), Pointer(aValues), errorAddress);
  ErrorHandler('LJM_eWriteNames', error, errorAddress);

  { Start stream }
  error := LJM_eStreamStart(handle, scansPerRead, numAddresses, Pointer(aScanList), scanRate);
  if IsError('LJM_eStreamStart', error) = False then
  begin
    Writeln('Stream started with a scan rate of ' + FormatFloat('0.00', scanRate) + '.');
    Writeln('');

    Writeln('Performing ' + IntToStr(maxRequests) + ' stream reads.');
    Writeln('');

    stTime := Now;

    for i := 0 to maxRequests-1 do
    begin
      error := LJM_eStreamRead(handle, Pointer(aData), deviceScanBacklog, ljmScanBacklog);
      if IsError('LJM_eStreamRead', error) then
        Break;

      totScans := totScans + scansPerRead;

      { Count the skipped samples which are indicated by -9999 values. Missed   }
      { samples occur after a device's stream buffer overflows and are reported }
      { after auto-recover mode ends.                                           }
      curSkipSamp := 0;
      for j := 0 to Length(aData)-1 do
      begin
        if aData[j] = -9999.0 then
          curSkipSamp := curSkipSamp + 1;
      end;
      totSkipSamp := totSkipSamp + curSkipSamp;

      WriteLn('LJM_eStreamRead ' + IntToStr(i+1));
      ainStr := '';
      for j := 0 to numAddresses-1 do
      begin
        ainStr := ainStr + aScanListNames[j] + ' = ' + FormatFloat('0.00000', aData[j]) + ' ';
      end;
      Writeln('  1st scan out of ' + IntToStr(scansPerRead) + ': ' + ainStr);
      Write('  Scans Skipped = ' + FormatFloat('0.0', curSkipSamp/numAddresses));
      Write(', Scan Backlogs: Device = ' + IntToStr(deviceScanBacklog));
      Writeln(', LJM = ' + IntToStr(ljmScanBacklog));
      Writeln('');
    end;

    endTime := Now;
    timeTaken := SecondSpan(stTime, endTime);

    Writeln('');
    Writeln('Total scans = ' + IntToStr(totScans));
    Writeln('Time taken = ' + FloatToStr(timeTaken) + ' seconds');
    Writeln('LJM scan rate = ' + FormatFloat('0.000', scanRate) + ' scans/second');
    Writeln('Timed scan rate = ' + FormatFloat('0.000', totScans/timeTaken) + ' scans/second');
    Writeln('Sample rate = ' + FormatFloat('0.000', totScans*numAddresses/timeTaken) + ' samples/second');
    Writeln('Skipped scans = ' + FormatFloat('0.0', totSkipSamp/numAddresses));
  end;

  error := LJM_eStreamStop(handle);
  ErrorHandler('LJM_eStreamStop', error);
  Writeln('');
  Writeln('Stream Stop');

  { Close the handle. }
  error := LJM_Close(handle);
  ErrorHandler('LJM_Close', error);

  WaitForUser('Press Enter to exit.');
end.

