{
  I2CEeprom.dpr

  Demonstrates I2C communication using the LJM driver. The demonstration uses a
  LJTick-DAC connected to FIO0/FIO1, configures I2C settings, and reads, writes
  and reads bytes from/to the EEPROM.
}

program I2CEeprom;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  LJMDelphi in '..\..\Wrapper\LJMDelphi.pas',
  LJMUtilities in '..\LJMUtilities.pas';

var
  i: Integer;
  handle: Integer;
  error: Integer = 0;
  errorAddress: Integer = -1;
  numFrames: Integer = 3;

  { Variables for the RX/TX transmissions. }
  aNames: array[0..0] of AnsiString;
  aWrites: array[0..0] of Integer;
  aNumValues: array[0..0] of Integer;
  aValues: array[0..4] of Double; { TX/RX bytes will go here. }

  { Time variables for the random generator seed. }
  h: Word;
  m: Word;
  s: Word;
  ms: Word;
begin
  { Open first found LabJack. }
  error := LJM_Open(LJM_dtAny, LJM_ctAny, 'Any', handle);
  //error := LJM_OpenS('Any', 'Any', 'Any', handle);
  ErrorHandler('LJM_Open', error);

  { Display opened device's information. }
  PrintDeviceInfoFromHandle(handle);


  { Configure the I2C communication. }
  error := LJM_eWriteName(handle, 'I2C_SDA_DIONUM', 1); { SDA pin number = 1 (FIO1) }
  ErrorHandler('LJM_WriteName', error);

  error := LJM_eWriteName(handle, 'I2C_SCL_DIONUM', 0); { SCA pin number = 0 (FIO0) }
  ErrorHandler('LJM_WriteName', error);

  { Speed throttle is inversely proportional to clock frequency. 0 = max. }
  error := LJM_eWriteName(handle, 'I2C_SPEED_THROTTLE', 0); { Speed throttle = 0 }
  ErrorHandler('LJM_WriteName', error);

  { Options bits: }
  {  bit0: Reset the I2C bus. }
  {  bit1: Restart w/o stop. }
  {  bit2: Disable clock stretching. }
  error := LJM_eWriteName(handle, 'I2C_OPTIONS', 0); { Options = 0 }
  ErrorHandler('LJM_WriteName', error);

  error := LJM_eWriteName(handle, 'I2C_SLAVE_ADDRESS', 80); { Slave Address of the I2C chip = 80 (0x50) }
  ErrorHandler('LJM_WriteName', error);


  { Initial read of EEPROM bytes 0-3 in the user memory area. We need a single
  { I2C transmission that writes the chip's memory pointer and then reads and
  { then reads the data. }
  error := LJM_eWriteName(handle, 'I2C_NUM_BYTES_TX', 1); { Set the number of bytes to transmit. }
  ErrorHandler('LJM_WriteName', error);
  error := LJM_eWriteName(handle, 'I2C_NUM_BYTES_RX', 4); { Set the number of bytes to receive. }
  ErrorHandler('LJM_WriteName', error);

  { Set the TX bytes. We are sending 1 byte for the address. }
  aNames[0] := 'I2C_DATA_TX';
  aWrites[0] := LJM_WRITE; { Indicates we are writing the values. }
  aNumValues[0] := 1; { The number of bytes. }
  aValues[0] := 0; //Byte 0: Memory pointer = 0
  error := LJM_eNames(handle, 1, @aNames, @aWrites, @aNumValues, @aValues, errorAddress);
  ErrorHandler('LJM_eNames', error, errorAddress);

  error := LJM_eWriteName(handle, 'I2C_GO', 1); {Do the I2C communications. }
  ErrorHandler('LJM_WriteName', error);

  { Read the RX bytes. }
  aNames[0] := 'I2C_DATA_RX';
  aWrites[0] := LJM_READ; { Indicates we are reading the values. }
  aNumValues[0] := 4; { The number of bytes. }
  { aValues[0] to aValues[3] will contain the data. }
  for i := 0 to 3 do
  begin
    aValues[i] := 0;
  end;
  error := LJM_eNames(handle, 1, @aNames, @aWrites, @aNumValues, @aValues, errorAddress);
  ErrorHandler('LJM_eNames', error, errorAddress);

  Writeln('');
  Write('Read User Memory [0-3] = ');
  for i := 0 to 3 do
  begin
    Write(FormatFloat('0.#', aValues[i]) + ' ');
  end;
	Writeln('');


  { Write EEPROM bytes 0-3 in the user memory area, using the page write }
  { technique.  Note that page writes are limited to 16 bytes max, and must be }
  { aligned with the 16-byte page intervals. For instance, if you start }
  { writing at address 14, you can only write two bytes because byte 16 is the }
  { start of a new page. }
  error := LJM_eWriteName(handle, 'I2C_NUM_BYTES_TX', 5); { Set the number of bytes to transmit. }
  ErrorHandler('LJM_eWriteName', error);
  error := LJM_eWriteName(handle, 'I2C_NUM_BYTES_RX', 0); { Set the number of bytes to receive. }
  ErrorHandler('LJM_eWriteName', error);

  { Set the TX bytes. }
  aNames[0] := 'I2C_DATA_TX';
  aWrites[0] := LJM_WRITE; { Indicates we are writing the values. }
  aNumValues[0] := 5; { The number of bytes. }
  aValues[0] := 0; { Byte 0: Memory pointer = 0 }
  { Create 4 new random numbers to write (aValues[1-4]). }
  DecodeTime(now, h, m, s, ms);
  RandSeed := ms;
  for i := 1 to 4 do
  begin
    aValues[i] := Random(256); { 0 to 255 }
  end;
  error := LJM_eNames(handle, 1, @aNames, @aWrites, @aNumValues, @aValues, errorAddress);
  ErrorHandler('LJM_eNames', error, errorAddress);

  error := LJM_eWriteName(handle, 'I2C_GO', 1); { Do the I2C communications. }
  ErrorHandler('LJM_eWriteName', error);

  Write('Write User Memory [0-3] = ');
  for i := 1 to 4 do
  begin
    Write(FormatFloat('0.#', aValues[i]) + ' ');
  end;
  Writeln('');


  { Final read of EEPROM bytes 0-3 in the user memory area. We need a single }
  { I2C transmission that writes the address and then reads the data. }
  error := LJM_eWriteName(handle, 'I2C_NUM_BYTES_TX', 1); { Set the number of bytes to transmit }
  ErrorHandler('LJM_eWriteName', error);
  error := LJM_eWriteName(handle, 'I2C_NUM_BYTES_RX', 4); { Set the number of bytes to receive }
  ErrorHandler('LJM_eWriteName', error);

  { Set the TX bytes. We are sending 1 byte for the address. }
  aNames[0] := 'I2C_DATA_TX';
  aWrites[0] := LJM_WRITE; { Indicates we are writing the values. }
  aNumValues[0] := 1; { The number of bytes }
  aValues[0] := 0; { Byte 0: Memory pointer = 0 }
  error := LJM_eNames(handle, 1, @aNames, @aWrites, @aNumValues, @aValues, errorAddress);
  ErrorHandler('LJM_eNames', error, errorAddress);

  error := LJM_eWriteName(handle, 'I2C_GO', 1); { Do the I2C communications. }
  ErrorHandler('LJM_eWriteName', error);

  { Read the RX bytes. }
  aNames[0] := 'I2C_DATA_RX';
  aWrites[0] := LJM_READ; { Indicates we are reading the values. }
  aNumValues[0] := 4; { The number of bytes.
  { aValues[0] to aValues[3] will contain the data. }
  for i := 0 to 3 do
  begin
    aValues[i] := 0;
  end;
  error := LJM_eNames(handle, 1, @aNames, @aWrites, @aNumValues, @aValues, errorAddress);
  ErrorHandler('LJM_eNames', error, errorAddress);

  Write('Read User Memory [0-3] = ');
	for i := 0 to 3 do
  begin
    Write(FormatFloat('0.#', aValues[i]) + ' ');
  end;
  Writeln('');


  { Close the handle. }
  error := LJM_Close(handle);
  ErrorHandler('LJM_Close', error);

  WaitForUser('Press Enter to exit.');
end.
