{
  CRSpeedTest.dpr

  Performs LabJack operations in a loop and reports the timing statistics for
  the operations.
}

program CRSpeedTest;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  Windows,
  Math,
  LJMDelphi in '..\..\Wrapper\LJMDelphi.pas',
  LJMUtilities in '..\LJMUtilities.pas';

const
  numIterations: Integer = 1000; { Number of iterations to perform in the operations loop }

  { Analog input settings }
  numAIN: Cardinal = 1; { Number of analog inputs to read }
  rangeAIN: Double = 10.0;
  resolutionAIN: Double = 1.0;

  { Digital settings }
  readDigital: Boolean = False;
  writeDigital: Boolean = False;

  { Analog output settings }
  writeDACs: Boolean = False;
var
  i: Integer;
  handle: Integer;
  error: Integer = 0;
  errorAddress: Integer = -1;
  curFrame: Integer = 0;

  { eWriteNames and eName parameters }
  numFrames: Integer;
  aNames: array of AnsiString;
  aWrites: array of Integer;
  aNumValues: array of Integer;
  aValues: array of Double;

  { Time variables }
  freq: int64 = 0;
  stTime: Int64 = 0;
  endTime: Int64 = 0;
  t0: Int64 = 0;
  t1: Int64 = 0;
  ttMS: Double = 0;
  maxMS: Double = 0;
  minMS: Double = 0;
  totalMS : Double = 0;
begin
  { Open first found LabJack. }
  error := LJM_Open(LJM_dtAny, LJM_ctAny, 'Any', handle);
  //error := LJM_OpenS('Any', 'Any', 'Any', handle);
  ErrorHandler('LJM_Open', error);

  { Display opened device's information. }
  PrintDeviceInfoFromHandle(handle);

  if numAIN > 0 then
  begin
    { Configure analog inputs }
    numFrames := numAIN*2;
    setLength(aNames, numAIN*2);
    setLength(aValues, numAIN*2);
    for i := 0 to numAIN-1 do
    begin
      aNames[i*2] := 'AIN' + IntToStr(i) + '_RANGE';
      aValues[i*2] := rangeAIN;
      aNames[i*2+1] := 'AIN' + IntToStr(i) + '_RESOLUTION_INDEX';
      aValues[i*2+1] := resolutionAIN;
    end;
    error := LJM_eWriteNames(handle, numFrames, Pointer(aNames), Pointer(aValues), errorAddress);
    ErrorHandler('LJM_eWriteNames (AIN config.)', error, errorAddress);
  end;

  { Initialize and configure parameters for loop's eNames call }
  numFrames := Trunc(numAIN) + Ord(readDigital) + Ord(WriteDigital) + Ord(writeDACs)*2;
  setLength(aNames, numFrames);
  setLength(aWrites, numFrames);
  setLength(aNumValues, numFrames);
  setLength(aValues, numFrames);
  curFrame := 0;

  { Add analog input reads (AIN 0 to numAIN-1) }
  for i := 0 to numAIN-1 do
  begin
    aNames[curFrame] := 'AIN' + IntToStr(i);
    aWrites[curFrame] := LJM_READ;
    aNumValues[curFrame] := 1;
    aValues[curFrame] := 0;
    curFrame := curFrame + 1;
  end;

  if readDigital then
  begin
    { Add digital I/O read }
    aNames[curFrame] := 'DIO_STATE';
    aWrites[curFrame] := LJM_READ;
    aNumValues[curFrame] := 1;
    aValues[curFrame] := 0;
    curFrame := curFrame + 1;
  end;

  if writeDigital then
  begin
    { Add digital I/O write }
    aNames[curFrame] := 'DIO_STATE';
    aWrites[curFrame] := LJM_WRITE;
    aNumValues[curFrame] := 1;
    aValues[curFrame] := 0; { All output-low }
    curFrame := curFrame + 1;
  end;

  if writeDACs then
  begin
    { Add analog output writes (DAC0-1) }
    for i := 0 to 1 do
    begin
      aNames[curFrame] := 'DAC' + IntToStr(i);
      aWrites[curFrame] := LJM_WRITE;
      aNumValues[curFrame] := 1;
      aValues[curFrame] := 0.0; { 0.0 V }
      curFrame := curFrame + 1;
    end;
  end;

  Writeln('Test Frames:');
  for i := 0 to numFrames-1 do
  begin
    if aWrites[i] = LJM_WRITE
    then Writeln('  WRITE ' + aNames[i])
    else Writeln('  READ ' + aNames[i]);
  end;

  Writeln('');

  Writeln('Performing ' + IntToStr(numIterations) + ' operations...');

  { Performing LJM_eNames operations in a loop. }
  QueryPerformanceFrequency(freq);
  QueryPerformanceCounter(stTime);
  for i := 0 to numIterations-1 do
  begin
    QueryPerformanceCounter(t0);
    error := LJM_eNames(handle, numFrames, Pointer(aNames), Pointer(aWrites), Pointer(aNumValues), Pointer(aValues), errorAddress);
    QueryPerformanceCounter(t1);
    ErrorHandler('LJM_eNames (iteration ' + IntToStr(i+1) + ')', error, errorAddress);
    ttMS := (t1-t0)/freq * 1000.0;
    if minMS = 0.0 then
      minMS := ttMS;
    minMS := Min(ttMS, minMS);
    maxMS := Max(ttMS, MaxMS);
  end;
  QueryPerformanceCounter(endTime);

  Writeln('');

  totalMS := ((endTime-stTime)/freq) * 1000.0;
  Writeln(IntToStr(numIterations) + ' iterations performed:');
  Writeln('  Time taken: ' + FormatFloat('0.000', totalMS) + ' ms');
  Writeln('  Average time per iteration: ' + FormatFloat('0.000', (totalMS/numIterations)) + ' ms');
  Writeln('  Min / Max time for one iteration: ' + FormatFloat('0.000', minMS) + ' ms / ' + FormatFloat('0.000', maxMS) + ' ms');

  Writeln('');
  Writeln('Last Write/Reads:');
  for i := 0 to numFrames-1 do
  begin
    if aWrites[i] = LJM_WRITE
    then Writeln('  WRITE ' + aNames[i] + ' = ' + FloatToStr(aValues[i]))
    else Writeln('  READ ' + aNames[i] + ' = ' + FloatToStr(aValues[i]));
  end;

  { Close the handle. }
  error := LJM_Close(handle);
  ErrorHandler('LJM_Close', error);

  WaitForUser('Press Enter to exit.');
end.
