{******************************************************************************}
{                                                                              }
{  LJMUtilites.pas                                                             }
{                                                                              }
{  Provides some basic help functions.                                         }
{                                                                              }
{******************************************************************************}

unit LJMUtilities;

interface

uses
  SysUtils,
  LJMDelphi; { Path specified in project }

{ Returns the device type based on the passed handle. }
function GetDeviceType(Handle: Integer): Integer;

{ Converts passed integer to its string equivalent. Used by PrintDeviceInfo. }
function NumberToConnectionType(ConnectionType: Integer): String;
function NumberToDeviceType(DeviceType: Integer): String;

{ Gets and displays the device information based on the passed handle. }
procedure PrintDeviceInfoFromHandle(Handle: Integer);

{ Displays the device information based on the passed parameters. }
procedure PrintDeviceInfo(DeviceType, ConnectionType, SerialNumber,
  IPAddressInt, PortOrPipe, PacketMaxBytes: Integer);

{ Waits for user to hit the Enter key. }
procedure WaitForUser(Message: String);

{ Displays error message and returns True if an error is detected, otherwise
  returns False. }
function IsError(Description: String; Error: Integer;
  ErrorAddress: Integer = -1): Boolean;

{ Displays error message, closes all devices, waits for user to hit Enter Key
  and ends program if an error is detected. }
procedure ErrorHandler(Description: String; Error: Integer;
  ErrorAddress: Integer = -1);

implementation

function GetDeviceType(Handle: Integer): Integer;
var
  DeviceType: Integer;
  ConnectionType: Integer;
  SerialNumber: Integer;
  IPAddress: Integer;
  Port: Integer;
  PacketMaxBytes: Integer;
  Error: Integer;
begin
  Error := LJM_GetHandleInfo(Handle, DeviceType, ConnectionType, SerialNumber,
    IPAddress, Port, PacketMaxBytes);
  ErrorHandler('GetDeviceInfo (LJM_GetHandleInfo)', Error);
  Result := DeviceType;
end;

function NumberToConnectionType(ConnectionType: Integer): String;
begin
  Case ConnectionType of
    LJM_ctANY : Result := 'LJM_ctANY';
    LJM_ctUSB : Result := 'LJM_ctUSB';
    LJM_ctTCP : Result := 'LJM_ctTCP';
    LJM_ctETHERNET : Result := 'LJM_ctETHERNET';
    LJM_ctWIFI : Result := 'LJM_ctWIFI';
  else
    Result := 'Unknown connection type';
  end;
end;

function NumberToDeviceType(DeviceType: Integer): String;
begin
  Case DeviceType of
    LJM_dtANY : Result := 'LJM_dtANY';
    LJM_dtT7 : Result := 'LJM_dtT7';
    LJM_dtT4 : Result := 'LJM_dtT4';
    LJM_dtDIGIT : Result := 'LJM_dtDIGIT';
  else
    Result := 'Unknown device type';
  end;
end;

procedure PrintDeviceInfoFromHandle(handle: Integer);
var
  DeviceType: Integer;
  ConnectionType: Integer;
  SerialNumber: Integer;
  IPAddress: Integer;
  Port: Integer;
  PacketMaxBytes: Integer;
  Error: Integer;
begin
  Error := LJM_GetHandleInfo(Handle, DeviceType, ConnectionType, SerialNumber,
    IPAddress, Port, PacketMaxBytes);
  ErrorHandler('PrintDeviceInfoFromHandle (LJM_GetHandleInfo)', Error);

  PrintDeviceInfo(DeviceType, ConnectionType, SerialNumber, IPAddress, Port,
    PacketMaxBytes);
end;

procedure PrintDeviceInfo(DeviceType, ConnectionType, SerialNumber,
  IPAddressInt, PortOrPipe, PacketMaxBytes: Integer);
var
  IPAddressString: array [0..LJM_IPv4_STRING_SIZE-1] of AnsiChar;
begin
  Writeln('DeviceType: ' + NumberToDeviceType(DeviceType));
  Writeln('ConnectionType: ' + NumberToConnectionType(ConnectionType));
  Writeln('SerialNumber: ' + IntToStr(SerialNumber));
  LJM_NumberToIP(IPAddressInt, @IPAddressString);
  Writeln('IP address: ' + IPAddressString);

  if ConnectionType = LJM_ctUSB then
  begin
    Writeln('Pipe: ' + IntToStr(PortOrPipe));
  end
  else
  begin
    Writeln('Port: ' + IntToStr(PortOrPipe));
  end;

  Writeln('The maximum number of bytes you can send to or receive from this ' +
    'device in one packet is ' + IntToStr(PacketMaxBytes) + ' bytes.');
  Writeln('');
end;

procedure WaitForUser(Message: String);
begin
  Writeln('');
  Writeln(Message);
  Readln;
end;

function IsError(Description: String; Error: Integer;
  ErrorAddress: Integer = -1): Boolean;
var
  ErrorString: array [0..LJM_MAX_NAME_SIZE] of AnsiChar;
  MsgType: String;
begin
  Result := False; { Default return }
  if (Error >= LJME_WARNINGS_BEGIN) and (Error <= LJME_WARNINGS_END) then
    MsgType := 'warning'
  else if error <> LJME_NOERROR then
    MsgType := 'error'
  else
    Exit;

  LJM_ErrorToString(Error, @ErrorString);
  Writeln(Description + ' ' + MsgType + ': ' + string(ErrorString) + '(' +
    IntToStr(Error) + ')');
  if ErrorAddress > -1
  then Writeln('Error address = ' + IntToStr(ErrorAddress));
  if MsgType = 'error' then
    Result := True;
end;

procedure ErrorHandler(Description: String; Error: Integer;
  ErrorAddress: Integer = -1);
begin
  if IsError(Description, Error, ErrorAddress) then
  begin
    Writeln('Closing all devices and terminating application.');
    LJM_CloseAll();
    WaitForUser('Press Enter to exit.');
    Halt;
  end;
end;

end.
