{******************************************************************************}
{                                                                              }
{  StreamBasicWithStreamOut.dpr                                                }
{                                                                              }
{  Demonstrates setting up stream-in and stream-out together, then reading     }
{  stream-in values.                                                           }
{                                                                              }
{  Connect a wire from AIN0 or AIN1 to DAC0 to see the effect of stream-out    }
{  on stream-in.                                                               }
{                                                                              }
{******************************************************************************}

program StreamBasicWithStreamOut;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  DateUtils,
  LJMDelphi in '..\..\..\Wrapper\LJMDelphi.pas',
  LJMUtilities in '..\..\LJMUtilities.pas';

var
  i: Integer;
  j: Integer;
  k: Integer;
  Handle: Integer;
  Error: Integer;
  ErrorAddress: Integer;
  Value: Double;

  { Stream out variables }
  NumAddressesOut: Integer;
  aNamesOut: array of AnsiString;
  aAddressesOut: array of Integer;
  aTypesOut: array of Integer;

  { Stream variables }
  NumAddressesIn: Integer;
  MaxRequests: Integer;
  aScanListNames: array of AnsiString;
  aScanList: array of Integer;
  aDataTypes: array of Integer;
  ScansPerRead: Integer;
  ScanRate: Double;
  aData: array of Double;
  DeviceScanBacklog: Integer;
  LJMScanBacklog: Integer;

  { LJM_eWriteNames parameters }
  NumFrames: Integer;
  aNames: array of AnsiString;
  aValues: array of Double;

  { Time variables }
  StTime: TDateTime;
  EndTime: TDateTime;
  TimeTaken: Double; { In seconds }

  { Stream loop variables }
  TotScans: Int64;
  CurSkipSamp: Int64;
  TotSkipSamp: Int64;
  AINStr: String;

begin
  Handle := 0;
  ErrorAddress := -1;

  { Open first found LabJack }

  { Any device, Any connection, Any identifier }
  Error := LJM_OpenS('Any', 'Any', 'Any', Handle);

  { T7 device, Any connection, Any identifier }
  { Error := LJM_OpenS('T7', 'Any', 'Any', Handle); }

  { T4 device, Any connection, Any identifier }
  { Error := LJM_OpenS('T4', 'Any', 'Any', Handle); }

  { Any device, Any connection, Any identifier }
  { Error := LJM_Open(LJM_dtAny, LJM_ctAny, 'Any', Handle); }

  ErrorHandler('LJM_OpenS', Error);

  { Display opened device's information }
  PrintDeviceInfoFromHandle(Handle);

  { In case stream mode is active, try to stop stream }
  LJM_eStreamStop(Handle);

  { Setup Stream Out }
  NumAddressesOut := 1; { Number of stream outputs. }
  SetLength(aNamesOut, NumAddressesOut);
  aNamesOut[0] := 'DAC0';
  SetLength(aAddressesOut, NumAddressesOut);
  SetLength(aTypesOut, NumAddressesOut);
  Error := LJM_NamesToAddresses(NumAddressesOut, Pointer(aNamesOut),
    Pointer(aAddressesOut), Pointer(aTypesOut));
  ErrorHandler('LJM_NamesToAddresses', Error);

  { Allocate memory for the stream-out buffer }
  Error := LJM_eWriteName(Handle, 'STREAM_OUT0_TARGET', aAddressesOut[0]);
  ErrorHandler('LJM_eWriteName (STREAM_OUT0_TARGET)', Error);
  Error := LJM_eWriteName(Handle, 'STREAM_OUT0_BUFFER_SIZE', 512);
  ErrorHandler('LJM_eWriteName (STREAM_OUT0_BUFFER_SIZE)', Error);
  Error := LJM_eWriteName(Handle, 'STREAM_OUT0_ENABLE', 1);
  ErrorHandler('LJM_eWriteName (STREAM_OUT0_ENABLE)', Error);

  { Write values to the stream-out buffer }
  Error := LJM_eWriteName(Handle, 'STREAM_OUT0_LOOP_SIZE', 6);
  ErrorHandler('LJM_eWriteName (STREAM_OUT0_LOOP_SIZE)', Error);
  Error := LJM_eWriteName(Handle, 'STREAM_OUT0_BUFFER_F32', 0.0); { 0.0 V }
  ErrorHandler('LJM_eWriteName (STREAM_OUT0_BUFFER_F32 0)', Error);
  Error := LJM_eWriteName(Handle, 'STREAM_OUT0_BUFFER_F32', 1.0); { 1.0 V }
  ErrorHandler('LJM_eWriteName (STREAM_OUT0_BUFFER_F32 1)', Error);
  Error := LJM_eWriteName(Handle, 'STREAM_OUT0_BUFFER_F32', 2.0); { 2.0 V }
  ErrorHandler('LJM_eWriteName (STREAM_OUT0_BUFFER_F32 2)', Error);
  Error := LJM_eWriteName(Handle, 'STREAM_OUT0_BUFFER_F32', 3.0); { 3.0 V }
  ErrorHandler('LJM_eWriteName (STREAM_OUT0_BUFFER_F32 3)', Error);
  Error := LJM_eWriteName(Handle, 'STREAM_OUT0_BUFFER_F32', 4.0); { 4.0 V }
  ErrorHandler('LJM_eWriteName (STREAM_OUT0_BUFFER_F32 4)', Error);
  Error := LJM_eWriteName(Handle, 'STREAM_OUT0_BUFFER_F32', 5.0); { 5.0 V }
  ErrorHandler('LJM_eWriteName (STREAM_OUT0_BUFFER_F32 5)', Error);

  Error := LJM_eWriteName(Handle, 'STREAM_OUT0_SET_LOOP', 1);
  ErrorHandler('LJM_eWriteName (STREAM_OUT0_SET_LOOP)', Error);

  Error := LJM_eReadName(Handle, 'STREAM_OUT0_BUFFER_STATUS', Value);
  ErrorHandler('LJM_eReadName (STREAM_OUT0_BUFFER_STATUS)', Error);
  Writeln('STREAM_OUT0_BUFFER_STATUS = ' + FloatToStr(Value));

  { Stream configuration }
  NumAddressesIn := 2;  { Number stream inputs. }

  { Allocate scan list array which will include both the stream in and out
    addresses. }
  SetLength(aScanList, NumAddressesIn+NumAddressesOut);

  { Scan list names to stream. }
  SetLength(aScanListNames, NumAddressesIn);
  aScanListNames[0] := 'AIN0';
  aScanListNames[1] := 'AIN1';
  SetLength(aDataTypes, NumAddressesIn);
  { Scan list addresses to stream. LJM_eStreamStart uses Modbus addresses. }
  Error := LJM_NamesToAddresses(NumAddressesIn, Pointer(aScanListNames),
    Pointer(aScanList), Pointer(aDataTypes));
  ErrorHandler('LJM_NamesToAddresses', Error);


  { Add the scan list outputs to the end of the scan list.
    STREAM_OUT0 = 4800, STREAM_OUT1 = 4801, etc. }
  aScanList[NumAddressesIn+0] := 4800; { STREAM_OUT0 }
  { If we had more STREAM_OUTs }
  { aScanList[numAddressesIn+1] := 4801; } { STREAM_OUT1 }
  { aScanList[numAddressesIn+2] := 4802; } { STREAM_OUT2 }
  { aScanList[numAddressesIn+3] := 4803; } { STREAM_OUT3 }

  { Configure the analog inputs negative channel, range, settling time and
    resolution.

  { When streaming, negative channels and ranges can be configured for
    individual analog inputs, but the stream has only one settling time and
    resolution. }
  if GetDeviceType(Handle) = LJM_dtT4 then
  begin
    { LabJack T4 configuration. }

    { AIN0 and AIN1 ranges are +/-10 V, stream settling is 0 (default) and
      stream resolution index is 0 (default). }
    NumFrames := 4;
    SetLength(aNames, NumFrames);
    aNames[0] := 'AIN0_RANGE';
    aNames[1] := 'AIN1_RANGE';
    aNames[2] := 'STREAM_SETTLING_US';
    aNames[3] := 'STREAM_RESOLUTION_INDEX';
    SetLength(aValues, NumFrames);
    aValues[0] := 10.0;
    aValues[1] := 10.0;
    aValues[2] := 0;
    aValues[3] := 0;
  end
  else
  begin
    { LabJack T7 and other devices configuration. }

    { Ensure triggered stream is disabled. }
    LJM_eWriteName(Handle, 'STREAM_TRIGGER_INDEX', 0);
    ErrorHandler('LJM_eWriteName (STREAM_TRIGGER_INDEX)', Error);

    { Enabling internally-clocked stream. }
    LJM_eWriteName(Handle, 'STREAM_CLOCK_SOURCE', 0);
    ErrorHandler('LJM_eWriteName (STREAM_CLOCK_SOURCE)', Error);

    { All negative channels are single-ended, AIN0 and AIN1 ranges are +/-10 V,
      stream settling is 0 (default) and stream resolution index is
      0 (default). }
    NumFrames := 5;
    SetLength(aNames, NumFrames);
    aNames[0] := 'AIN_ALL_NEGATIVE_CH';
    aNames[1] := 'AIN0_RANGE';
    aNames[2] := 'AIN1_RANGE';
    aNames[3] := 'STREAM_SETTLING_US';
    aNames[4] := 'STREAM_RESOLUTION_INDEX';
    SetLength(aValues, NumFrames);
    aValues[0] := LJM_GND;
    aValues[1] := 10.0;
    aValues[2] := 10.0;
    aValues[3] := 0;
    aValues[4] := 0;
  end;
  { Write the analog inputs' negative channels (when applicable), ranges,
    stream settling time and stream resolution configuration. }
  Error := LJM_eWriteNames(Handle, NumFrames, Pointer(aNames), Pointer(aValues),
    ErrorAddress);
  ErrorHandler('LJM_eWriteNames', Error, ErrorAddress);

  { Configure and start stream }
  ScanRate := 1000; { Scan rate = 1000 Hz }
  ScansPerRead := 60; { # scans returned by LJM_eStreamRead call }
  SetLength(aData, ScansPerRead*NumAddressesIn);
  Error := LJM_eStreamStart(Handle, ScansPerRead,
    NumAddressesIn+NumAddressesOut, Pointer(aScanList), ScanRate);
  if IsError('LJM_eStreamStart', Error) = False then
  begin
    Writeln('Stream started with a scan rate of ' +
      FormatFloat('0.00', ScanRate) + '.');
    Writeln('');

    MaxRequests := 50;
    Writeln('Performing ' + IntToStr(MaxRequests) + ' stream reads.');
    Writeln('');

    TotScans := 0;
    TotSkipSamp := 0;

    StTime := Now;

    for i := 0 to MaxRequests-1 do
    begin
      Error := LJM_eStreamRead(Handle, Pointer(aData), DeviceScanBacklog,
        LJMScanBacklog);
      if IsError('LJM_eStreamRead', Error) then
        Break;

      TotScans := TotScans + ScansPerRead;

      { Count the skipped samples which are indicated by -9999 values. Missed
        samples occur after a device's stream buffer overflows and are reported
        after auto-recover mode ends. }
      CurSkipSamp := 0;
      for j := 0 to Length(aData)-1 do
      begin
        if aData[j] = -9999.0 then
          CurSkipSamp := CurSkipSamp + 1;
      end;
      TotSkipSamp := TotSkipSamp + CurSkipSamp;

      Writeln('LJM_eStreamRead #' + IntToStr(i+1));
      AINStr := '';
      for j := 0 to ScansPerRead-1 do
      begin
        for k := 0 to NumAddressesIn-1 do
        begin
          Write('  ' + aScanListNames[k] +  ' = ' +
            FormatFloat('0.00000', aData[j*NumAddressesIn + k]) + ',');
        end;
        Writeln('');
      end;
      Writeln('  Scans Skipped = ' +
        FormatFloat('0.0', CurSkipSamp/NumAddressesIn) +
        ', Scan Backlogs: Device = ' + IntToStr(DeviceScanBacklog) +
        ', LJM = ' + IntToStr(LJMScanBacklog));
      Writeln('');
    end;

    EndTime := Now;
    TimeTaken := SecondSpan(StTime, EndTime);

    Writeln('');
    Writeln('Total Scans = ' + IntToStr(TotScans));
    Writeln('Skipped Scans = ' + FormatFloat('0.0', TotSkipSamp/NumAddressesIn));
    Writeln('Time Taken = ' + FloatToStr(TimeTaken) + ' seconds');
    Writeln('LJM Scan Rate = ' + FormatFloat('0.000', ScanRate) +
      ' scans/second');
    Writeln('Timed Scan Rate = ' + FormatFloat('0.000', TotScans/TimeTaken) +
      ' scans/second');
    Writeln('Timed Sample Rate = ' +
      FormatFloat('0.000', TotScans*NumAddressesIn/TimeTaken) +
      ' samples/second');
  end;

  Error := LJM_eStreamStop(Handle);
  ErrorHandler('LJM_eStreamStop', Error);
  Writeln('');
  Writeln('Stream Stop');

  { Close the handle. }
  Error := LJM_Close(Handle);
  ErrorHandler('LJM_Close', Error);

  WaitForUser('Press Enter to exit.');
end.

