{******************************************************************************}
{                                                                              }
{  StreamBurst.dpr                                                             }
{                                                                              }
{  Demonstrates how to use the LJM_StreamBurst function for streaming.         }
{                                                                              }
{******************************************************************************}

program StreamBurst;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  DateUtils,
  LJMDelphi in '..\..\..\Wrapper\LJMDelphi.pas',
  LJMUtilities in '..\..\LJMUtilities.pas';

var
  i: Integer;
  Handle: Integer;
  Error: Integer;
  ErrorAddress: Integer;

  { Stream variables }
  NumAddresses: Integer;
  aScanListNames: array of AnsiString;
  aScanList: array of Integer;
  aDataTypes: array of Integer;
  ScanRate: Double;
  NumScans: Cardinal;
  aData: array of Double;
  SkippedTotal: Int64;

  { LJM_eWriteNames parameters }
  NumFrames: Integer;
  aNames: array of AnsiString;
  aValues: array of Double;

  { Time variables }
  StTime: TDateTime;
  EndTime: TDateTime;
  Time: Double; { In seconds }
begin
  Handle := 0;
  ErrorAddress := -1;

  { Open first found LabJack }

  { Any device, Any connection, Any identifier }
  Error := LJM_OpenS('Any', 'Any', 'Any', Handle);

  { T7 device, Any connection, Any identifier }
  { Error := LJM_OpenS('T7', 'Any', 'Any', Handle); }

  { T4 device, Any connection, Any identifier }
  { Error := LJM_OpenS('T4', 'Any', 'Any', Handle); }

  { Any device, Any connection, Any identifier }
  { Error := LJM_Open(LJM_dtAny, LJM_ctAny, 'Any', Handle); }

  ErrorHandler('LJM_OpenS', Error);

  { Display opened device's information }
  PrintDeviceInfoFromHandle(Handle);

  { Stream configuration }

  NumAddresses := 2;
  ScanRate := 10000; { Scan rate = 10000 Hz }
  NumScans := 20000; { Number of scans to perform }
  SetLength(aData, NumScans*Cardinal(NumAddresses)); { Data array for samples. }

  { Scan list names to stream. }
  SetLength(aScanListNames, NumAddresses);
  aScanListNames[0] := 'AIN0';
  aScanListNames[1] := 'AIN1';
  SetLength(aScanList, NumAddresses);
  SetLength(aDataTypes, NumAddresses);
  { Scan list addresses to stream. LJM_eStreamBurst uses Modbus addresses. }
  Error := LJM_NamesToAddresses(NumAddresses, Pointer(aScanListNames),
    Pointer(aScanList), Pointer(aDataTypes));
  ErrorHandler('LJM_NamesToAddresses', Error);

  { Configure the analog inputs negative channel, range, settling time and
    resolution.
  { When streaming, negative channels and ranges can be configured for
    individual analog inputs, but the stream has only one settling time and
    resolution. }
  if GetDeviceType(Handle) = LJM_dtT4 then
  begin
    { LabJack T4 configuration. }

    { AIN0 and AIN1 ranges are +/-10 V, stream settling is 0 (default) and
      stream resolution index is 0 (default). }
    NumFrames := 4;
    SetLength(aNames, NumFrames);
    aNames[0] := 'AIN0_RANGE';
    aNames[1] := 'AIN1_RANGE';
    aNames[2] := 'STREAM_SETTLING_US';
    aNames[3] := 'STREAM_RESOLUTION_INDEX';
    SetLength(aValues, NumFrames);
    aValues[0] := 10.0;
    aValues[1] := 10.0;
    aValues[2] := 0;
    aValues[3] := 0;
  end
  else
  begin
    { LabJack T7 and other devices configuration. }

    { Ensure triggered stream is disabled. }
    LJM_eWriteName(Handle, 'STREAM_TRIGGER_INDEX', 0);
    ErrorHandler('LJM_eWriteName (STREAM_TRIGGER_INDEX)', Error);

    { Enabling internally-clocked stream. }
    LJM_eWriteName(Handle, 'STREAM_CLOCK_SOURCE', 0);
    ErrorHandler('LJM_eWriteName (STREAM_CLOCK_SOURCE)', Error);

    { All negative channels are single-ended, AIN0 and AIN1 ranges are +/-10 V,
      stream settling is 0 (default) and stream resolution index is
      0 (default). }
    NumFrames := 5;
    SetLength(aNames, NumFrames);
    aNames[0] := 'AIN_ALL_NEGATIVE_CH';
    aNames[1] := 'AIN0_RANGE';
    aNames[2] := 'AIN1_RANGE';
    aNames[3] := 'STREAM_SETTLING_US';
    aNames[4] := 'STREAM_RESOLUTION_INDEX';
    SetLength(aValues, NumFrames);
    aValues[0] := LJM_GND;
    aValues[1] := 10.0;
    aValues[2] := 10.0;
    aValues[3] := 0;
    aValues[4] := 0;
  end;
  { Write the analog inputs' negative channels (when applicable), ranges,
    stream settling time and stream resolution configuration. }
  Error := LJM_eWriteNames(Handle, NumFrames, Pointer(aNames), Pointer(aValues),
    ErrorAddress);
  ErrorHandler('LJM_eWriteNames', Error, ErrorAddress);

  Writeln('Scan list:');
  for i := 0 to NumAddresses-1 do
  begin
    Writeln('  ' + aScanListNames[i]);
  end;
  Writeln('Scan rate = ' + FloatToStr(ScanRate) + ' Hz');
  Writeln('Sample rate = ' + FloatToStr(ScanRate*NumAddresses) + ' Hz');
  Writeln('Total number of scans = ' + IntToStr(NumScans));
  Writeln('Total number of samples = ' +
    IntToStr(NumScans*Cardinal(NumAddresses)));
  Writeln('Seconds of samples = ' + FloatToStr(NumScans/ScanRate) + ' seconds');

  Writeln('');
  Writeln('Streaming with LJM_StreamBurst...');

  { Stream data using LJM_StreamBurst }
  StTime := Now;

  Error := LJM_StreamBurst(Handle, NumAddresses, Pointer(aScanList), ScanRate,
    NumScans, Pointer(aData));
  ErrorHandler('LJM_StreamBurst', Error);

  EndTime := Now;
  Time := SecondSpan(StTime, EndTime);

  Writeln('Done');

  { Count the skipped samples which are indicated by -9999 values. Missed
    samples occur after a device's stream buffer overflows and are reported
    after auto-recover mode ends. }
  SkippedTotal := 0;
  for i := 0 to Length(aData)-1 do
  begin
  if aData[i] = -9999.0 then
    SkippedTotal := SkippedTotal + 1;
  end;

  Writeln('');
  Writeln('Skipped scans = ' + IntToStr(SkippedTotal div NumAddresses));
  Writeln('Time taken = ' + FormatFloat('0.000', Time) + ' seconds');

  Writeln('');
  Writeln('Last scan (' + IntToStr(NumScans-1) + '):');
  for i := 0 to NumAddresses-1 do
  begin
    Writeln('  ' + aScanListNames[i] + ' = ' +
      FormatFloat('0.00000',
      aData[(NumScans-1)*Cardinal(NumAddresses)+Cardinal(i)]));
  end;

  { Close the handle. }
  Error := LJM_Close(Handle);
  ErrorHandler('LJM_Close', Error);

  WaitForUser('Press Enter to exit.');
end.
