{******************************************************************************}
{                                                                              }
{  DualAINLoop.dpr                                                             }
{                                                                              }
{  Demonstrates reading 2 analog inputs (AINs) in a loop from a LabJack.       }
{                                                                              }
{******************************************************************************}

program DualAINLoop;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  LJMDelphi in '..\..\..\Wrapper\LJMDelphi.pas',
  LJMUtilities in '..\..\LJMUtilities.pas';

var
  i: Integer;
  Handle: Integer;
  Error: Integer;
  NumFrames: Integer;
  ErrorAddress: Integer;
  aNames: array of AnsiString;
  aValues: array of Double;
begin
  Handle := 0;
  ErrorAddress := -1;

  { Open first found LabJack }

  { Any device, Any connection, Any identifier }
  Error := LJM_OpenS('Any', 'Any', 'Any', Handle);

  { T7 device, Any connection, Any identifier }
  { Error := LJM_OpenS('T7', 'Any', 'Any', Handle); }

  { T4 device, Any connection, Any identifier }
  { Error := LJM_OpenS('T4', 'Any', 'Any', Handle); }

  { Any device, Any connection, Any identifier }
  { Error := LJM_Open(LJM_dtAny, LJM_ctAny, 'Any', Handle); }

  ErrorHandler('LJM_OpenS', Error);

  { Display opened device's information. }
  PrintDeviceInfoFromHandle(Handle);

  { Setup and call LJM_eWriteNames to configure AINs. }
  if GetDeviceType(Handle) = LJM_dtT4 then
  begin
    { LabJack T4 configuration. }

    { AIN0 and AIN1:
        Range = +/-10 V (10.0). Only AIN0-AIN3 can support +/-10 V range.
        Resolution index = 0 (default)
        Settling = 0 (auto) }
    NumFrames := 6;
    SetLength(aNames, NumFrames);
    aNames[0] := 'AIN0_RANGE';
    aNames[1] := 'AIN0_RESOLUTION_INDEX';
    aNames[2] := 'AIN0_SETTLING_US';
    aNames[3] := 'AIN1_RANGE';
    aNames[4] := 'AIN1_RESOLUTION_INDEX';
    aNames[5] := 'AIN1_SETTLING_US';
    SetLength(aValues, NumFrames);
    aValues[0] := 10;
    aValues[1] := 0;
    aValues[2] := 0;
    aValues[3] := 10;
    aValues[4] := 0;
    aValues[5] := 0;
  end
  else
  begin
    { LabJack T7 and other devices configuration. }

    { AIN0 and AIN1:
        Negative Channel = 199 (Single-ended)
        Range = +/-10 V (10.0)
        Resolution index = 0 (default)
        Settling = 0 (auto) }
    NumFrames := 8;
    SetLength(aNames, NumFrames);
    aNames[0] := 'AIN0_NEGATIVE_CH';
    aNames[1] := 'AIN0_RANGE';
    aNames[2] := 'AIN0_RESOLUTION_INDEX';
    aNames[3] := 'AIN0_SETTLING_US';
    aNames[4] := 'AIN1_NEGATIVE_CH';
    aNames[5] := 'AIN1_RANGE';
    aNames[6] := 'AIN1_RESOLUTION_INDEX';
    aNames[7] := 'AIN1_SETTLING_US';
    SetLength(aValues, NumFrames);
    aValues[0] := 199;
    aValues[1] := 10;
    aValues[2] := 0;
    aValues[3] := 0;
    aValues[4] := 199;
    aValues[5] := 10;
    aValues[6] := 0;
    aValues[7] := 0;
  end;
  Error := LJM_eWriteNames(Handle, NumFrames, Pointer(aNames), Pointer(aValues),
    ErrorAddress);
  ErrorHandler('LJM_eWriteNames (Configuration)', Error, ErrorAddress);

  Writeln('Configuration:');
  for i := 0 to NumFrames-1 do
  begin
    Writeln('  ' + aNames[i] + ' = ' + FloatToStr(aValues[i]));
  end;

  Writeln('');

  { Reading AIN0 and AIN1 in a loop with a 1 second delay. }

  { Setup aNames and aValues for LJM_eReadNames calls. }
  NumFrames := 2;
  SetLength(aNames, NumFrames);
  aNames[0] := 'AIN0';
  aNames[1] := 'AIN1';
  SetLength(aValues, NumFrames);
  aValues[0] := 0;
  aValues[1] := 0;

  Writeln('Starting read loop. Press Ctrl+C to exit the application.');
  Sleep(1000); { Extra 1 second delay to read the message. }
  while True do
  begin
    Sleep(1000); { Wait 1 second }
    { LJM_eReadNames call to read analog inputs. }
    Error := LJM_eReadNames(Handle, NumFrames, Pointer(aNames),
      Pointer(aValues), ErrorAddress);
    ErrorHandler('LJM_eReadNames (Readings)', Error, ErrorAddress);

    Writeln('');
    for i := 0 to NumFrames-1 do
    begin
      Writeln(aNames[i] + ' = ' + FormatFloat('0.00000', aValues[i]) + ' V');
    end;
  end;

  { Close the handle. }
  Error := LJM_Close(Handle);
  ErrorHandler('LJM_Close', Error);

  WaitForUser('Press Enter to exit.');
end.
