{******************************************************************************}
{                                                                              }
{  I2CEeprom.dpr                                                               }
{                                                                              }
{  Demonstrates I2C communication using a LabJack. The demonstration uses a    }
{  LJTick-DAC connected to FIO0/FIO1 for the T7 or FIO4/FIO5 for the T4, and   }
{  configures the I2C settings. Then a read, write and again a read are        }
{  performed on the LJTick-DAC EEPROM.                                         }
{                                                                              }
{******************************************************************************}

program I2CEeprom;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  LJMDelphi in '..\..\..\Wrapper\LJMDelphi.pas',
  LJMUtilities in '..\..\LJMUtilities.pas';

var
  i: Integer;
  Handle: Integer;
  Error: Integer;
  ErrorAddress: Integer;
  Name: AnsiString;
  NumBytes: Integer;
  aBytes: array [0..4] of Byte;

  { Time variables for the random generator seed. }
  h: Word;
  m: Word;
  s: Word;
  ms: Word;
begin
  Handle := 0;
  ErrorAddress := -1;

  { Open first found LabJack }

  { Any device, Any connection, Any identifier }
  Error := LJM_OpenS('Any', 'Any', 'Any', Handle);

  { T7 device, Any connection, Any identifier }
  { Error := LJM_OpenS('T7', 'Any', 'Any', Handle); }

  { T4 device, Any connection, Any identifier }
  { Error := LJM_OpenS('T4', 'Any', 'Any', Handle); }

  { Any device, Any connection, Any identifier }
  { Error := LJM_Open(LJM_dtAny, LJM_ctAny, 'Any', Handle); }

  ErrorHandler('LJM_OpenS', Error);

  { Display opened device's information. }
  PrintDeviceInfoFromHandle(Handle);

  { Configure the I2C communication. }
  if GetDeviceType(Handle) = LJM_dtT4 then
  begin
    { Configure FIO4 and FIO5 as digital I/O. }
    Error := LJM_eWriteName(handle, 'DIO_INHIBIT', $FFFCF);
    ErrorHandler('LJM_WriteName (DIO_INHIBIT)', Error);
    Error := LJM_eWriteName(handle, 'DIO_ANALOG_ENABLE', $00000);
    ErrorHandler('LJM_WriteName (DIO_ANALOG_ENABLE)', Error);

    { For the T4, using FIO4 and FIO5 for SCL and SDA pins. FIO0 to FIO3 are
      reserved for analog inputs, and digital lines are required. }
    LJM_eWriteName(Handle, 'I2C_SDA_DIONUM', 5);  { FIO5 }
    ErrorHandler('LJM_WriteName (I2C_SDA_DIONUM)', Error);
    LJM_eWriteName(Handle, 'I2C_SCL_DIONUM', 4);  { FIO4 }
    ErrorHandler('LJM_WriteName (I2C_SCL_DIONUM)', Error);
  end
  else
  begin
    { For the T7 and other devices, using FIO0 and FIO1 for the SCL and SDA
      pins. }
    LJM_eWriteName(Handle, 'I2C_SDA_DIONUM', 1);  { FIO1 }
    ErrorHandler('LJM_WriteName (I2C_SDA_DIONUM)', Error);
    LJM_eWriteName(Handle, 'I2C_SCL_DIONUM', 0);  { FIO0 }
    ErrorHandler('LJM_WriteName (I2C_SCL_DIONUM)', Error);
  end;

  { Speed throttle is inversely proportional to clock frequency. 0 = max. }

  { Speed throttle = 65516 (~100 kHz) }
  Error := LJM_eWriteName(Handle, 'I2C_SPEED_THROTTLE', 65516);
  ErrorHandler('LJM_WriteName (I2C_SPEED_THROTTLE)', Error);

  { Options bits:
      bit0: Reset the I2C bus.
      bit1: Restart w/o stop.
      bit2: Disable clock stretching. }
  Error := LJM_eWriteName(Handle, 'I2C_OPTIONS', 0); { Options = 0 }
  ErrorHandler('LJM_WriteName (I2C_OPTIONS)', Error);

  { Slave Address of the I2C chip = 80 (0x50) }
  Error := LJM_eWriteName(Handle, 'I2C_SLAVE_ADDRESS', 80);
  ErrorHandler('LJM_WriteName (I2C_SLAVE_ADDRESS)', Error);

  { Initial read of EEPROM bytes 0-3 in the user memory area. We need a single
    I2C transmission that writes the chip's memory pointer and then reads and
    then reads the data. }

  { Number of bytes to transmit }
  Error := LJM_eWriteName(Handle, 'I2C_NUM_BYTES_TX', 1);
  ErrorHandler('LJM_WriteName', Error);
  { Number of bytes to receive }
  Error := LJM_eWriteName(Handle, 'I2C_NUM_BYTES_RX', 4);
  ErrorHandler('LJM_WriteName', Error);

  { Set the TX bytes. We are sending 1 byte for the address. }
  Name := 'I2C_DATA_TX';
  NumBytes := 1; { The number of bytes. }
  aBytes[0] := 0; { Byte 0: Memory pointer = 0 }
  Error := LJM_eWriteNameByteArray(Handle, PAnsiChar(Name), NumBytes, @aBytes,
    ErrorAddress);
  ErrorHandler('LJM_eWriteNameByteArray', Error, ErrorAddress);

  Error := LJM_eWriteName(Handle, 'I2C_GO', 1); { Do the I2C communications. }
  ErrorHandler('LJM_WriteName', Error);

  { Read the RX bytes. }
  Name := 'I2C_DATA_RX';
  NumBytes := 4; { The number of bytes. }
  { aBytes[0] to aBytes[3] will contain the data. }
  for i := 0 to NumBytes-1 do
  begin
    aBytes[i] := 0;
  end;
  Error := LJM_eReadNameByteArray(Handle, PAnsiChar(Name), NumBytes, @aBytes,
    ErrorAddress);
  ErrorHandler('LJM_eReadNameByteArray', Error, ErrorAddress);

  Writeln('');
  Write('Read User Memory [0-3] = ');
  for i := 0 to NumBytes-1 do
  begin
    Write(FormatFloat('0.#', aBytes[i]) + ' ');
  end;
  Writeln('');

  { Write EEPROM bytes 0-3 in the user memory area, using the page write
    technique. Note that page writes are limited to 16 bytes max, and must be
    aligned with the 16-byte page intervals. For instance, if you start
    writing at address 14, you can only write two bytes because byte 16 is the
    start of a new page. }

  { Number of bytes to transmit }
  Error := LJM_eWriteName(Handle, 'I2C_NUM_BYTES_TX', 5);
  ErrorHandler('LJM_eWriteName', Error);
  { Number of bytes to receive }
  Error := LJM_eWriteName(Handle, 'I2C_NUM_BYTES_RX', 0);
  ErrorHandler('LJM_eWriteName', Error);

  { Set the TX bytes. }
  Name := 'I2C_DATA_TX';
  NumBytes := 5; { The number of bytes. }
  aBytes[0] := 0; { Byte 0: Memory pointer = 0 }
  { Create 4 new random numbers to write (aBytes[1] to aBytes[4]). }
  DecodeTime(now, h, m, s, ms);
  RandSeed := ms;
  for i := 1 to NumBytes-1 do
  begin
    aBytes[i] := Random(256); { 0 to 255 }
  end;
  Error := LJM_eWriteNameByteArray(Handle, PAnsiChar(Name), NumBytes, @aBytes,
    ErrorAddress);
  ErrorHandler('LJM_eWriteNameByteArray', Error, ErrorAddress);

  Error := LJM_eWriteName(Handle, 'I2C_GO', 1); { Do the I2C communications. }
  ErrorHandler('LJM_eWriteName', Error);

  Write('Write User Memory [0-3] = ');
  for i := 1 to NumBytes-1 do
  begin
    Write(FormatFloat('0.#', aBytes[i]) + ' ');
  end;
  Writeln('');

  { Final read of EEPROM bytes 0-3 in the user memory area. We need a single
    I2C transmission that writes the address and then reads the data. }

  { Number of bytes to transmit }
  Error := LJM_eWriteName(Handle, 'I2C_NUM_BYTES_TX', 1);
  ErrorHandler('LJM_eWriteName', Error);
  { Number of bytes to receive }
  Error := LJM_eWriteName(Handle, 'I2C_NUM_BYTES_RX', 4);
  ErrorHandler('LJM_eWriteName', Error);

  { Set the TX bytes. We are sending 1 byte for the address. }
  Name := 'I2C_DATA_TX';
  NumBytes := 1; { The number of bytes }
  aBytes[0] := 0; { Byte 0: Memory pointer = 0 }
  Error := LJM_eWriteNameByteArray(Handle, PAnsiChar(Name), NumBytes, @aBytes,
    ErrorAddress);
  ErrorHandler('LJM_eWriteNameByteArray', Error, ErrorAddress);

  Error := LJM_eWriteName(handle, 'I2C_GO', 1); { Do the I2C communications. }
  ErrorHandler('LJM_eWriteName', Error);

  { Read the RX bytes. }
  Name := 'I2C_DATA_RX';
  NumBytes := 4; { The number of bytes. }
  { aBytes[0] to aBytes[3] will contain the data. }
  for i := 0 to NumBytes-1 do
  begin
    aBytes[i] := 0;
  end;
  Error := LJM_eReadNameByteArray(Handle, PAnsiChar(Name), NumBytes, @aBytes,
    ErrorAddress);
  ErrorHandler('LJM_eReadNameByteArray', Error, ErrorAddress);

  Write('Read User Memory [0-3] = ');
  for i := 0 to NumBytes-1 do
  begin
    Write(FormatFloat('0.#', aBytes[i]) + ' ');
  end;
  Writeln('');

  { Close the handle. }
  Error := LJM_Close(Handle);
  ErrorHandler('LJM_Close', Error);

  WaitForUser('Press Enter to exit.');
end.
