{******************************************************************************}
{                                                                              }
{  SPI.dpr                                                                     }
{                                                                              }
{  Demonstrates SPI communication.                                             }
{                                                                              }
{  You can short MOSI to MISO for testing.                                     }
{                                                                              }
{  T7:                                                                         }
{    MOSI    FIO2                                                              }
{    MISO    FIO3                                                              }
{    CLK     FIO0                                                              }
{    CS      FIO1                                                              }
{                                                                              }
{  T4:                                                                         }
{    MOSI    FIO6                                                              }
{    MISO    FIO7                                                              }
{    CLK     FIO4                                                              }
{    CS      FIO5                                                              }
{                                                                              }
{  If you short MISO to MOSI, then you will read back the same bytes that you  }
{  write.  If you short MISO to GND, then you will read back zeros.  If you    }
{  short MISO to VS or leave it unconnected, you will read back 255s.          }
{                                                                              }
{******************************************************************************}

program SPI;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  LJMDelphi in '..\..\..\Wrapper\LJMDelphi.pas',
  LJMUtilities in '..\..\LJMUtilities.pas';

var
  i: Integer;
  Handle: Integer;
  Error: Integer;
  ErrorAddress: Integer;
  NumFrames: Integer;
  aNames: array [0..6] of AnsiString;
  aValues: array [0..6] of Double;
  NumBytes: Integer;
  aBytes: array [0..3] of byte;

  { Time variables for the random generator seed. }
  h: Word;
  m: Word;
  s: Word;
  ms: Word;
begin
  Handle := 0;
  ErrorAddress := -1;

  { Open first found LabJack }

  { Any device, Any connection, Any identifier }
  Error := LJM_OpenS('Any', 'Any', 'Any', Handle);

  { T7 device, Any connection, Any identifier }
  { Error := LJM_OpenS('T7', 'Any', 'Any', Handle); }

  { T4 device, Any connection, Any identifier }
  { Error := LJM_OpenS('T4', 'Any', 'Any', Handle); }

  { Any device, Any connection, Any identifier }
  { Error := LJM_Open(LJM_dtAny, LJM_ctAny, 'Any', Handle); }

  ErrorHandler('LJM_OpenS', Error);

  { Display opened device's information. }
  PrintDeviceInfoFromHandle(Handle);

  if GetDeviceType(Handle) = LJM_dtT4 then
  begin
    { Configure FIO4 to FIO7 as digital I/O. }
    Error := LJM_eWriteName(handle, 'DIO_INHIBIT', $FFF0F);
    ErrorHandler('LJM_WriteName (DIO_INHIBIT)', Error);
    Error := LJM_eWriteName(handle, 'DIO_ANALOG_ENABLE', $00000);
    ErrorHandler('LJM_WriteName (DIO_ANALOG_ENABLE)', Error);

    { Setting CS, CLK, MISO, and MOSI lines for the T4. FIO0 to FIO3 are
      reserved for analog inputs, and SPI requires digital lines. }
    Error := LJM_eWriteName(Handle, 'SPI_CS_DIONUM', 5); { CS is FIO5 }
    ErrorHandler('LJM_WriteName (SPI_CS_DIONUM)', Error);
    Error := LJM_eWriteName(Handle, 'SPI_CLK_DIONUM', 4); { CLK is FIO4 }
    ErrorHandler('LJM_WriteName (SPI_CLK_DIONUM)', Error);
    Error := LJM_eWriteName(Handle, 'SPI_MISO_DIONUM', 7); { MISO is FIO7 }
    ErrorHandler('LJM_WriteName (PI_MISO_DIONUM)', Error);
    Error := LJM_eWriteName(Handle, 'SPI_MOSI_DIONUM', 6); { MOSI is FIO6 }
    ErrorHandler('LJM_WriteName (SPI_MOSI_DIONUM)', Error);
  end
  else
  begin
    { Setting CS, CLK, MISO, and MOSI lines for the T7 and other devices. }
    Error := LJM_eWriteName(Handle, 'SPI_CS_DIONUM', 1); { CS is FIO1 }
    ErrorHandler('LJM_WriteName (SPI_CS_DIONUM)', Error);
    Error := LJM_eWriteName(Handle, 'SPI_CLK_DIONUM', 0); { CLK is FIO0 }
    ErrorHandler('LJM_WriteName (SPI_CLK_DIONUM)', Error);
    Error := LJM_eWriteName(Handle, 'SPI_MISO_DIONUM', 3); { MISO is FIO3 }
    ErrorHandler('LJM_WriteName (PI_MISO_DIONUM)', Error);
    Error := LJM_eWriteName(Handle, 'SPI_MOSI_DIONUM', 2); { MOSI is FIO2 }
    ErrorHandler('LJM_WriteName (SPI_MOSI_DIONUM)', Error);
  end;

  { Selecting Mode CPHA=1 (bit 0), CPOL=1 (bit 1) }
  Error := LJM_eWriteName(Handle, 'SPI_MODE', 3);
  ErrorHandler('LJM_WriteName (SPI_MODE)', Error);

  { Speed Throttle:
    Valid speed throttle values are 1 to 65536 where 0 = 65536.
    Configuring Max. Speed (~800 kHz) = 0 }
  Error := LJM_eWriteName(Handle, 'SPI_SPEED_THROTTLE', 0);
  ErrorHandler('LJM_WriteName (SPI_SPEED_THROTTLE)', Error);

  { Options
    bit 0:
        0 = Active low clock select enabled.
        1 = Active low clock select disabled.
    bit 1:
        0 = DIO directions are automatically changed
        1 = DIO directions are not automatically changed.
    bits 2-3: Reserved
    bits 4-7: Number of bits in the last byte. 0 = 8.
    bits 8-15: Reserved }

  { Enabling active low clock select pin }
  Error := LJM_eWriteName(Handle, 'SPI_OPTIONS', 0);
  ErrorHandler('LJM_WriteName (SPI_OPTIONS)', Error);

  { Read back and display the SPI settings }
  NumFrames := 7;
  aNames[0] := 'SPI_CS_DIONUM';
  aNames[1] := 'SPI_CLK_DIONUM';
  aNames[2] := 'SPI_MISO_DIONUM';
  aNames[3] := 'SPI_MOSI_DIONUM';
  aNames[4] := 'SPI_MODE';
  aNames[5] := 'SPI_SPEED_THROTTLE';
  aNames[6] := 'SPI_OPTIONS';
  Error := LJM_eReadNames(handle, NumFrames, @aNames, @aValues, ErrorAddress);
  ErrorHandler('LJM_ReadNames', Error, ErrorAddress);

  Writeln('SPI Configuration:');
  for i := 0 to NumFrames-1 do
  begin
      Writeln('  ' + aNames[i] + ' = ' + FloatToStr(aValues[i]));
  end;

  { Write(TX)/Read(RX) 4 bytes }
  NumBytes := 4;
  Error := LJM_eWriteName(Handle, 'SPI_NUM_BYTES', NumBytes);
  ErrorHandler('LJM_WriteName (SPI_NUM_BYTES)', Error);

  { Setup and write the bytes }
  DecodeTime(Now, h, m, s, ms);
  RandSeed := ms;
  for i := 0 to NumBytes-1 do
  begin
      aBytes[i] := Random(256);
  end;
  Error := LJM_eWriteNameByteArray(Handle, PAnsiChar('SPI_DATA_TX'), NumBytes,
    @aBytes, ErrorAddress);
  ErrorHandler('LJM_eWriteNameByteArray (SPI_DATA_TX)', Error, ErrorAddress);
  Error := LJM_eWriteName(Handle, 'SPI_GO', 1); { Do the SPI communications }
  ErrorHandler('LJM_eWriteName (SPI_GO)', Error);

  { Display the bytes written }
  Writeln('');
  for i := 0 to NumBytes-1 do
  begin
    Writeln('TXByte[' + IntToStr(i) + '] = ' +
      FormatFloat('0.#', aBytes[i]));
  end;

  { Read the bytes }
  for i := 0 to NumBytes-1 do
  begin
      aBytes[i] := 0;
  end;
  Error := LJM_eReadNameByteArray(Handle, PAnsiChar('SPI_DATA_RX'), NumBytes,
    @aBytes, ErrorAddress);
  ErrorHandler('LJM_eReadNameByteArray (SPI_DATA_RX)', Error, ErrorAddress);
  Error := LJM_eWriteName(Handle, 'SPI_GO', 1); { Do the SPI communications }
  ErrorHandler('LJM_eWriteName (SPI_GO)', Error);

  { Display the bytes read }
  Writeln('');
  for i := 0 to NumBytes-1 do
  begin
    Writeln('RXByte[' + IntToStr(i) + '] = ' + FormatFloat('0.#', aBytes[i]));
  end;

  { Close the handle. }
  Error := LJM_Close(Handle);
  ErrorHandler('LJM_Close', Error);

  WaitForUser('Press Enter to exit.');
end.
