{******************************************************************************}
{                                                                              }
{  StreamSequentialAIN.dpr                                                     }
{                                                                              }
{  Demonstrates how to stream a range of sequential analog inputs using the    }
{  eStream functions. Useful when streaming many analog inputs. AIN channel    }
{  scan list is FIRST_AIN_CHANNEL to FIRST_AIN_CHANNEL + NUMBER_OF_AINS - 1.   }
{                                                                              }
{******************************************************************************}

program StreamSequentialAIN;

{$APPTYPE CONSOLE}

uses
  DateUtils,
  Math,
  SysUtils,
  LJMDelphi in '..\..\..\Wrapper\LJMDelphi.pas',
  LJMUtilities in '..\..\LJMUtilities.pas';

const
  FIRST_AIN_CHANNEL = 0;  { 0 = AIN0 }
  NUMBER_OF_AINS = 8;
var
  i: Integer;
  j: Integer;
  Handle: Integer;
  Error: Integer;
  ErrorAddress: Integer;

  DIOInhibit: Cardinal;
  DIOAnalogEnable: Cardinal;
  RangeAINHV: Double;
  RangeAINLV: Double;

  { Stream variables }
  MaxRequests: Integer;
  aScanListNames: array of AnsiString;
  aScanList: array of Integer;
  aDataTypes: array of Integer;
  ScansPerRead: Integer;
  ScanRate: Double;
  aData: array of Double;
  DeviceScanBacklog: Integer;
  LJMScanBacklog: Integer;

  { LJM_eWriteNames parameters }
  NumFrames: Integer;
  aNames: array of AnsiString;
  aValues: array of Double;

  { Time variables }
  StTime: TDateTime;
  EndTime: TDateTime;
  TimeTaken: Double; { In seconds }

  { Stream loop variables }
  TotScans: Int64;
  CurSkipSamp: Int64;
  TotSkipSamp: Int64;
  AINStr: String;

begin
  Handle := 0;
  ErrorAddress := -1;

  { Open first found LabJack }

  { Any device, Any connection, Any identifier }
  Error := LJM_OpenS('Any', 'Any', 'Any', Handle);

  { T7 device, Any connection, Any identifier }
  { Error := LJM_OpenS('T7', 'Any', 'Any', Handle); }

  { T4 device, Any connection, Any identifier }
  { Error := LJM_OpenS('T4', 'Any', 'Any', Handle); }

  { Any device, Any connection, Any identifier }
  { Error := LJM_Open(LJM_dtAny, LJM_ctAny, 'Any', Handle); }

  ErrorHandler('LJM_OpenS', Error);

  { Display opened device's information }
  PrintDeviceInfoFromHandle(Handle);

  { When streaming, negative channels and ranges can be configured for
    individual analog inputs, but the stream has only one settling time and
    resolution. }
  if GetDeviceType(Handle) = LJM_dtT4 then
  begin
    { T4 configuration }

    { Configure the channels to analog input or digital I/O }

    SetLength(aNames, Max(2, NUMBER_OF_AINS));
    SetLength(aValues, Max(2, NUMBER_OF_AINS));

    { Updating  all digital I/O channels. b1 = Ignored. b0 = Affected. }
    DIOInhibit := $00000; { b00000000000000000000 }
    { Set AIN0-AIN3 and AIN FIRST_AIN_CHANNEL to
      FIRST_AIN_CHANNEL+NUMBER_OF_AINS-1 as analog inputs (b1), the rest as
      digital I/O (b0). }
    DIOAnalogEnable := (Trunc(Power(2, NUMBER_OF_AINS)-1) Shl FIRST_AIN_CHANNEL)
      or $0000F;
    NumFrames := 2;
    aNames[0] := 'DIO_INHIBIT';
    aNames[1] := 'DIO_ANALOG_ENABLE';
    aValues[0] := DIOInhibit;
    aValues[1] := DIOAnalogEnable;
    LJM_eWriteNames(Handle, NumFrames, Pointer(aNames), Pointer(aValues),
      ErrorAddress);
    ErrorHandler('LJM_eWriteNames', Error, ErrorAddress);

    { Configure the analog input ranges. }
    RangeAINHV := 10.0;  { HV channels range (AIN0-AIN3) }
    RangeAINLV := 2.5;  { LV channels range (AIN4+) }
    NumFrames := NUMBER_OF_AINS;
    for i := 0 to NumFrames-1 do
    begin
      aNames[i] := 'AIN' + IntToStr(FIRST_AIN_CHANNEL+i) + '_RANGE';
      if (FIRST_AIN_CHANNEL+i) < 4 then aValues[i] := RangeAINHV else
        aValues[i] := RangeAINLV;
    end;
    LJM_eWriteNames(Handle, NumFrames, Pointer(aNames), Pointer(aValues),
      ErrorAddress);
    ErrorHandler('LJM_eWriteNames', Error, ErrorAddress);

    { Configure the stream settling times and stream resolution index. }
    NumFrames := 2;
    aNames[0] := 'STREAM_SETTLING_US';
    aNames[1] := 'STREAM_RESOLUTION_INDEX';
    aValues[0] := 0; { 0 (default) }
    aValues[1] := 0; { 0 (default) }
    LJM_eWriteNames(Handle, NumFrames, Pointer(aNames), Pointer(aValues),
      ErrorAddress);
    ErrorHandler('LJM_eWriteNames', Error, ErrorAddress);
  end
  else
  begin
    { T7 and other devices configuration }

    { Configure the analog input negative channels, ranges, stream settling
      times and stream resolution index. }
    NumFrames := 4;
    SetLength(aNames, NumFrames);
    SetLength(aValues, NumFrames);
    aNames[0] := 'AIN_ALL_NEGATIVE_CH';
    aNames[1] := 'AIN_ALL_RANGE';
    aNames[2] := 'STREAM_SETTLING_US';
    aNames[3] := 'STREAM_RESOLUTION_INDEX';
    aValues[0] := LJM_GND; { Single-ended }
    aValues[1] := 10.0; { +/-10 V }
    aValues[2] := 20; { 0 (default) }
    aValues[3] := 0; { 0 (default) }
    Error := LJM_eWriteNames(Handle, NumFrames, Pointer(aNames),
      Pointer(aValues), ErrorAddress);
    ErrorHandler('LJM_eWriteNames', Error, ErrorAddress);
  end;

  { Stream configuration }

  { Scan list names to stream. }
  SetLength(aScanListNames, NUMBER_OF_AINS);
  SetLength(aScanList, NUMBER_OF_AINS);
  SetLength(aDataTypes, NUMBER_OF_AINS);
  for i := 0 to NUMBER_OF_AINS-1 do
  begin
    aScanListNames[i] := 'AIN' + IntToStr(FIRST_AIN_CHANNEL+i);
  end;
  { Scan list addresses to stream. LJM_eStreamStart uses Modbus addresses. }
  Error := LJM_NamesToAddresses(NUMBER_OF_AINS, Pointer(aScanListNames),
    Pointer(aScanList), Pointer(aDataTypes));
  ErrorHandler('LJM_NamesToAddresses', Error);

  { Configure and start stream }

  { Ensure triggered stream is disabled. }
  LJM_eWriteName(Handle, 'STREAM_TRIGGER_INDEX', 0);
  ErrorHandler('LJM_eWriteName (STREAM_TRIGGER_INDEX)', Error);

  { Enabling internally-clocked stream. }
  LJM_eWriteName(Handle, 'STREAM_CLOCK_SOURCE', 0);
  ErrorHandler('LJM_eWriteName (STREAM_CLOCK_SOURCE)', Error);

  ScanRate := 1000; { Scan rate = 1000 Hz }
  ScansPerRead := Trunc(ScanRate/2);
  SetLength(aData, ScansPerRead*NUMBER_OF_AINS);
  Error := LJM_eStreamStart(Handle, ScansPerRead, NUMBER_OF_AINS,
    Pointer(aScanList), ScanRate);
  if IsError('LJM_eStreamStart', Error) = False then
  begin
    Writeln('Stream started with a scan rate of ' +
      FormatFloat('0.00', ScanRate) + '.');
    Writeln('');

    MaxRequests := 50;
    Writeln('Performing ' + IntToStr(MaxRequests) + ' stream reads.');
    Writeln('');

    TotScans := 0;
    TotSkipSamp := 0;

    StTime := Now;

    for i := 0 to MaxRequests-1 do
    begin
      Error := LJM_eStreamRead(Handle, Pointer(aData), DeviceScanBacklog,
        LJMScanBacklog);
      if IsError('LJM_eStreamRead', Error) then
        Break;

      TotScans := TotScans + ScansPerRead;

      { Count the skipped samples which are indicated by -9999 values. Missed
        samples occur after a device's stream buffer overflows and are reported
        after auto-recover mode ends. }
      CurSkipSamp := 0;
      for j := 0 to Length(aData)-1 do
      begin
        if aData[j] = -9999.0 then
          CurSkipSamp := CurSkipSamp + 1;
      end;
      TotSkipSamp := TotSkipSamp + CurSkipSamp;

      Writeln('LJM_eStreamRead ' + IntToStr(i+1));
      AINStr := '';
      for j := 0 to NUMBER_OF_AINS-1 do
      begin
        AINStr := AINStr + aScanListNames[j] + ' = ' +
          FormatFloat('0.00000', aData[j]) + ', ';
      end;

      Writeln('  First scan out of ' + IntToStr(ScansPerRead) + ' : ' + AINStr);
      Write('  Scans Skipped = ' +
        FormatFloat('0.0', CurSkipSamp/NUMBER_OF_AINS));
      Write(', Scan Backlogs: Device = ' + IntToStr(DeviceScanBacklog));
      Writeln(', LJM = ' + IntToStr(LJMScanBacklog));
      Writeln('');
    end;

    EndTime := Now;
    TimeTaken := SecondSpan(StTime, EndTime);

    Writeln('');
    Writeln('Total Scans = ' + IntToStr(TotScans));
    Writeln('Skipped Scans = ' +
      FormatFloat('0.0', TotSkipSamp/NUMBER_OF_AINS));
    Writeln('Time Taken = ' + FloatToStr(TimeTaken) + ' seconds');
    Writeln('LJM Scan Rate = ' + FormatFloat('0.000', ScanRate) +
      ' scans/second');
    Writeln('Timed Scan Rate = ' + FormatFloat('0.000', TotScans/TimeTaken) +
      ' scans/second');
    Writeln('Timed Sample Rate = ' +
      FormatFloat('0.000', TotScans*NUMBER_OF_AINS/TimeTaken) +
      ' samples/second');
  end;

  Error := LJM_eStreamStop(Handle);
  ErrorHandler('LJM_eStreamStop', Error);
  Writeln('');
  Writeln('Stream Stop');

  { Close the handle. }
  Error := LJM_Close(Handle);
  ErrorHandler('LJM_Close', Error);

  WaitForUser('Press Enter to exit.');
end.


