{******************************************************************************}
{                                                                              }
{  CRSpeedTest.dpr                                                             }
{                                                                              }
{  Performs LabJack operations in a loop and reports the timing statistics for }
{  the operations.                                                             }
{                                                                              }
{******************************************************************************}

program CRSpeedTest;

{$APPTYPE CONSOLE}

uses
  Math,
  SysUtils,
  Windows,
  LJMDelphi in '..\..\..\Wrapper\LJMDelphi.pas',
  LJMUtilities in '..\..\LJMUtilities.pas';

const
  { Number of iterations to perform in the operations loop }
  NumIterations = 1000;

  { Analog input settings }
  NumAIN = 1; { Number of analog inputs to read }
  RangeAIN = 10.0;
  RangeAINHV = 10.0;
  RangeAINLV  = 2.5;
  ResolutionAIN = 1.0;

  { Digital settings }
  ReadDigital = False;
  WriteDigital = False;

  { Analog output settings }
  WriteDACs = False;
var
  i: Integer;
  Handle: Integer;
  DeviceType: Integer;
  Error: Integer;
  ErrorAddress: Integer;
  CurFrame: Integer;

  DIOInhibit: Double;
  DIOAnalogEnable: Double;


  { LJM_eWriteNames and LJM_eName parameters }
  NumFrames: Integer;
  aNames: array of AnsiString;
  aWrites: array of Integer;
  aNumValues: array of Integer;
  aValues: array of Double;

  { Time variables }
  Freq: Int64;
  StTime: Int64;
  EndTime: Int64;
  t0: Int64;
  t1: Int64;
  TTMS: Double;
  MaxMS: Double;
  MinMS: Double;
  TotalMS: Double;
begin
  Handle := 0;
  ErrorAddress := -1;

  { Open first found LabJack }

  { Any device, Any connection, Any identifier }
  Error := LJM_OpenS('Any', 'Any', 'Any', Handle);

  { T7 device, Any connection, Any identifier }
  { Error := LJM_OpenS('T7', 'Any', 'Any', Handle); }

  { T4 device, Any connection, Any identifier }
  { Error := LJM_OpenS('T4', 'Any', 'Any', Handle); }

  { Any device, Any connection, Any identifier }
  { Error := LJM_Open(LJM_dtAny, LJM_ctAny, 'Any', Handle); }

  ErrorHandler('LJM_OpenS', Error);

  { Display opened device's information. }
  PrintDeviceInfoFromHandle(Handle);

  DeviceType := GetDeviceType(Handle);

  if DeviceType = LJM_dtT4 then
  begin
    { For the T4, configure the channels to analog input or digital I/O. }

    { Update all digital I/O channels. b1 = Ignored. b0 = Affected. }
    DIOInhibit := $00000;  { b00000000000000000000 }
    { Set AIN 0 to numAIN-1 as analog inputs (b1), the rest as
      digital I/O (b0). }
    DIOAnalogEnable := Power(2, NumAIN) - 1;
    SetLength(aNames, 2);
    SetLength(aValues, 2);
    aNames[0] := 'DIO_INHIBIT';
    aNames[1] := 'DIO_ANALOG_ENABLE';
    aValues[0] := DIOInhibit;
    aValues[1] := DIOAnalogEnable;
    LJM_eWriteNames(Handle, 2, Pointer(aNames), Pointer(aValues), ErrorAddress);
    ErrorHandler('LJM_eWriteNames (DIO enable config.)', Error, ErrorAddress);

    if WriteDigital then  { double check this }
    begin
      { Update only digital I/O channels in future digital write calls.
        b1 = Ignored. b0 = Affected. }
      DIOInhibit := DIOAnalogEnable;
      LJM_eWriteName(Handle, 'DIO_INHIBIT', DIOInhibit);
    ErrorHandler('LJM_eWriteNames (DIO inhibit config.)', Error, ErrorAddress);
    end;
  end;

  if NumAIN > 0 then
  begin
    { Configure analog inputs }
    NumFrames := NumAIN*2;
    SetLength(aNames, NumAIN*2);
    SetLength(aValues, NumAIN*2);
    for i := 0 to NumAIN-1 do
    begin
      aNames[i*2] := 'AIN' + IntToStr(i) + '_RANGE';
      if DeviceType = LJM_dtT4 Then
      begin
        { T4 range }
        if i < 4 then
          aValues[i*2] := RangeAINHV { HV line }
        else
          aValues[i*2] := RangeAINLV { LV line }
      end
      else
      begin
        { T7 and other devices range }
        aValues[i*2] := RangeAIN;
      end;
      aNames[i*2+1] := 'AIN' + IntToStr(i) + '_RESOLUTION_INDEX';
      aValues[i*2+1] := ResolutionAIN;
    end;
    Error := LJM_eWriteNames(Handle, NumFrames, Pointer(aNames),
      Pointer(aValues), ErrorAddress);
    ErrorHandler('LJM_eWriteNames (AIN config.)', Error, ErrorAddress);
  end;

  { Initialize and configure parameters for loop's eNames call }
  NumFrames := Trunc(NumAIN) + Ord(ReadDigital) + Ord(WriteDigital) +
    Ord(WriteDACs)*2;
  SetLength(aNames, NumFrames);
  SetLength(aWrites, NumFrames);
  SetLength(aNumValues, NumFrames);
  SetLength(aValues, NumFrames);
  CurFrame := 0;

  { Add analog input reads (AIN 0 to numAIN-1) }
  for i := 0 to NumAIN-1 do
  begin
    aNames[CurFrame] := 'AIN' + IntToStr(i);
    aWrites[CurFrame] := LJM_READ;
    aNumValues[CurFrame] := 1;
    aValues[CurFrame] := 0;
    CurFrame := CurFrame + 1;
  end;

  if ReadDigital then
  begin
    { Add digital I/O read }
    aNames[CurFrame] := 'DIO_STATE';
    aWrites[CurFrame] := LJM_READ;
    aNumValues[CurFrame] := 1;
    aValues[CurFrame] := 0;
    CurFrame := CurFrame + 1;
  end;

  if writeDigital then
  begin
    { Add digital I/O write }
    aNames[CurFrame] := 'DIO_STATE';
    aWrites[CurFrame] := LJM_WRITE;
    aNumValues[CurFrame] := 1;
    aValues[CurFrame] := 0; { All output-low }
    curFrame := CurFrame + 1;
  end;

  if WriteDACs then
  begin
    { Add analog output writes (DAC0-1) }
    for i := 0 to 1 do
    begin
      aNames[CurFrame] := 'DAC' + IntToStr(i);
      aWrites[CurFrame] := LJM_WRITE;
      aNumValues[CurFrame] := 1;
      aValues[CurFrame] := 0.0; { 0.0 V }
      CurFrame := CurFrame + 1;
    end;
  end;

  Writeln('Test Frames:');
  for i := 0 to NumFrames-1 do
  begin
    if aWrites[i] = LJM_WRITE
    then Writeln('  WRITE ' + aNames[i])
    else Writeln('  READ ' + aNames[i]);
  end;

  Writeln('');

  Writeln('Performing ' + IntToStr(NumIterations) + ' operations...');

  MaxMS := 0.0;
  MinMS := 0.0;

  { Performing LJM_eNames operations in a loop. }
  QueryPerformanceFrequency(Freq);
  QueryPerformanceCounter(StTime);
  for i := 0 to NumIterations-1 do
  begin
    QueryPerformanceCounter(t0);
    Error := LJM_eNames(Handle, NumFrames, Pointer(aNames), Pointer(aWrites),
      Pointer(aNumValues), Pointer(aValues), ErrorAddress);
    QueryPerformanceCounter(t1);
    ErrorHandler('LJM_eNames (iteration ' + IntToStr(i+1) + ')', Error,
      ErrorAddress);
    TTMS := (t1-t0)/Freq * 1000.0;
    if MinMS = 0.0 then
      MinMS := TTMS;
    MinMS := Min(TTMS, MinMS);
    MaxMS := Max(TTMS, MaxMS);
  end;
  QueryPerformanceCounter(EndTime);

  Writeln('');

  TotalMS := ((EndTime-StTime)/Freq) * 1000.0;
  Writeln(IntToStr(NumIterations) + ' iterations performed:');
  Writeln('  Time taken: ' + FormatFloat('0.000', TotalMS) + ' ms');
  Writeln('  Average time per iteration: ' +
    FormatFloat('0.000', (TotalMS/NumIterations)) + ' ms');
  Writeln('  Min / Max time for one iteration: ' + FormatFloat('0.000', MinMS) +
    ' ms / ' + FormatFloat('0.000', MaxMS) + ' ms');

  Writeln('');
  Writeln('Last Write/Reads:');
  for i := 0 to NumFrames-1 do
  begin
    if aWrites[i] = LJM_WRITE
    then Writeln('  WRITE ' + aNames[i] + ' = ' + FloatToStr(aValues[i]))
    else Writeln('  READ ' + aNames[i] + ' = ' + FloatToStr(aValues[i]));
  end;

  { Close the handle. }
  Error := LJM_Close(Handle);
  ErrorHandler('LJM_Close', Error);

  WaitForUser('Press Enter to exit.');
end.
