{******************************************************************************}
{  ReadWiFiConfig.dpr                                                          }
{                                                                              }
{  Demonstrates how to read the WiFi configuration from a LabJack.             }
{                                                                              }
{******************************************************************************}

program ReadWiFiConfig;

{$APPTYPE CONSOLE}

uses
  SysUtils,
  StrUtils,
  LJMDelphi in '..\..\..\Wrapper\LJMDelphi.pas',
  LJMUtilities in '..\..\LJMUtilities.pas';

var
  i: Integer;
  Handle: Integer;
  Error: Integer;
  ErrorAddress: Integer;
  NumFrames: Integer;
  aNames: array [0..8] of AnsiString;
  aValues: array [0..8] of Double;
  IPStr: array [0..LJM_IPv4_STRING_SIZE-1] of AnsiChar;
  SSIDStr: array [0..LJM_STRING_ALLOCATION_SIZE-1] of AnsiChar;
begin
  Handle := 0;
  ErrorAddress := -1;

  { Open first found LabJack }

  { Any device, Any connection, Any identifier }
  Error := LJM_OpenS('Any', 'Any', 'Any', Handle);

  { T7 device, Any connection, Any identifier }
  { Error := LJM_OpenS('T7', 'Any', 'Any', Handle); }

  { Any device, Any connection, Any identifier }
  { Error := LJM_Open(LJM_dtAny, LJM_ctAny, 'Any', Handle); }

  ErrorHandler('LJM_OpenS', Error);

  { Display opened device's information. }
  PrintDeviceInfoFromHandle(Handle);

  if GetDeviceType(Handle) = LJM_dtT4 then
  begin
    Writeln('The LabJack T4 does not support WiFi.');
    LJM_Close(Handle); { Close the handle. }
    WaitForUser('Press Enter to exit.');
    Halt;
  end;

  { Setup and call LJM_eReadNames to read WiFi configuration from the LabJack. }
  NumFrames := 9;
  aNames[0] := 'WIFI_IP';
  aNames[1] := 'WIFI_SUBNET';
  aNames[2] := 'WIFI_GATEWAY';
  aNames[3] := 'WIFI_DHCP_ENABLE';
  aNames[4] := 'WIFI_IP_DEFAULT';
  aNames[5] := 'WIFI_SUBNET_DEFAULT';
  aNames[6] := 'WIFI_GATEWAY_DEFAULT';
  aNames[7] := 'WIFI_DHCP_ENABLE_DEFAULT';
  aNames[8] := 'WIFI_STATUS';
  Error := LJM_eReadNames(Handle, NumFrames, @aNames, @aValues, ErrorAddress);
  ErrorHandler('LJM_eReadNames', Error, ErrorAddress);

  Writeln('Read WiFi Configuration:');
  for i := 0 to NumFrames-1 do
  begin
    if (AnsiStartsStr('WIFI_STATUS', aNames[i])) or
      (AnsiStartsStr('WIFI_DHCP_ENABLE', aNames[i])) then
    begin
      Writeln('  ' + aNames[i] + ' = ' + FloatToStr(aValues[i]));
    end
    else
    begin
      Error := LJM_NumberToIP(Cardinal(Trunc(aValues[i])), IPStr);
      ErrorHandler('LJM_NumberToIP (' + aNames[i] + ')', Error);
      Writeln('  ' + aNames[i] + ' = ' + IPStr + '  (' +
        FloatToStr(aValues[i]) + ')');
    end;
  end;

  { Setup and call eReadNameString to read the WiFi SSID string from the
    LabJack. }
  Error := LJM_eReadNameString(Handle, 'WIFI_SSID', SSIDStr);
  ErrorHandler('LJM_eReadNameString', Error);

  Writeln('  WIFI_SSID: ' + SSIDStr);

  { Close the handle. }
  Error := LJM_Close(Handle);
  ErrorHandler('LJM_Close', Error);

  WaitForUser('Press Enter to exit.');
end.
