/********************************************************************************************************************************/
// AOUpdate
//
// Sets the voltages of the analog outputs.  Also controls/reads all 20 digital I/O and the counter. Execution time for this 
// function is 50 ms or less.
//
// MATLAB syntax:
//		[stateD stateIO count errorcode idnum]  = aoupdate(idnum demo, trisD, trisIO, stateD, stateIO,updateDigital resetCounter, 
//															analogOut0, analogOut1)
//
// Inputs:
//		idnum - Local ID, serial number, or -1 for first LabJack found.
//		demo - Send 0 for normal operation, >0 for demo mode. Demo mode allows this function to be called without a LabJack.
//		trisD - Directions for D0-D15.  0 = Input, 1 = Output.
//		trisIO - Directions for IO0-IO3.  0 = Input, 1 = Output.
//		stateD - Output states for D0-D15.
//		stateIO - Output states for IO0-IO3.
//		updateDigital - If >0, tris and state values will be written.  Otherwise, just a read is performed.
//		resetCounter - If > 0, the counter is reset to zero after being read.
//		analogOut0 - Voltage from 0.0 to 5.0 for AO0.
//		analogOut1 - Voltae from 0.0 to 5.0 for AO1.
//
// Outputs:
//		stateD - States for D0 - D15.
//		StateIO - States for IO0 - IO3.
//		count - Current value of the 32-bit counter (CNT).  This value is read before the counter is reset.
//		errorcode = LabJack errorcodes or 0 for no error.
//		Idnum = Local ID number of Labjack, or -1 if no LabJack is found.
// 
/********************************************************************************************************************************/

#include "mex.h"
#include <wtypes.h>
#include "ljackuw.h"

void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[] )
{
  double  inp1,inp2,inp3,inp4,inp5,inp6,inp7,inp8, inp9, inp10,*out1,*out2,*out3,*out4,*out5;
  int i, m, n;
  long idnum, demo, trisD, trisIO, updateDigital, resetCounter, stateD, stateIO, errorcode, overvoltage;
  unsigned long count;
  float analogOut0, analogOut1;

  /* Check for proper number of arguments. */
  if(nrhs != 10) 
  {
    mexErrMsgTxt("Input Error --> (idnum, demo, trisD, trisIO, stateD, stateIO, updateDigital, resetCounter, analogOut0, analogOut1)");
  } 
  else if(nlhs > 5) 
  {
    mexErrMsgTxt("Output Error --> [stateD stateIO count errorcode idnum]");
  }

  /* Get dimensions of inputs */
  for (i=0;i<=9;i++)
  {
	 m = mxGetM(prhs[i]);
	 n = mxGetN(prhs[i]);
	 
	 if(m*n != 1)
	 {
		 mexErrMsgTxt("Inputs must to be scalar");
	 }
  }
  
  /* Retrive Matlab input values from function call. */
  inp1 = mxGetScalar(prhs[0]);
  inp2 = mxGetScalar(prhs[1]);
  inp3 = mxGetScalar(prhs[2]);
  inp4 = mxGetScalar(prhs[3]);
  inp5 = mxGetScalar(prhs[4]);
  inp6 = mxGetScalar(prhs[5]);
  inp7 = mxGetScalar(prhs[6]);
  inp8 = mxGetScalar(prhs[7]);
  inp9 = mxGetScalar(prhs[8]);
  inp10 = mxGetScalar(prhs[9]);

  /* Create matrix for the return arguments. */
  plhs[0] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[1] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[2] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[3] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[4] = mxCreateDoubleMatrix(1,1, mxREAL);

  /* Get pointer to real part of return argument */
  out1 = mxGetPr(plhs[0]);
  out2 = mxGetPr(plhs[1]);
  out3 = mxGetPr(plhs[2]);
  out4 = mxGetPr(plhs[3]);
  out5 = mxGetPr(plhs[4]);
  
  /* Recast Matlab inputs */
  idnum = (long)inp1;
  demo = (long)inp2;
  trisD = (long)inp3;
  trisIO = (long)inp4;
  stateD = (long)inp5;
  stateIO = (long)inp6;
  updateDigital = (long)inp7;
  resetCounter = (long)inp8;
  analogOut0 = (float)inp9;
  analogOut1 = (float)inp10;

  /* Call the AOUpdate function. */
  errorcode = AOUpdate(&idnum, demo, trisD, trisIO, &stateD, &stateIO, updateDigital, resetCounter, &count, analogOut0, analogOut1);
  
  /* Set Matlab outputs equal to returned LJ values */
  out1[0] = (double)stateD;
  out2[0] = (double)stateIO;
  out3[0] = (double)count;
  out4[0] = (double)errorcode;
  out5[0] = (double)idnum;
}
