/********************************************************************************************************************************/
// Counter
//
// Controls and reads the counter.  The counter is disabled if the watchdog timer is enabled.  Executing time for this function 
// is 50 ms or less.
//
// MATLAB Syntax:
//		[stated stateIO count errorcode idnum] = Counter(idnum, demo, resetCounter, enableSTB) 
//
// Inputs:
//		idnum - Local ID, serial number, or -1 for first LabJack found.
//		demo - Send 0 for normal operation, >0 for demo mode.  Demo mode allows this function to be called without a LabJack.
//		resetCounter - If>0, the counter is reset to zero after being read.
//		enableSTB - If>0, STB is enabled.  Used for testing and calibration.  (This input has no effect with firmware V1.02 or 
//				    earlier, in which case STB is always enabled).
//
// Outputs:
//		stateD - States of D0 - D15.
//		stateIO - States of IO0 - IO3.
//		count - Current value of the 32-bit counter (CNT).  This value is read before the counter is reset.
//		errorcode - LabJack error codes or 0 for no error.
//		idnum - Local ID number of Labjack, or -1 if no LabJack is found.
//
/********************************************************************************************************************************/

#include "mex.h"
#include <wtypes.h>
#include "ljackuw.h"

void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[] )
{
  double  inp1,inp2,inp3,inp4,*out1,*out2,*out3,*out4,*out5;
  int i, m, n;
  long idnum, demo, resetCounter, stateD, stateIO, errorcode, enableSTB;
  unsigned long count;

  /* Check for proper number of arguments. */
  if(nrhs != 4) 
  {
    mexErrMsgTxt("Input Error --> (Idnum, demo, resetCounter, enableSTB)");
  } 
  else if(nlhs > 5) 
  {
    mexErrMsgTxt("Output Error --> [StateD StateIO count Errorcode Idnum]");
  }

  /* Get dimensions of inputs */
  for (i=0;i<=3;i++)
  {
	 m = mxGetM(prhs[i]);
	 n = mxGetN(prhs[i]);
	 
	 if(m*n != 1)
	 {
		 mexErrMsgTxt("Inputs must to be scalar");
	 }
  }
  
  /* Retrive Matlab input values from function call. */
  inp1 = mxGetScalar(prhs[0]);
  inp2 = mxGetScalar(prhs[1]);
  inp3 = mxGetScalar(prhs[2]);
  inp4 = mxGetScalar(prhs[3]);

  /* Create matrix for the return arguments. */
  plhs[0] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[1] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[2] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[3] = mxCreateDoubleMatrix(1,1, mxREAL);

  /* Get pointer to real part of return argument */
  out1 = mxGetPr(plhs[0]);
  out2 = mxGetPr(plhs[1]);
  out3 = mxGetPr(plhs[2]);
  out4 = mxGetPr(plhs[3]);
  
  /* Recast Matlab inputs */
  idnum = (long)inp1;
  demo = (long)inp2;
  resetCounter = (long)inp3;
  enableSTB = (long)inp4;

  /* Call the Counter function. */
  errorcode = Counter(&idnum, demo, &stateD, &stateIO, resetCounter, enableSTB, &count);
  
  /* Set Matlab outputs equal to returned LJ values */
  out1[0] = (double)stateD;
  out2[0] = (double)stateIO;
  out3[0] = (double)count;
  out4[0] = (double)errorcode;
  out5[0] = (double)idnum;
}
