/********************************************************************************************************************************/
// DigitalIO
//
// Reads and writes to all 20 digital I/O.  Execution time for this funtion is 50 ms or less.
//
// MATLAB Syntax:
//		[stateD stateIO trisD outputD errorcode Idnum] = DigitalIO(idnum, demo, trisD, trisIO, stateD, stateIO, updatedigital)
//
// Inputs:
//		idnum - Local Id, serial number, or -1 for first LJ found
//		demo - 0 for normal operation, >0 for demo mode.  Demo mode allows this function to be called without a LabJack.
//		trisD - Directions for D0-D15. 0=Input, 1=Output.
//		trisIO - Directions for IO0-IO3.  0=Input, 1=Output.
//		stateD - Output states for D0-D15.
//		stateIO - Output states for IO0-IO3.
//		updateDigital - If>0, tris and state values will be written.  Otherwise, just a read is performed.	
//
// Outputs:
//		stateD - States of D0-D15.
//		stateIO - States of IO0-IO3.
//		trisD - Returns a read of the direction registers for D0-D15.
//		outputD - Returns a read from the output registers for DO-D15.
//		errorcode - LabJack error codes or 0 for no error.
//		idnum - Local ID number of Labjack, or -1 if no LabJack is found. 
// 
/********************************************************************************************************************************/

#include "mex.h"
#include <wtypes.h>
#include "ljackuw.h"

void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[] )
{
  double  inp1,inp2,inp3,inp4,inp5,inp6,inp7,*out1,*out2,*out3,*out4,*out5,*out6;
  int i, m, n;
  long idnum, demo, trisD, trisIO, stateD, stateIO, updateDigital, outputD, errorcode;

  /* Check for proper number of arguments. */
  if(nrhs != 7) 
  {
    mexErrMsgTxt("Input Error --> (idnum, demo, trisD, trisIO, stateD, stateIO, updatedigital)");
  } 
  else if(nlhs > 6) 
  {
    mexErrMsgTxt("Output Error --> [stateD stateIO trisD outputD errorcode idnum]");
  }

  /* Get dimensions of inputs */
  for (i=0;i<=6;i++)
  {
	 m = mxGetM(prhs[i]);
	 n = mxGetN(prhs[i]);
	 
	 if(m*n != 1)
	 {
		 mexErrMsgTxt("Inputs must to be scalar");
	 }
  }
  
  /* Retrive Matlab input values from function call. */
  inp1 = mxGetScalar(prhs[0]);
  inp2 = mxGetScalar(prhs[1]);
  inp3 = mxGetScalar(prhs[2]);
  inp4 = mxGetScalar(prhs[3]);
  inp5 = mxGetScalar(prhs[4]);
  inp6 = mxGetScalar(prhs[5]);
  inp7 = mxGetScalar(prhs[6]);

  /* Create matrix for the return arguments. */
  plhs[0] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[1] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[2] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[3] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[4] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[5] = mxCreateDoubleMatrix(1,1, mxREAL);

  /* Get pointer to real part of return argument */
  out1 = mxGetPr(plhs[0]);
  out2 = mxGetPr(plhs[1]);
  out3 = mxGetPr(plhs[2]);
  out4 = mxGetPr(plhs[3]);
  out5 = mxGetPr(plhs[4]);
  out6 = mxGetPr(plhs[5]);
  
  /* Recast Matlab inputs */
  idnum = (long)inp1;
  demo = (long)inp2;
  trisD = (long)inp3;
  trisIO = (long)inp4;
  stateD = (long)inp5;
  stateIO = (long)inp6;
  updateDigital = (long)inp7;

  /* Call the DigitalIO function. */
  errorcode = DigitalIO(&idnum, demo, &trisD, trisIO, &stateD, &stateIO, updateDigital, &outputD);
  
  /* Set Matlab outputs equal to returned LJ values */
  out1[0] = (double)stateD;
  out2[0] = (double)stateIO;
  out3[0] = (double)trisD;
  out4[0] = (double)outputD;
  out5[0] = (double)errorcode;
  out6[0] = (double)idnum;
}
