/********************************************************************************************************************************/
// LocalID
//
// Changes the local ID of a specified LabJack.  Changes will not take effect until the LabJack is re-enumerated or reset, either 
// manually by disconnecting and reconnecting the USB cable or by calling ReEnumn or ResetLJ.
//
// MATLAB Syntax:
//		[localID errorcode] = LocalID(idnum, localID)
//
// Inputs:
//		idnum - Local ID, serial number, or -1 for first found
//		localID - New local ID number	
//
// Outputs:
//		idnum - Returns the local ID or -1 if no LabJack is found.
//		errorcode - LabJack errorcodes or 0 for no error.
// 
/********************************************************************************************************************************/

#include "mex.h"
#include "matrix.h"
#include <wtypes.h>
#include "ljackuw.h"

void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[] )
{
  double  *out1, *out2;
  int i, m, n;
  double inp1, inp2;
  long idnum, localID, errorcode;

  /* Check for proper number of arguments. */
  if(nrhs != 2) 
  {
    mexErrMsgTxt("Input Error --> (idnum localID)");
  } 
  else if(nlhs > 2) 
  {
    mexErrMsgTxt("Output Error --> [idnum errorcode]");
  }

  /* Get dimensions of inputs */
  for (i=0;i<=1;i++)
  {
	 m = mxGetM(prhs[i]);
	 n = mxGetN(prhs[i]);
	 
	 if(m*n != 1)
	 {
		 mexErrMsgTxt("Input must to be scalar");
	 }
  }

  /* Retrive Matlab input values from function call. */
  inp1 = mxGetScalar(prhs[0]);
  inp2 = mxGetScalar(prhs[1]);

  /* Create matrix for the return arguments. */;
  plhs[0]= mxCreateDoubleMatrix(1,1,mxREAL);
  plhs[1]= mxCreateDoubleMatrix(1,1,mxREAL);

  /* Get pointer to real part of return argument */
  out1 = mxGetPr(plhs[0]);
  out2 = mxGetPr(plhs[1]);

  /* Recast Matlab inputs */
  idnum = long(inp1);
  localID = long(inp2);

  /* Call the LocalID function. */
  errorcode = LocalID(&idnum, localID);
  
  /* Set Matlab outputs equal to returned LJ values */
  out1[0] = (double)idnum;
  out2[0] = (double)errorcode;

}
