/******************************************************************************************************************************/
// Synch
//
// Requires firmware V1.1 or higher. This function performs SPI communication.  Data rate is about 160 kbps with no extra delay,
// although delays of 100 us or 1 ms per bit can be enabled. 
// 
// Execution time for this function is about 20 milliseconds to write and/or read up to 4 bytes, plus about 20 milliseconds for
// each additional 4 bytes written or read.  Extra 20 milliseconds if configD is true, and extra time if delays are enabled.
//
// MATLAB Syntax:
//		[data errorcode Idnum] = Synch(idnum, demo, mode, msDelay, husDelay, controlCS, csLine, csState, configD, numWriteRead, data)
//
// Input:
//		*idnum - Local Id, serial number, or -1 for first LJ found
//		demo - 0 for normal operation, >0 for demo mode.  Demo mode allows this function to be called    without a LabJack.
//		mode - Specify SPI mode as: 0=A, 1=B, 2=C, 3=D (0-3).
//		msDelay - If > 0, a 1 ms delay is added between each bit.
//		husDelay - If > 0, a hundred us delay is added between each bit.
//		controlCS - If >0, D0-D7 is automatically controlled as CS. The state and direction of CS is only tested if control is
//					enabled.
//		csLine - D line to use as CS if enabled (0-7).
//		csState - Active state for CS line.  This would be 0 for the normal !CS, or >0 for the less common CS.
//		configD - If >0, state and tris are configured for D13, D14, D15, and !CS.
//		numWriteRead - Number of bytes to write and read (1-18).
//		*data - Serial Data Buffer.  Send an 18 element array of bytes.  Fill unused locations with zeros.
//		
// Output:
//		*idnum - Returns the local ID or -1 for first found.
//		*data - Serial data buffer.  Returns any serial read data.  Unused locations are
//			filled with 9999s.
//
/********************************************************************************************************************************/

#include "mex.h"
#include <wtypes.h>
#include "ljackuw.h"

void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[] )
{
  double  inp1,inp2,inp3,inp4,inp5,inp6,inp7,inp8,inp9,inp10, *inp11;

  double* out1, *out2, *out3;
  int i, m, n, g;
  long idnum, demo, mode, msDelay, husDelay, controlCS, csLine, csState, configD, numWriteRead, errorcode;
  long  data[18] = {0};

  /* Check for proper number of arguments. */
  if(nrhs != 11) 
  {
    mexErrMsgTxt("Input Error --> (idnum, demo, mode, msDelay, husDelay, controlCS, csLine, csState, configD, numWriteRead, data)");
  } 				   
  else if(nlhs > 3) 
  {
    mexErrMsgTxt("Output Error --> [data errorcode Idnum]");
  }

  /* Get dimensions of inputs */
  for (i=0;i<=8;i++)
  {
	 m = mxGetM(prhs[i]);
	 n = mxGetN(prhs[i]);
	 
	 if(m*n != 1)
	 {
		 mexErrMsgTxt("Inputs must to be scalar:");
	 }
  }
	
  /* Retrive Matlab input values from function call. */
  inp1 = mxGetScalar(prhs[0]);
  inp2 = mxGetScalar(prhs[1]);
  inp3 = mxGetScalar(prhs[2]);
  inp4 = mxGetScalar(prhs[3]);
  inp5 = mxGetScalar(prhs[4]);
  inp6 = mxGetScalar(prhs[5]);
  inp7 = mxGetScalar(prhs[6]);
  inp8 = mxGetScalar(prhs[7]);
  inp9 = mxGetScalar(prhs[8]);
  inp10 = mxGetScalar(prhs[9]);
  inp11 = mxGetPr(prhs[10]);

  /* Create matrix for the return arguments. */
  plhs[0] = mxCreateDoubleMatrix(inp10,1, mxREAL);
  plhs[1] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[2] = mxCreateDoubleMatrix(1,1, mxREAL);

  /* Get pointer to real part of return argument */
  out1 = mxGetPr(plhs[0]);
  out2 = mxGetPr(plhs[1]);
  out3 = mxGetPr(plhs[2]);

  /* Recast Matlab inputs */
  idnum = (long)inp1;
  demo = (long)inp2;
  mode = (long)inp3;
  msDelay = (long)inp4;
  husDelay = (long)inp5;
  controlCS = (long)inp6;
  csLine = (long)inp7;
  csState = (long)inp8;
  configD = (long)inp9;
  numWriteRead = (long)inp10;

  for(i=0;i<17;i++)
	{
		data[i] = (long)*inp11;	//Fill data array
		inp11++;						//Increment pointer to channel array from Matlab
	}

  /* Call the Synch function. */
 errorcode = Synch(&idnum, demo, mode, msDelay, husDelay, controlCS, 
	  csLine, csState, configD, numWriteRead, data);

  /* Matlab outputs equal to returned LJ values */
  for(i=0;i<inp10;i++)
	{
	  out1[i] = (long)data[i];	//Fill data output array
	}

  out2[0] = (double)errorcode;
  out3[0] = (long)idnum;
   
}
