/********************************************************************************************************************************/
// VoltsToBits
//
// Converts a voltage to its 12-bit (0-4095) binary representation.
// Bits=(4096*((Volts*Gain)+Vmax))/(2*Vmax) where Vmax=10 for SE, 20 for Diff.
//
// MATLAB Syntax:
//		[bits errorcode] = VoltsToBits(chnum, chgain, volts)
//
// Input:
//		chnum - Channel idex.  0-7 SE, 8-11 Diff.
//		chgain - Gain index. 0=1, 1=2, .., 7=20.
//		volts - Voltage.
// Output:
//		bits - Binary value 0-4095.
// 
/********************************************************************************************************************************/

#include "mex.h"
#include <wtypes.h>
#include "ljackuw.h"

void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[] )
{
  double  inp1,inp2,inp3,*out1,*out2;
  int i, m, n;
  long errorcode, chnum, chgain, bits;
  float volts;
  bits = 0;

  /* Check for proper number of arguments. */
  if(nrhs != 3) 
  {
    mexErrMsgTxt("Input Error --> (chnum, chgain, volts)");
  } 
  else if(nlhs > 2) 
  {
    mexErrMsgTxt("Output Error --> [bits errorcode]");
  }

  /* Get dimensions of inputs */
  for (i=0;i<=2;i++)
  {
	 m = mxGetM(prhs[i]);
	 n = mxGetN(prhs[i]);
	 
	 if(m*n != 1)
	 {
		 mexErrMsgTxt("Inputs must to be scalar");
	 }
  }
  
  /* Retrive Matlab input values from function call. */
  inp1 = mxGetScalar(prhs[0]);
  inp2 = mxGetScalar(prhs[1]);
  inp3 = mxGetScalar(prhs[2]);

  /* Create matrix for the return arguments. */
  plhs[0] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[1] = mxCreateDoubleMatrix(1,1, mxREAL);

  /* Get pointer to real part of return argument */
  out1 = mxGetPr(plhs[0]);
  out2 = mxGetPr(plhs[1]);
  
  /* Recast Matlab inputs */
  chnum = (long)inp1;
  chgain = (long)inp2;
  volts = (float)inp3;

  /* Call the VoltsToBits function. */
  errorcode = VoltsToBits(chnum, chgain, volts, &bits);
  
  /* Set Matlab outputs equal to returned LJ values */
  out1[0] = (double)bits;
  out1[1] = (double)errorcode;
}
