/******************************************************************************************************************************/
// AIBurst
//
// Reads a specified number of scans (up to 4096) at a specified scan rate (up to 8192 Hz) from 1,2, or 4 analog inputs.  First, 
// data is acquired and stored in the LabJack's 4096 sample RAM buffer.  Then, the data is transferred to the PC If the LED is 
// enabled (ledOn>0), it will blink at about 4 Hz while waiting for a trigger, turn off during acquisition, blink rapidly while 
// transferring data to the PC.
//
// Matlab Syntax:
//		[voltages stateIOout scanRate overVoltage errorcode idnum] = AIBurst(idnum, demo, stateIOin, updateIO, ledOn, numChannels, 
//																			channels, gains, scanRate, diableCal, trigerIO, 
//																			triggerState, numScans, timeout,transfermode)
//
// Inputs:
//		idnum - Local Id, serial number, or -1 for first LJ found
//		demo - 0 for normal operation, >0 for demo mode.  Demo mode allows this function to be called without a LabJack.
//		stateIOin - Output states for IO3 - IO4.
//		updateIO - If > 0, sate values will be written.  Otherwise, just a read is performed.
//		ledOn - if > 0, the LabJack LED is turned on.
//		numChannels - Number of analog input channels to read (1,2, or 4).  
//		channels - Array of channel commands with at least numChannels elements. Each channel command is 0-7 for single-ended, 
//				   or 8-11 for differential.  
//		gains - Array of gain commands with at least numChannel elements.  Gain commands are 0=1, 1=2, ..., 7=20.  This 
//				amplification is available only for differential channels.  scanRate - Scans acquired per second.  A scan is a 
//				reading from every channel (1,2, or 4).  The sample rate (scanRate*numChannels) must be between 400 and 8192.
//		disableCal - If >0, voltages returned will be raw readings that are note corrected using calibration constants.
//		triggerIO - The IO port to trigger on (0=none, 1=IO0, ..., 4=IO3).
//		triggerState - If > 0, the acquisition will be triggered when the selected IO port reads high.
//		numScans - Number of scans which will be returned.  Minimum is 1.  Maximum numSamples is 4096, where numSamples is 
//				   numScans*numChannels.
//		timeout - This function will return immediately with a timeout error if it does not receive a scan within this number 
//				  of seconds 
//		transferMode - Always send 0			
//
// Outputs:
//		voltages - Array where voltage readings are returned.  Size of array is numScans x numChannels
//		stateIOout = Array where IO states are returned.
//		scanRate - Returns the actual scan rate, which due to clock resolution is not always exactly the same as the desired scan 
//				   rate.
//		overVoltage = if > 0 over voltage has been detected on one of the selected analog inputs
//		errorcode = LabJack error codes or 0 for no error.
//		idnum = Local ID number of LabJack, or -1 if no LabJack is found.
// 
/********************************************************************************************************************************/

#include "mex.h"
#include <wtypes.h>
#include "ljackuw.h"

void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[] )
{
  double  inp1,inp2,inp3,inp4,inp5,inp6,*inp7,*inp8,inp9,inp10, inp11, inp12, inp13, inp14, inp15;
  double  *out1,*out2,*out3,*out4,*out5,*out6;
  int i, j, m, n, cnt;
  long idnum, demo, stateIOin, updateIO, ledOn, numChannels, disableCal, triggerIO, triggerState, numScans;
  long timeout, transfermode, errorcode, overvoltage, transferMode, *ptr;
  long channels[4], gains[4];
  long stateIOout[4096] = {0};
  float Voltages[4096][4] = {0};
  float scanRate, *ptrf;


  /* Check for proper number of arguments. */
  if(nrhs != 15) 
  {
    mexErrMsgTxt("Input Error --> (Idnum, demo, stateIO, updateIO, ledOn, numChannels, channels, gains, disableCal)");
  } 
  else if(nlhs > 6) 
  {
    mexErrMsgTxt("Output Error --> [Voltages stateIO OverVoltage Errorcode Idnum]");
  }

  /* Get dimensions of inputs */
  //for (i=0;i<=9;i++)
  //{
//	 m = mxGetM(prhs[i]);
//	 n = mxGetN(prhs[i]);
	 
//	 if(m*n != 1 & i==)
//	 {
//		 mexErrMsgTxt("Inputs must to be scalar");
//	 }
//  }
  
  /* Retrive Matlab input values from function call. */
  inp1 = mxGetScalar(prhs[0]);
  inp2 = mxGetScalar(prhs[1]);
  inp3 = mxGetScalar(prhs[2]);
  inp4 = mxGetScalar(prhs[3]);
  inp5 = mxGetScalar(prhs[4]);
  inp6 = mxGetScalar(prhs[5]);
  inp7 = mxGetPr(prhs[6]);
  inp8 = mxGetPr(prhs[7]);
  inp9 = mxGetScalar(prhs[8]);
  inp10 = mxGetScalar(prhs[9]);
  inp11 = mxGetScalar(prhs[10]);
  inp12 = mxGetScalar(prhs[11]);
  inp13 = mxGetScalar(prhs[12]);
  inp14 = mxGetScalar(prhs[13]);
  inp15 = mxGetScalar(prhs[14]);

  /* Create matrix for the return arguments. */
  plhs[0] = mxCreateDoubleMatrix(inp13,inp6, mxREAL);
  plhs[1] = mxCreateDoubleMatrix(inp13,1, mxREAL);
  plhs[2] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[3] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[4] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[5] = mxCreateDoubleMatrix(1,1, mxREAL);

  /* Get pointer to real part of return argument */
  out1 = mxGetPr(plhs[0]);
  out2 = mxGetPr(plhs[1]);
  out3 = mxGetPr(plhs[2]);
  out4 = mxGetPr(plhs[3]);
  out5 = mxGetPr(plhs[4]);
  out6 = mxGetPr(plhs[5]);


  /* Recast Matlab inputs */
  idnum = (long)inp1;
  demo = (long)inp2;
  stateIOin = (long)inp3;
  updateIO = (long)inp4;
  ledOn = (long)inp5;
  numChannels = (long)inp6;
  scanRate = (float)inp9;
  disableCal = (long)inp10;
  triggerIO = (long)inp11;
  triggerState = (long)inp12;
  numScans = (long)inp13;
  timeout = (long)inp14;
  transferMode = (long)inp15;



  for(i=0;i<=3;i++)
	{
		channels[i] = (long)*inp7;	//Fill channel array
		gains[i] = (long)*inp8;		//Fill gain array
		inp7++;						//Increment pointer to channel array from Matlab
		inp8++;						//Increment pointer to gains array from matlab
	}

  /* Call the AIBurst function. */
  errorcode = AIBurst(&idnum, demo, stateIOin, updateIO, ledOn, numChannels, &channels[0], &gains[0], &scanRate, disableCal, triggerIO, triggerState, numScans, timeout, &Voltages[0], stateIOout, &overvoltage, transferMode);
  

  /* Matlab outputs equal to returned LJ values */
  cnt = 0;
  for(i=0;i<=(int)numChannels - 1;i++)		//num of channels
  {
	  for(j=0;j<=(int)numScans - 1;j++)//num of samples
		{
		out1[cnt] = Voltages[j][i];	//Fill voltage output array
		cnt++;
		}
  }

  ptr = &stateIOout[0];				//set pointer to first value in stateIOout array (4096x1)
  for(i=0;i<=(int)numScans - 1;i++)
	{
	out2[i] = (double)*ptr;			//Fill steteIOout array
	ptr++;
	}
  
  out3[0] = (double)scanRate;
  out4[0] = (double)overvoltage;
  out5[0] = (double)errorcode;
  out6[0] = (double)idnum;
}
