/******************************************************************************************************************************/
// AIStreamRead
//
// Waits for specified number of scans to be available and reads them.  AIStreamStart should be called before this function and AIStreamClear should be called when finished wht the stream.
//
// MATLAB Syntax:
//		[voltages stateIOout LjScanBacklog overVoltage errorcode] = AIStreamRead(localID, numChannels, numScans, timeout)
//
// Inputs:
//		localId - Send the local ID from AIStreamStart
//		numChannels - Number of analog input channels to read (1,2, or 4).  If readCount is >0, numChannels should be 4.
//		numScans - Function will wait until this number of scans is available.  Minimum is 1.  Maximum numSamples is 4096, where numSamples is numScans*numChannels.  Internally this function gets data from the LabJack in blocks of 64 samples, so it is recommended that numSamples be at least 64.
//		timeout - Function timeout value in seconds. 
//
// Outputs:
//		voltages - Array where voltage readings are returned.  Array size is numScans x numChannels.
//		stateIOout - Array where IO readings are returned.  Array size is 
//					 numScans x numChannels.
//		ljScanBacklog - Returns the scan backlog of the LabJack Ram buffer.  The size of the buffer in terms of scans is 4096/numChannels.
//		overVoltage - If>0 an overvoltage has been detected on at least one sample of one of the selected analog inputs.
//		errorcode = LabJack error codes or 0 for no error.
//
/********************************************************************************************************************************/

#include "mex.h"
#include <wtypes.h>
#include "ljackuw.h"

void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[] )
{
  double  inp1,inp2,inp3, inp4,*out1,*out2,*out3,*out4,*out5;
  int i, j, m, n, cnt;
  long localId, numScans, numChannels, timeout, reserved, ljScanBacklog, overVoltage, errorcode, overvoltage, *ptr;
  float scanRate;
  long stateIOout[4096] = {0};
  float voltages[4096][4] = {0};


  /* Check for proper number of arguments. */
  if(nrhs != 4) 
  {
    mexErrMsgTxt("Input Error --> (localID, numChannels, numScans, timeout)");
  } 
  else if(nlhs > 5) 
  {
    mexErrMsgTxt("Output Error --> [voltages stateIOout LjScanBacklog overVoltage errorcode]");
  }

  /* Get dimensions of inputs */
  //for (i=0;i<=9;i++)
  //{
//	 m = mxGetM(prhs[i]);
//	 n = mxGetN(prhs[i]);
	 
//	 if(m*n != 1 & i==)
//	 {
//		 mexErrMsgTxt("Inputs must to be scalar");
//	 }
//  }
  
  /* Retrive Matlab input values from function call. */
  inp1 = mxGetScalar(prhs[0]);
  inp2 = mxGetScalar(prhs[1]);
  inp3 = mxGetScalar(prhs[2]);
  inp4 = mxGetScalar(prhs[3]);

  /* Create matrix for the return arguments. */
  plhs[0] = mxCreateDoubleMatrix(inp3,inp2, mxREAL);
  plhs[1] = mxCreateDoubleMatrix(inp3,1, mxREAL);
  plhs[2] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[3] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[4] = mxCreateDoubleMatrix(1,1, mxREAL);

  /* Get pointer to real part of return argument */
  out1 = mxGetPr(plhs[0]);
  out2 = mxGetPr(plhs[1]);
  out3 = mxGetPr(plhs[2]);
  out4 = mxGetPr(plhs[3]);
  out5 = mxGetPr(plhs[4]);
  
  /* Recast Matlab inputs */
  localId = (long)inp1;
  numChannels = (long)inp2;
  numScans = (long)inp3;
  timeout = (long)inp4;
  reserved = 0;


  /* Call the AIStreamRead function. */
  errorcode = AIStreamRead(localId, numScans, timeout, &voltages[0], stateIOout, &reserved, &ljScanBacklog, &overVoltage);
  
  /* Matlab outputs equal to returned LJ values */
  cnt = 0;
  for(i=0;i<=(int)numChannels - 1;i++)	//num of channels
  {
	  for(j=0;j<=(int)numScans - 1;j++)	//num of samples
		{
		out1[cnt] = voltages[j][i];		//Fill voltage output array
		cnt++;
		}
  }

  ptr = &stateIOout[0];				//set pointer to first value in stateIOout array (4096x1)
  for(i=0;i<=(int)numScans - 1;i++)
	{
	out2[i] = (double)*ptr;			//Fill steteIOout array
	ptr++;
	}


  out3[0] = (double)ljScanBacklog;
  out4[0] = (double)overVoltage;
  out5[0] = (double)errorcode;
}