/******************************************************************************************************************************/
// Asynch:		Requires firmware V1.05 or higher.
//
//				This function writes and then reads half-duplex asynchronous
//				data on 1 of two pairs of D lines (8,n,1).  Call AsynchConfig
//				to set the baud rate.  Similar to RS232, except that logic is
//				normal CMOS/TTL (0=low=GND, 1=high=+5V, idle state of
//				transmit line is high).  Connection to a normal RS232 device
//				will probably require a converter chip such as the MAX233.
//
//				PortA =>  TX is D8 and RX is D9
//				PortB =>  TX is D10 and RX is D11
//				Transmit Enable is D12
//
//				Up to 18 bytes can be written and read.  If more than 4 bytes
//				are written or read, this function uses calls to
//				WriteMem/ReadMem to load/read the LabJack's data buffer.
//
//MATLAB Syntax:
//		[data errorcode idnum] = Asynch(idnum, demo, portB, enableTE, enableTO, enableDel,
//										baudrate, numWrite, numRead, data)
//
//	Returns:	LabJack errorcodes or 0 for no error (I32).
//	Inputs:		*idnum		-Local ID, Serial Number, or -1 for first
//							 found (I32).
//				demo		-Send 0 for normal operation, >0 for demo
//							 mode (I32).  Demo mode allows this function to
//							 be called without a LabJack, and does little but
//							 simulate execution time.
//				portB		-If >0, asynch PortB is used instead of PortA.
//				enableTE	-If >0, D12 (Transmit Enable) is set high during
//							 transmit and low during receive (I32).
//				enableTO	-If >0, timeout is enabled for the receive phase (per byte).
//				enableDel	-If >0, a 1 bit delay is inserted between each
//							 transmit byte.
//				baudrate	-This is the bps as set by AsynchConfig.  Asynch needs this
//							 so it has an idea how long the transfer should take.
//				numWrite	-Number of bytes to write (I32, 0-18).
//				numRead		-Number of bytes to read (I32, 0-18).
//				*data		-Serial data buffer.  Send an 18 element
//							 array.  Fill unused locations with zeros (I32).
//	Outputs:	*idnum		-Returns the Local ID or -1 if no LabJack is
//							 found (I32).
//				*data		-Serial data buffer.  Returns any serial read
//							 data.  Unused locations are filled
//							 with 9999s. (I32).
//
//	Time:		20 ms to read & write up to 4 bytes, plus 40 ms for each
//				additional 4 bytes to read or write.  Possibly extra
//				time for slow baud rates.

/********************************************************************************************************************************/

#include "mex.h"
#include <wtypes.h>
#include "ljackuw.h"

void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[] )
{
  double  inp1,inp2,inp3,inp4,inp5,inp6,inp7,inp8,inp9,*inp10;

  double* out1, *out2, *out3;
  int i, m, n, g;
  long idnum, demo, portB, enableTE, enableTO, enableDel, baudrate, numWrite, numRead, errorcode;
  long  data[18];

  for (g=0;g<=17;g++)
  {
	  data[g]=0;
  }

  /* Check for proper number of arguments. */
  if(nrhs != 10) 
  {
    mexErrMsgTxt("Input Error --> (idnum, demo, portB, enableTE, enableTO, enableDel, baudrate, numWrite, numRead, data)");
  } 				   
  else if(nlhs > 3) 
  {
    mexErrMsgTxt("Output Error --> [data errorcode idnum]");
  }

  /* Get dimensions of inputs */
  for (i=0;i<=8;i++)
  {
	 m = mxGetM(prhs[i]);
	 n = mxGetN(prhs[i]);
	 
	 if(m*n != 1)
	 {
		 mexErrMsgTxt("Inputs must be scalar:");
	 }
  }
	
  /* Retrive Matlab input values from function call. */
  inp1 = mxGetScalar(prhs[0]);
  inp2 = mxGetScalar(prhs[1]);
  inp3 = mxGetScalar(prhs[2]);
  inp4 = mxGetScalar(prhs[3]);
  inp5 = mxGetScalar(prhs[4]);
  inp6 = mxGetScalar(prhs[5]);
  inp7 = mxGetScalar(prhs[6]);
  inp8 = mxGetScalar(prhs[7]);
  inp9 = mxGetScalar(prhs[8]);
  inp10 = mxGetPr(prhs[9]);

  /* Create matrix for the return arguments. */
  plhs[0] = mxCreateDoubleMatrix(inp8,1, mxREAL);
  plhs[1] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[2] = mxCreateDoubleMatrix(1,1, mxREAL);

  /* Get pointer to real part of return argument */
  out1 = mxGetPr(plhs[0]);
  out2 = mxGetPr(plhs[1]);
  out3 = mxGetPr(plhs[2]);

  /* Recast Matlab inputs */
  idnum = (long)inp1;
  demo = (long)inp2;
  portB = (long)inp3;
  enableTE = (long)inp4;
  enableTO = (long)inp5;
  enableDel = (long)inp6;
  baudrate = (long)inp7;
  numWrite = (long)inp8;
  numRead = (long)inp9;

  for(i=0;i<inp8;i++)
	{
	if (i < inp9)
		{
		  data[i] = (long)*inp10;		//Fill data array
		  inp10++;						//Increment pointer to channel array from Matlab
		}
	else data[i]=0;
  }

  /* Call the Asynch function. */
 errorcode = Asynch(&idnum, demo, portB, enableTE, enableTO, enableDel, 
	  baudrate, numWrite, numRead, data);

  /* Matlab outputs equal to returned LJ values */
  for(i=0;i<=inp8;i++)
	{
	  out1[i] = (long)data[i];	//Fill data output array
	}

  out2[0] = (double)errorcode;
  out3[0] = (long)idnum;
   
}
