/*************************************************************************************************/
// EAnalogIn
//
// Easy function.  This is a simplified version of AISample. Reads the voltage from 1 analog input.  
// Calling this function turns/leaves the status LED on.  Execution time for this function is 50 ms 
// or less.
//
// MATLAB Syntax:
//		[voltage OverVoltage errorcode Idnum] = EAnlaogIn(Idnum, demo, channel, gain)
// 
// Inputs:
//		idnum - Local Id, serial number, or -1 for first LJ found
//		demo - 0 for normal operation, >0 for demo mode.  Demo mode allows this function to be called      
//			   without a LabJack.
//		channel - Channel command is 0-7 for single-ended, or 8-11 for differential
//		gain - Gain command is 0=1, 1=2, ..., 7=20.
//
// Outputs:
//		voltage - Returns the voltage reading.
//		OverVoltage - If >0 over voltage has been detected on one of the selected analog inputs
//		errorcode - LabJack error codes or 0 for no error.
//		idnum - Local ID number of Labjack, or -1 if no LabJack is found.
//
// 
/**************************************************************************************************/

#include "mex.h"
#include <wtypes.h>
#include "ljackuw.h"

void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[] )
{
  double  inp1,inp2,inp3,inp4,inp5,*out1,*out2,*out3,*out4;
  int i, m, n;
  long idnum, demo, channel, gain, errorcode, overvoltage;
  float voltage;

  /* Check for proper number of arguments. */
  if(nrhs != 4) 
  {
    mexErrMsgTxt("Input Error --> (Idnum, demo, channel, gain)");
  } 
  else if(nlhs > 4) 
  {
    mexErrMsgTxt("Output Error --> [Voltage OverVoltage Errorcode Idnum]");
  }

  /* Get dimensions of inputs */
  for (i=0;i<=3;i++)
  {
	 m = mxGetM(prhs[i]);
	 n = mxGetN(prhs[i]);
	 
	 if(m*n != 1)
	 {
		 mexErrMsgTxt("Inputs must to be scalar");
	 }
  }
  
  /* Create matrix for the return arguments. */
  plhs[0] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[1] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[2] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[3] = mxCreateDoubleMatrix(1,1, mxREAL);
  
  /* Retrive Matlab input values from function call. */
  inp1 = mxGetScalar(prhs[0]);
  inp2 = mxGetScalar(prhs[1]);
  inp3 = mxGetScalar(prhs[2]);
  inp4 = mxGetScalar(prhs[3]);

  /* Get pointer to real part of return argument */
  out1 = mxGetPr(plhs[0]);
  out2 = mxGetPr(plhs[1]);
  out3 = mxGetPr(plhs[2]);
  out4 = mxGetPr(plhs[3]);
  
  /* Recast Matlab inputs */
  idnum = (long)inp1;
  demo = (long)inp2;
  channel = (long)inp3;
  gain = (long)inp4;

  /* Call the EAnalogin function. */
  errorcode = EAnalogIn(&idnum, demo, channel, gain, &overvoltage, &voltage);
  
  /* Matlab outputs equal to returned LJ values */
  out1[0] = (double)voltage;
  out2[0] = (double)overvoltage;
  out3[0] = (double)errorcode;
  out4[0] = (double)idnum;
}
