/*************************************************************************************************/
// ECount
//
// Easy function.  This is a simplified version of Counter.  Reads and resets the counter (CNT) 
// Calling this function disables STB (which is the default anyway).  Execution time for this 
// function is 50 milliseconds or less.
//
// MATLAB Syntax:
//		[count ms errorcode Idnum] = ECount(idnum, demo, resetCounter)
//
// Inputs:
//		idnum - Local Id, serial number, or -1 for first LJ found
//		demo - 0 for normal operation, >0 for demo mode.  Demo mode allows this function to be 
//		called without a LabJack.
//		resetCounter - If >0, the counter is reset to zero after being read.
//
// Outputs:
//		count = Current count, before reset.
//		ms - Value of Window's millisecond timer at the time of the counter read (within a few ms).  
//			 Note that the millisecond timer rolls over about every 50 days.  In general, the 
//			 millisecond timer starts counting from zero whenever the computer reboots.
//		errorcode - LabJack error code or 0 for no error.
//		idnum - Returns the local ID or -1 if no LabJack is found. ECount
//
/**************************************************************************************************/

#include "mex.h"
#include <wtypes.h>
#include "ljackuw.h"

void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[] )
{
  double  inp1,inp2,inp3,*out1,*out2,*out3,*out4;
  int i, m, n;
  long idnum, demo, resetCounter, errorcode;
  double count, ms;

  /* Check for proper number of arguments. */
  if(nrhs != 3) 
  {
    mexErrMsgTxt("Input Error --> (idnum, demo, resetCounter)");
  } 
  else if(nlhs > 4) 
  {
    mexErrMsgTxt("Output Error --> [count, ms, errorcode, idnum]");
  }

  /* Get dimensions of inputs */
  for (i=0;i<=2;i++)
  {
	 m = mxGetM(prhs[i]);
	 n = mxGetN(prhs[i]);
	 
	 if(m*n != 1)
	 {
		 mexErrMsgTxt("Inputs must to be scalar");
	 }
  }
  
  /* Create matrix for the return arguments. */
  plhs[0] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[1] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[2] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[3] = mxCreateDoubleMatrix(1,1, mxREAL);
  
  /* Retrive Matlab input values from function call. */
  inp1 = mxGetScalar(prhs[0]);
  inp2 = mxGetScalar(prhs[1]);
  inp3 = mxGetScalar(prhs[2]);

  /* Get pointer to real part of return argument */
  out1 = mxGetPr(plhs[0]);
  out2 = mxGetPr(plhs[1]);
  out3 = mxGetPr(plhs[2]);
  out4 = mxGetPr(plhs[3]);
  
  /* Recast Matlab inputs */
  idnum = (long)inp1;
  demo = (long)inp2;
  resetCounter = (long)inp3;

  /* Call the ECount function. */
  errorcode = ECount(&idnum, demo, resetCounter, &count, &ms);
  
  /* Matlab outputs equal to returned LJ values */
  out1[0] = count;
  out2[0] = ms;
  out3[0] = (double)errorcode;
  out4[0] = (double)idnum;

}
