/*************************************************************************************************/
// EDigitalOut
//
// Easy function.  This is a simplified version of DigitalIO that sets/clears the state of one digital output.  Also configures the requested pin to output and leave it that way.  Execution time for this function is 50 ms or less.  
//
// Note that this is a simplified version of the lower level function DigitalIO, which operates on all 20 digital lines.  The DLL (ljackuw) attempts to keep track of the current direction and output state of all lines, so that this easy function can operate on a single line without changing the others.  When the DLL is first loaded, though, it does not know the direction and state for the lines and assumes all directions are input and output stares are low.
//
// Matlab Syntax:
//		[errorcode idnum] = EDigitalOut(idnum, demo, channel, writeD, state)
//
// Inputs:
//		idnum - Local Id, serial number, or -1 for first LJ found
//		demo - 0 for normal operation, >0 for demo mode.  Demo mode allows this function to be called   without a LabJack.
//		channel - Line to read.  0-3 for IO or 0-15 for D.
//		writeD - If > 0, a D line is written as opposed to an IO line.
//		state - If > 0 the line is set, otherwise the line is cleared.
//
// Outputs:
//		errorCode - LabJack errorcode or 0 for no error.
//		dnum - Returns the local ID or -1 if no LabJack is found.
//
/**************************************************************************************************/

#include "mex.h"
#include <wtypes.h>
#include "ljackuw.h"

void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[] )
{
  double  inp1,inp2,inp3,inp4,inp5,*out1,*out2;
  int i, m, n;
  long idnum, demo, channel, writeD, state, errorcode;

  /* Check for proper number of arguments. */
  if(nrhs != 5) 
  {
    mexErrMsgTxt("Input Error --> (Idnum, demo, channel, writeD, state)");
  } 
  else if(nlhs > 2) 
  {
    mexErrMsgTxt("Output Error --> [Errorcode, Idnum]");
  }

  /* Get dimensions of inputs */
  for (i=0;i<=4;i++)
  {
	 m = mxGetM(prhs[i]);
	 n = mxGetN(prhs[i]);
	 
	 if(m*n != 1)
	 {
		 mexErrMsgTxt("Inputs must to be scalar");
	 }
  }
  
  /* Create matrix for the return arguments. */
  plhs[0] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[1] = mxCreateDoubleMatrix(1,1, mxREAL);
  
  /* Retrive Matlab input values from function call. */
  inp1 = mxGetScalar(prhs[0]);
  inp2 = mxGetScalar(prhs[1]);
  inp3 = mxGetScalar(prhs[2]);
  inp4 = mxGetScalar(prhs[3]);
  inp5 = mxGetScalar(prhs[4]);

  /* Get pointer to real part of return argument */
  out1 = mxGetPr(plhs[0]);
  out2 = mxGetPr(plhs[1]);
  
  /* Recast Matlab inputs */
  idnum = (long)inp1;
  demo = (long)inp2;
  channel = (long)inp3;
  writeD = (long)inp4;
  state = (long)inp5;

  /* Call the EDigitalOut function. */
  errorcode = EDigitalOut(&idnum, demo, channel, writeD, state);
  
  /* Matlab outputs equal to returned LJ values */
  out1[0] = (double)errorcode;
  out2[0] = (double)idnum;

}
