/******************************************************************************************************************************/
// PulseOut
//
// PulseOut:	Requires firmware V1.05 or higher.
//
//				The timeout of this function, in milliseconds, is set to:
//					5000+numPulses*((B1*C1*0.02)+(B2*C2*0.02))
//
//				This command creates pulses on any/all of D0-D7.  The
//				desired D lines must be set to output using another
//				function (DigitalIO or AOUpdate).  All selected lines
//				are pulsed at the same time, at the same rate, for the
//				same number of pulses.
//
//				This function commands the time for the first half cycle
//				of each pulse, and the second half cycle of each pulse.
//				Each time is commanded by sending a value B & C, where
//				the time is,
//
//				1st half-cycle microseconds = ~17 + 0.83*C + 20.17*B*C
//				2nd half-cycle microseconds = ~12 + 0.83*C + 20.17*B*C
//
//				which can be approximated as,
//
//				microseconds = 20*B*C
//
//				For best accuracy when using the approximation, minimize C.
//				B and C must be between 1 and 255, so each half cycle can
//				vary from about 38/33 microseconds to just over 1.3 seconds.
//
//				If you have enabled the LabJack Watchdog function, make sure
//				it's timeout is longer than the time it takes to output all
//				pulses.
//				
//MATLAB Syntax:
//		[errorcode idnum] = PulseOut(idnum, demo, lowFirst, bitSelect, numPulses,
//								timeB1, timeC1, timeB2, timeC2)
//
//	Returns:	LabJack errorcodes or 0 for no error (I32).
//	Inputs:		*idnum		-Local ID, Serial Number, or -1 for first
//							 found (I32).
//				demo		-Send 0 for normal operation, >0 for demo
//							 mode (I32).  Demo mode allows this function to
//							 be called without a LabJack, and does little but
//							 simulate execution time.
//				lowFirst	-If >0, each line is set low then high, otherwise
//							 the lines are set high then low (I32).
//				bitSelect	-Set bits 0 to 7 to enable pulsing on each of
//							 D0-D7 (I32, 0-255).
//				numPulses	-Number of pulses for all lines (I32, 1-32767).
//				timeB1		-B value for first half cycle (I32, 1-255).
//				timeC1		-C value for first half cycle (I32, 1-255).
//				timeB2		-B value for second half cycle (I32, 1-255).
//				timeC2		-C value for second half cycle (I32, 1-255).
//	Outputs:	*idnum		-Returns the Local ID or -1 if no LabJack is
//							 found (I32).
//
//	Time:		20 ms plus pulse time (make sure watchdog is longer if active)
/********************************************************************************************************************************/

#include "mex.h"
#include <wtypes.h>
#include "ljackuw.h"

void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[] )
{
  double  inp1,inp2,inp3,inp4,inp5,inp6,inp7,inp8,inp9;
  double *out1, *out2;
  int i, m, n;
  long idnum, demo, lowFirst, bitSelect, numPulses, timeB1, timeC1, timeB2, timeC2, errorcode;

  /* Check for proper number of arguments. */
  if(nrhs != 9) 
  {
    mexErrMsgTxt("Input Error --> (idnum, demo, lowFirst, bitSelect, numPulses, timeB1, timeC1, timeB2, timeC2)");
  } 
  else if(nlhs > 2) 
  {
    mexErrMsgTxt("Output Error --> [errorcode idnum]");
  }

  /* Get dimensions of inputs */
  for (i=0;i<9;i++)
  {
	 m = mxGetM(prhs[i]);
	 n = mxGetN(prhs[i]);
	 
	 if(m*n != 1)
	 {
		 mexErrMsgTxt("Inputs must be scalar:");
	 }
  }
	
  /* Retrive Matlab input values from function call. */
  inp1 = mxGetScalar(prhs[0]);
  inp2 = mxGetScalar(prhs[1]);
  inp3 = mxGetScalar(prhs[2]);
  inp4 = mxGetScalar(prhs[3]);
  inp5 = mxGetScalar(prhs[4]);
  inp6 = mxGetScalar(prhs[5]);
  inp7 = mxGetScalar(prhs[6]);
  inp8 = mxGetScalar(prhs[7]);
  inp9 = mxGetScalar(prhs[8]);

  /* Create matrix for the return arguments. */
  plhs[0] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[1] = mxCreateDoubleMatrix(1,1, mxREAL);

  /* Get pointer to real part of return argument */
  out1 = mxGetPr(plhs[0]);
  out2 = mxGetPr(plhs[1]);

  /* Recast Matlab inputs */
  idnum = (long)inp1;
  demo = (long)inp2;
  lowFirst = (long)inp3;
  bitSelect = (long)inp4;
  numPulses = (long)inp5;
  timeB1 = (long)inp6;
  timeC1 = (long)inp7;
  timeB2 = (long)inp8;
  timeC2 = (long)inp9;

  /* Call the PulseOut function. */
 errorcode = PulseOut(&idnum, demo, lowFirst, bitSelect, numPulses, 
	 timeB1, timeC1, timeB2, timeC2);

  /* Matlab outputs equal to returned LJ values */

	  out1[0] = (double)errorcode;
	  out2[0] = (long)idnum;	
}
