/********************************************************************************************************************************/
// Watchdog
//
// Controls the LabJack watchdog function.  When activated, the watchdog can change the states of digital I/O if the LabJack does 
// not successfully communicate with the PC within a specified timeout periods.  This function could be used to reboot the PC 
// allowing for reliable unattended operation.  The 32-bit counter (CNT) is disabled when the watchdog is enabled.  Execution time 
// for this function is 50 ms or less.  
//
// If you set the watchdog to reset the LabJack, and choose too small of timeout period, it might be difficult to make the device 
// stop resetting.  To disable the watchdog, reset the LabJack with IO0 shorted to STB, and then reset again without the short.
//
// MATLAB Syntax:  
//		[idnum errorcode] = Reset(idnum, demo, active, timeout, reset, activeD0, activeD1, activeD8, stateD0, stateD1, stateD8)
// 
// Input:
//		idnum - Local ID, serial number, or -1 for first found
//		demo - Send 0 for normal operation >0 for demo mode.  Demo mode allows this function to be called without a LabJack
//		active - Enables the LabJack watchdog function.  If enabled, the 32-bit counter is disabled.
//		timeout - Timer reset value in seconds.
//		reset - If>0, the LabJack will reset on timeout.
//		activeDn - If >0, Dn will be set to stateDn upon timeout (activeD0 - D8).
//		stateDn - Timeout state of Dn, 0 = Low, >0 = High (stateD0 - D8)
//
// Output:
//		idnum - Returns the local ID or -1 if no LabJack is found.
//		errorcode - LabJack error codes or 0 for no error.
// 
/********************************************************************************************************************************/

#include "mex.h"
#include "matrix.h"
#include <wtypes.h>
#include "ljackuw.h"

void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[] )
{
  double  *out1, *out2;
  int i, m, n;
  double inp1, inp2, inp3, inp4, inp5, inp6, inp7, inp8, inp9, inp10, inp11;
  long idnum, errorcode, demo, active, timeout, reset, activeD0, activeD1, activeD8, stateD0, stateD1, stateD8;

  /* Check for proper number of arguments. */
  if(nrhs != 11) 
  {
    mexErrMsgTxt("Input Error --> (idnum, demo, active, timeout, reset, activeD0, activeD1, activeD8, stateD0, stateD1, stateD8)");
  } 
  else if(nlhs > 2) 
  {
    mexErrMsgTxt("Output Error --> [idnum errorcode]");
  }

  /* Get dimensions of inputs */
  for (i=0;i<=10;i++)
  {
	 m = mxGetM(prhs[i]);
	 n = mxGetN(prhs[i]);
	 
	 if(m*n != 1)
	 {
		 mexErrMsgTxt("Input must to be scalar");
	 }
  }

  /* Retrive Matlab input values from function call. */
  inp1 = mxGetScalar(prhs[0]);
  inp2 = mxGetScalar(prhs[1]);
  inp3 = mxGetScalar(prhs[2]);
  inp4 = mxGetScalar(prhs[3]);
  inp5 = mxGetScalar(prhs[4]);
  inp6 = mxGetScalar(prhs[5]);
  inp7 = mxGetScalar(prhs[6]);
  inp8 = mxGetScalar(prhs[7]);
  inp9 = mxGetScalar(prhs[8]);
  inp10 = mxGetScalar(prhs[9]);
  inp11 = mxGetScalar(prhs[10]);

  /* Create matrix for the return arguments. */;
  plhs[0]= mxCreateDoubleMatrix(1,1,mxREAL);
  plhs[1]= mxCreateDoubleMatrix(1,1,mxREAL);

  /* Get pointer to real part of return argument */
  out1 = mxGetPr(plhs[0]);
  out2 = mxGetPr(plhs[1]);

  /* Recast Matlab inputs */
  idnum = long(inp1);
  demo = long(inp2); 
  active = long(inp3);
  timeout = long(inp4);
  reset = long(inp5);
  activeD0 = long(inp6);
  activeD1 = long(inp7);
  activeD8 = long(inp8);
  stateD0 = long(inp9);
  stateD1 = long(inp10);
  stateD8 = long(inp11);

  /* Call the Watchdog function. */
  errorcode = Watchdog(&idnum, demo, active, timeout, reset, activeD0, activeD1, activeD8, stateD0, stateD1, stateD8);
  
  /* Set Matlab outputs equal to returned LJ values */
  out1[0] = (double)idnum;
  out2[0] = (double)errorcode;

}
