/********************************************************************************************************************************/
// WriteMem
//
// Writes 4 bytes to the LabJack's 8,192 byte nonvolatile memory at a specified address.  The data is read back and verified after 
// the write.  Memory 0-511 is reserved for the configuration and calibration data.  Memory from512-1023 is unused by the LabJack 
// and available for the user (this corresponds to starting addresses from 512-1020).  Memory 1024-8191 is used as a data buffer 
// in hardware timed AI modes (burst and stream).  Execution time for this function is 50 milliseconds or less.
//
// MATLAB Syntax:
//		[errorcode idnum] = WriteMem(idnum, unlocked, address, data3, data2, data1, data0)
//
//
// Inputs:
//		idnum - Local ID, serial number, or -1 for first LabJack found.
//		unlocked - If>0, addresses 0-511 ae unlocked fro writing.
//		address - Starting address of data to read (0-8188)
//		data3 - Byte for address.
//		data2 - Byte for address+1.
//		data1 - Byte for address+2.
//		data0 - Byte for address+3.				
//
// Outputs:
//
//		idnum - returns the local ID or -1 if no LabJack is found.
//		errorcode - LabJack error codes or 0 for no error.
//
//*******************************************************************************************************************************/

#include "mex.h"
#include "matrix.h"
#include <wtypes.h>
#include "ljackuw.h"

void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[] )
{
  double  inp1, inp2, inp3, inp4, inp5, inp6, inp7, *out1, *out2;
  int i, m, n;
  long unlocked, address, data3, data2, data1, data0, idnum, errorcode;

  /* Check for proper number of arguments. */
  if(nrhs != 7) 
  {
    mexErrMsgTxt("Input Error --> (idnum unlocked address data3 data2 data1 data0)");
  } 
  else if(nlhs > 2) 
  {
    mexErrMsgTxt("Output Error --> [errorcode idnum]");
  } 

  /* Get dimensions of inputs */
  for (i=0;i<=6;i++)
  {
	 m = mxGetM(prhs[i]);
	 n = mxGetN(prhs[i]);
	 
	 if(m*n != 1)
	 {
		 mexErrMsgTxt("Input must to be scalar");
	 }
  }

  /* Retrive Matlab input values from function call. */
  inp1 = mxGetScalar(prhs[0]);
  inp2 = mxGetScalar(prhs[1]);
  inp3 = mxGetScalar(prhs[2]);
  inp4 = mxGetScalar(prhs[3]);
  inp5 = mxGetScalar(prhs[4]);
  inp6 = mxGetScalar(prhs[5]);
  inp7 = mxGetScalar(prhs[6]);


  /* Create matrix for the return arguments based upon the number of LabJacks present */
  plhs[0]= mxCreateDoubleMatrix(1,1,mxREAL);
  plhs[1]= mxCreateDoubleMatrix(1,1,mxREAL);

  /* Get pointer to real part of return argument */
  out1 = mxGetPr(plhs[0]);
  out2 = mxGetPr(plhs[1]);

  /* Recast Matlab inputs */
  idnum = long(inp1);
  unlocked = long(inp2);
  address = long(inp3);
  data3 = long(inp4);
  data2 = long(inp5);
  data1 = long(inp6);
  data0 = long(inp7);
  
  /* Call the WriteMem function. */
  errorcode = WriteMem(&idnum, unlocked, address, data3, data2, data1, data0);

  /* Matlab outputs equal to returned LJ values */

  out1[0] = (double)errorcode;
  out2[0] = (double)idnum;

}
