//Author: LabJack
//April 29, 2008
//This example program calls the Feedback low-level function in a continuous loop
//and logs readings to a file.  It will read the voltages from AIN0-AIN3, set and 
//read the states and directions from FIO4 - FIO5, set DAC0, read Counter0 (FIO7).  
//This example also uses the ConfigIO and ConfigTimerClock low-level functions to configure the 
//IO channels and set the timer clock.  The first U3 device found is used in this example. 

using System;
using System.Drawing;
using System.Collections;
using System.Windows.Forms;
using System.Data;
using System.Threading;
using System.IO;
using System.Reflection;
using LabJack;

namespace u3Logger
{
	/// <summary>
	/// Summary description for Form1.
	/// </summary>
	public class u3Logger : System.Windows.Forms.Form
	{
		private System.Windows.Forms.Label label1;
		private System.Windows.Forms.Label label2;
		private System.Windows.Forms.Label label3;
		private System.Windows.Forms.Label label4;
		private System.Windows.Forms.Label label5;
		private System.Windows.Forms.Label label6;
		private System.Windows.Forms.Label label8;
		private System.Windows.Forms.Label label9;
		private System.Windows.Forms.TextBox errorBox;
		private System.Windows.Forms.Label ain0lbl;
		private System.Windows.Forms.Label ain1lbl;
		private System.Windows.Forms.Label ain2lbl;
		private System.Windows.Forms.Panel panel1;
		private System.Windows.Forms.Panel panel2;
		private System.Windows.Forms.Label label10;
		private System.Windows.Forms.Label label11;
		private System.Windows.Forms.Label label12;
		private System.Windows.Forms.Label label13;
		private System.Windows.Forms.Label label14;
		private System.Windows.Forms.Panel panel3;
		private System.Windows.Forms.Panel panel4;
		private System.Windows.Forms.Label label16;
		private System.Windows.Forms.MainMenu mainMenu1;
		private System.Windows.Forms.CheckBox fio4StateCB;
		private System.Windows.Forms.CheckBox fio4DirCB;
		private System.Windows.Forms.CheckBox fio5DirCB;
		private System.Windows.Forms.CheckBox fio5StateCB;
		private System.Windows.Forms.TextBox dac0Box;
		private System.Windows.Forms.Label ain3lbl;
		private System.Windows.Forms.Panel panel5;
		private System.Windows.Forms.Label label7;
		private System.Windows.Forms.Button startButton;
		private System.Windows.Forms.CheckBox hzCB;
		private System.Windows.Forms.TextBox intervalBox;
		private System.Windows.Forms.Label count0lbl;

		//ain0-3 voltage
		private double ain0;
		private double ain1;
		private double ain2;
		private double ain3;
		
		//fio4-5 state
		private bool fio4State;	
		private bool fio5State;
		
		private UInt32 count0;				//Counter0 count
		private UInt32 lastCount0;			//The previous counter count
		private string error;				//error
		private double dac0;				//dac0
		private bool updateFIODAC;			//update dac0, fio4, and fio5?
		private bool openU3;				//open u3 device?
		private bool exitApp;				//exit application?
		private U3 u3;						//U3 device object

		private StreamWriter sWriter;		//Writes to a log file
		private long intervalTickCount;		//The tick count at the start of each backround thread loop iteration.
		//Used to keep track of the time elapsed.

		private UInt32 lastU3SystemTimer;	//The previous U3 system timer value
		private UInt32 U3SystemTimer;		//The current U3 system timer value

		private long tickCount;				//The tick count after a Feedback call
		private long startingTickCount;		//The tick count that will be written to the file (TimeStamp)
		private long tickCountOverlap;		//The number of times the windows TickCount has started over and passed startingTickCount (to handle the 32bit signed int wrap around)
		private bool logging;				//Indicates whether logging to a file should be performed
		private bool firstLog;				//Indicates that this is the first time logging to this file
		private bool resetCounter;			//Indicates whether the counter should be reset or not
		private long loggingTimeTrack;		//The time elapsed in ms since the last log file write
		private long displayTimeTrack;		//The time elapsed in ms since the last GUI display update
		private const long displayTimeInterval = 1000;	//How often the GUI display should update (in ms)
		private int loggingTimeInterval;	//How often the log file should be written to
		private const int backgroundTimeInterval = 10;	//How often the background thread executes (in ms)


		public u3Logger()
		{
			//
			// Required for Windows Form Designer support
			//
			InitializeComponent();

			myInit();
			fio4State = false;
			fio5State = false;
			dac0 = 0;
			error = "";
			errorBox.Text = "";
			exitApp = false;
			sWriter = null;
			tickCount = 0;
			logging = false;
			resetCounter = false;
			firstLog = false;

			loggingTimeInterval = 1000;
			intervalTickCount = U3.getTickCount();

			Thread updateThread = new Thread(new ThreadStart(backgroundThread));
			updateThread.Start();
		}

		/// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			base.Dispose( disposing );
		}


		#region Windows Form Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			this.mainMenu1 = new System.Windows.Forms.MainMenu();
			this.label1 = new System.Windows.Forms.Label();
			this.label2 = new System.Windows.Forms.Label();
			this.label3 = new System.Windows.Forms.Label();
			this.label4 = new System.Windows.Forms.Label();
			this.label5 = new System.Windows.Forms.Label();
			this.label6 = new System.Windows.Forms.Label();
			this.label8 = new System.Windows.Forms.Label();
			this.label9 = new System.Windows.Forms.Label();
			this.errorBox = new System.Windows.Forms.TextBox();
			this.ain0lbl = new System.Windows.Forms.Label();
			this.ain1lbl = new System.Windows.Forms.Label();
			this.ain2lbl = new System.Windows.Forms.Label();
			this.ain3lbl = new System.Windows.Forms.Label();
			this.count0lbl = new System.Windows.Forms.Label();
			this.panel1 = new System.Windows.Forms.Panel();
			this.label13 = new System.Windows.Forms.Label();
			this.label10 = new System.Windows.Forms.Label();
			this.panel2 = new System.Windows.Forms.Panel();
			this.fio4StateCB = new System.Windows.Forms.CheckBox();
			this.fio4DirCB = new System.Windows.Forms.CheckBox();
			this.label14 = new System.Windows.Forms.Label();
			this.label12 = new System.Windows.Forms.Label();
			this.label11 = new System.Windows.Forms.Label();
			this.fio5DirCB = new System.Windows.Forms.CheckBox();
			this.fio5StateCB = new System.Windows.Forms.CheckBox();
			this.panel3 = new System.Windows.Forms.Panel();
			this.hzCB = new System.Windows.Forms.CheckBox();
			this.panel4 = new System.Windows.Forms.Panel();
			this.dac0Box = new System.Windows.Forms.TextBox();
			this.label16 = new System.Windows.Forms.Label();
			this.panel5 = new System.Windows.Forms.Panel();
			this.startButton = new System.Windows.Forms.Button();
			this.intervalBox = new System.Windows.Forms.TextBox();
			this.label7 = new System.Windows.Forms.Label();
			// 
			// label1
			// 
			this.label1.Font = new System.Drawing.Font("Tahoma", 9F, System.Drawing.FontStyle.Bold);
			this.label1.Location = new System.Drawing.Point(16, 32);
			this.label1.Size = new System.Drawing.Size(16, 16);
			this.label1.Text = "0";
			// 
			// label2
			// 
			this.label2.Font = new System.Drawing.Font("Tahoma", 9F, System.Drawing.FontStyle.Bold);
			this.label2.Location = new System.Drawing.Point(16, 56);
			this.label2.Size = new System.Drawing.Size(16, 16);
			this.label2.Text = "1";
			// 
			// label3
			// 
			this.label3.Font = new System.Drawing.Font("Tahoma", 9F, System.Drawing.FontStyle.Bold);
			this.label3.Location = new System.Drawing.Point(16, 80);
			this.label3.Size = new System.Drawing.Size(16, 16);
			this.label3.Text = "2";
			// 
			// label4
			// 
			this.label4.Font = new System.Drawing.Font("Tahoma", 9F, System.Drawing.FontStyle.Bold);
			this.label4.Location = new System.Drawing.Point(16, 104);
			this.label4.Size = new System.Drawing.Size(16, 16);
			this.label4.Text = "3";
			// 
			// label5
			// 
			this.label5.Font = new System.Drawing.Font("Tahoma", 9F, System.Drawing.FontStyle.Bold);
			this.label5.Location = new System.Drawing.Point(16, 32);
			this.label5.Size = new System.Drawing.Size(16, 20);
			this.label5.Text = "4";
			// 
			// label6
			// 
			this.label6.Font = new System.Drawing.Font("Tahoma", 9F, System.Drawing.FontStyle.Bold);
			this.label6.Location = new System.Drawing.Point(16, 56);
			this.label6.Size = new System.Drawing.Size(16, 20);
			this.label6.Text = "5";
			// 
			// label8
			// 
			this.label8.Font = new System.Drawing.Font("Tahoma", 9F, System.Drawing.FontStyle.Bold);
			this.label8.Location = new System.Drawing.Point(8, 8);
			this.label8.Size = new System.Drawing.Size(80, 16);
			this.label8.Text = "Cnt0(FIO7):";
			// 
			// label9
			// 
			this.label9.Font = new System.Drawing.Font("Tahoma", 9F, System.Drawing.FontStyle.Bold);
			this.label9.Location = new System.Drawing.Point(8, 240);
			this.label9.Size = new System.Drawing.Size(56, 16);
			this.label9.Text = "Error";
			// 
			// errorBox
			// 
			this.errorBox.BackColor = System.Drawing.Color.FromArgb(((System.Byte)(224)), ((System.Byte)(224)), ((System.Byte)(224)));
			this.errorBox.Font = new System.Drawing.Font("Tahoma", 8.25F, System.Drawing.FontStyle.Regular);
			this.errorBox.Location = new System.Drawing.Point(56, 232);
			this.errorBox.Multiline = true;
			this.errorBox.ReadOnly = true;
			this.errorBox.ScrollBars = System.Windows.Forms.ScrollBars.Vertical;
			this.errorBox.Size = new System.Drawing.Size(168, 32);
			this.errorBox.Text = "";
			// 
			// ain0lbl
			// 
			this.ain0lbl.Location = new System.Drawing.Point(48, 32);
			this.ain0lbl.Size = new System.Drawing.Size(48, 16);
			this.ain0lbl.Text = "99999.999";
			// 
			// ain1lbl
			// 
			this.ain1lbl.Location = new System.Drawing.Point(48, 56);
			this.ain1lbl.Size = new System.Drawing.Size(48, 16);
			this.ain1lbl.Text = "99999.999";
			// 
			// ain2lbl
			// 
			this.ain2lbl.Location = new System.Drawing.Point(48, 80);
			this.ain2lbl.Size = new System.Drawing.Size(48, 16);
			this.ain2lbl.Text = "99999.999";
			// 
			// ain3lbl
			// 
			this.ain3lbl.Location = new System.Drawing.Point(48, 104);
			this.ain3lbl.Size = new System.Drawing.Size(48, 16);
			this.ain3lbl.Text = "99999.999";
			// 
			// count0lbl
			// 
			this.count0lbl.Location = new System.Drawing.Point(96, 8);
			this.count0lbl.Size = new System.Drawing.Size(76, 16);
			this.count0lbl.Text = "4294967296";
			// 
			// panel1
			// 
			this.panel1.BackColor = System.Drawing.Color.White;
			this.panel1.Controls.Add(this.label13);
			this.panel1.Controls.Add(this.label10);
			this.panel1.Controls.Add(this.ain1lbl);
			this.panel1.Controls.Add(this.label2);
			this.panel1.Controls.Add(this.label4);
			this.panel1.Controls.Add(this.label1);
			this.panel1.Controls.Add(this.label3);
			this.panel1.Controls.Add(this.ain3lbl);
			this.panel1.Controls.Add(this.ain0lbl);
			this.panel1.Controls.Add(this.ain2lbl);
			this.panel1.Location = new System.Drawing.Point(8, 56);
			this.panel1.Size = new System.Drawing.Size(104, 128);
			// 
			// label13
			// 
			this.label13.Font = new System.Drawing.Font("Tahoma", 9F, ((System.Drawing.FontStyle)((System.Drawing.FontStyle.Bold | System.Drawing.FontStyle.Underline))));
			this.label13.Location = new System.Drawing.Point(8, 8);
			this.label13.Size = new System.Drawing.Size(32, 16);
			this.label13.Text = "AIN";
			// 
			// label10
			// 
			this.label10.Font = new System.Drawing.Font("Tahoma", 9F, ((System.Drawing.FontStyle)((System.Drawing.FontStyle.Bold | System.Drawing.FontStyle.Underline))));
			this.label10.Location = new System.Drawing.Point(40, 8);
			this.label10.Size = new System.Drawing.Size(56, 16);
			this.label10.Text = "Voltage";
			// 
			// panel2
			// 
			this.panel2.BackColor = System.Drawing.Color.White;
			this.panel2.Controls.Add(this.fio4StateCB);
			this.panel2.Controls.Add(this.fio4DirCB);
			this.panel2.Controls.Add(this.label14);
			this.panel2.Controls.Add(this.label12);
			this.panel2.Controls.Add(this.label11);
			this.panel2.Controls.Add(this.label6);
			this.panel2.Controls.Add(this.label5);
			this.panel2.Controls.Add(this.fio5DirCB);
			this.panel2.Controls.Add(this.fio5StateCB);
			this.panel2.Location = new System.Drawing.Point(120, 56);
			this.panel2.Size = new System.Drawing.Size(112, 80);
			// 
			// fio4StateCB
			// 
			this.fio4StateCB.Location = new System.Drawing.Point(78, 32);
			this.fio4StateCB.Size = new System.Drawing.Size(16, 16);
			this.fio4StateCB.Click += new System.EventHandler(this.fio4StateCB_Click);
			// 
			// fio4DirCB
			// 
			this.fio4DirCB.Location = new System.Drawing.Point(46, 32);
			this.fio4DirCB.Size = new System.Drawing.Size(16, 16);
			this.fio4DirCB.Click += new System.EventHandler(this.fioDirCB_Click);
			// 
			// label14
			// 
			this.label14.Font = new System.Drawing.Font("Tahoma", 9F, ((System.Drawing.FontStyle)((System.Drawing.FontStyle.Bold | System.Drawing.FontStyle.Underline))));
			this.label14.Location = new System.Drawing.Point(8, 8);
			this.label14.Size = new System.Drawing.Size(28, 16);
			this.label14.Text = "FIO";
			// 
			// label12
			// 
			this.label12.Font = new System.Drawing.Font("Tahoma", 9F, ((System.Drawing.FontStyle)((System.Drawing.FontStyle.Bold | System.Drawing.FontStyle.Underline))));
			this.label12.Location = new System.Drawing.Point(64, 8);
			this.label12.Size = new System.Drawing.Size(40, 16);
			this.label12.Text = "State";
			// 
			// label11
			// 
			this.label11.Font = new System.Drawing.Font("Tahoma", 9F, ((System.Drawing.FontStyle)((System.Drawing.FontStyle.Bold | System.Drawing.FontStyle.Underline))));
			this.label11.Location = new System.Drawing.Point(40, 8);
			this.label11.Size = new System.Drawing.Size(24, 16);
			this.label11.Text = "Dir";
			// 
			// fio5DirCB
			// 
			this.fio5DirCB.Location = new System.Drawing.Point(46, 56);
			this.fio5DirCB.Size = new System.Drawing.Size(16, 16);
			this.fio5DirCB.Click += new System.EventHandler(this.fioDirCB_Click);
			// 
			// fio5StateCB
			// 
			this.fio5StateCB.Location = new System.Drawing.Point(78, 56);
			this.fio5StateCB.Size = new System.Drawing.Size(16, 16);
			this.fio5StateCB.Click += new System.EventHandler(this.fio5StateCB_Click);
			// 
			// panel3
			// 
			this.panel3.BackColor = System.Drawing.Color.White;
			this.panel3.Controls.Add(this.hzCB);
			this.panel3.Controls.Add(this.count0lbl);
			this.panel3.Controls.Add(this.label8);
			this.panel3.Location = new System.Drawing.Point(8, 192);
			this.panel3.Size = new System.Drawing.Size(224, 32);
			// 
			// hzCB
			// 
			this.hzCB.Font = new System.Drawing.Font("Tahoma", 8.25F, System.Drawing.FontStyle.Bold);
			this.hzCB.Location = new System.Drawing.Point(184, 8);
			this.hzCB.Size = new System.Drawing.Size(40, 20);
			this.hzCB.Text = "Hz";
			this.hzCB.CheckStateChanged += new System.EventHandler(this.hzCB_CheckStateChanged);
			// 
			// panel4
			// 
			this.panel4.BackColor = System.Drawing.Color.White;
			this.panel4.Controls.Add(this.dac0Box);
			this.panel4.Controls.Add(this.label16);
			this.panel4.Location = new System.Drawing.Point(120, 144);
			this.panel4.Size = new System.Drawing.Size(112, 40);
			// 
			// dac0Box
			// 
			this.dac0Box.Location = new System.Drawing.Point(56, 8);
			this.dac0Box.Size = new System.Drawing.Size(48, 22);
			this.dac0Box.Text = "0.000";
			this.dac0Box.LostFocus += new System.EventHandler(this.dac0Box_LostFocus);
			this.dac0Box.GotFocus += new System.EventHandler(this.dac0Box_GotFocus);
			this.dac0Box.Validating += new System.ComponentModel.CancelEventHandler(this.dac0Box_Validating);
			// 
			// label16
			// 
			this.label16.Font = new System.Drawing.Font("Tahoma", 9F, System.Drawing.FontStyle.Bold);
			this.label16.Location = new System.Drawing.Point(8, 8);
			this.label16.Size = new System.Drawing.Size(44, 16);
			this.label16.Text = "DAC0:";
			// 
			// panel5
			// 
			this.panel5.BackColor = System.Drawing.Color.White;
			this.panel5.Controls.Add(this.startButton);
			this.panel5.Controls.Add(this.intervalBox);
			this.panel5.Controls.Add(this.label7);
			this.panel5.Location = new System.Drawing.Point(8, 8);
			this.panel5.Size = new System.Drawing.Size(224, 40);
			// 
			// startButton
			// 
			this.startButton.Font = new System.Drawing.Font("Tahoma", 8.25F, System.Drawing.FontStyle.Bold);
			this.startButton.Location = new System.Drawing.Point(120, 8);
			this.startButton.Size = new System.Drawing.Size(96, 24);
			this.startButton.Text = "Start Logging";
			this.startButton.Click += new System.EventHandler(this.startButton_Click);
			// 
			// intervalBox
			// 
			this.intervalBox.Location = new System.Drawing.Point(64, 8);
			this.intervalBox.Size = new System.Drawing.Size(48, 22);
			this.intervalBox.Text = "1.00";
			this.intervalBox.LostFocus += new System.EventHandler(this.intervalBox_LostFocus);
			this.intervalBox.GotFocus += new System.EventHandler(this.intervalBox_GotFocus);
			this.intervalBox.Validating += new System.ComponentModel.CancelEventHandler(this.intervalBox_Validating);
			// 
			// label7
			// 
			this.label7.Font = new System.Drawing.Font("Tahoma", 8.25F, System.Drawing.FontStyle.Bold);
			this.label7.Location = new System.Drawing.Point(8, 8);
			this.label7.Size = new System.Drawing.Size(56, 32);
			this.label7.Text = "Interval (in sec):";
			// 
			// u3Logger
			// 
			this.BackColor = System.Drawing.Color.FromArgb(((System.Byte)(224)), ((System.Byte)(224)), ((System.Byte)(224)));
			this.Controls.Add(this.panel5);
			this.Controls.Add(this.panel4);
			this.Controls.Add(this.panel3);
			this.Controls.Add(this.panel2);
			this.Controls.Add(this.panel1);
			this.Controls.Add(this.errorBox);
			this.Controls.Add(this.label9);
			this.Menu = this.mainMenu1;
			this.MinimizeBox = false;
			this.Text = "U3 Logger";
			this.Click += new System.EventHandler(this.focusClick);
			this.Closing += new System.ComponentModel.CancelEventHandler(this.u3Logging_Closing);

		}
		#endregion


		/// <summary>
		/// The main entry point for the application.
		/// </summary>
		static void Main() 
		{
			Application.Run(new u3Logger());
		}


		private void myInit()
		{
			//Initializing variables
			ain0 = 99999.999;
			ain1 = 99999.999;
			ain2 = 99999.999;
			ain3 = 99999.999;
			count0 = 0;
			lastCount0 = 0;
			updateFIODAC = true;
			openU3 = true;
			this.Invoke(new EventHandler(updateAINLabels));
			this.Invoke(new EventHandler(updateCount0Label));
		}


		/// <summary>
		/// This thread is responsible for U3 operations, file logging and updating
		/// the GUI's diplayed readings.  All this is done in a while loop that executes
		/// every timerTimeInterval milliseconds
		/// </summary>
		private void backgroundThread()
		{
			int err = 0;
			long curTickCount = 0;
			long timeElapsed = 0;

			while(true) 
			{		
				try 
				{
					if(exitApp == true )
					{
						try
						{
							if(openU3 == false)
							{
								configIO_example(false);
								u3.closeUSBConnection();
							}
				
							sWriter.Flush();
							sWriter.Close();
						}
						catch{}

						this.Close();
						return;
					}

					err = 0;

					try
					{
						curTickCount = U3.getTickCount();
						timeElapsed = ((curTickCount < intervalTickCount)?(2147483648 - intervalTickCount + curTickCount):(curTickCount - intervalTickCount));

						intervalTickCount = curTickCount;

						if(logging == true) 
							loggingTimeTrack += timeElapsed;
						displayTimeTrack += timeElapsed;

						if((updateFIODAC || logNow() || displayNow()) && this.Visible) 
						{
							if(displayNow())
							{
								error = "";
								this.Invoke(new EventHandler(updateErrorBox));
							}

							if(openU3 == true)
							{
								//Creating new U3 object and opening first found U3
								u3 = new U3();
								if(u3.openUSBConnection(-1) < 0)
									throw new Exception((error = u3.errorText));

								openU3 = false;

								//Getting the opened U3's calibration information
								if(u3.getCalibrationInfo() < 0)
									throw new Exception((error = u3.errorText));

								//configure timer clock
								if( (err = configTimerClock_example()) < 0)
								{
									if(err != -100)
										error = "ConfigTimerClock " + getError(err);
									throw new Exception();
								}

								//Configure analog, digital, counters IOs
								if( (err = configIO_example(true)) < 0)
								{
									if(err != -100)
										error = "ConfigIO " + getError(err);
									throw new Exception();
								}		
				
								//Configure timer0
								if( (err = timer0Config_example()) < 0)
								{
									if(err != -100)
										error = "Timer0Config " + getError(err);
									throw new Exception();
								}		
							}
							
							if( (err = feedbacklog_example()) < 0)
							{
								if(err != -100)
									error = "Feedback " + getError(err);
								throw new Exception();
							}
						}
					}
					catch(Exception e)
					{
						if(displayNow())
						{
							this.Invoke(new EventHandler(updateErrorBox));
							while(displayTimeTrack >= displayTimeInterval)
								displayTimeTrack -= displayTimeInterval;
						}

						if(openU3 == false)
							u3.closeUSBConnection();
		
						myInit();
					}
			
					Thread.Sleep(backgroundTimeInterval);
				}
				catch(Exception e)
				{
					e.ToString();
					this.Invoke(new EventHandler(updateErrorBox));
				}				
			}
			exitApp = true;
			this.Close();
		}


		/// <summary>
		///  Configures analog, digital, timers, counters IOs with the ConfigIO low-level function.  
		///  If enabled set as true, Timer0 and Counter0 will be enabled with a TimerCounterPinOffset 
		///  of 6, and FIO0 - 3 wil be set as analog and FIO 4-5 will be set as digital.
		/// </summary>
		/// <param name="enable">Pass true enable counter and set the FIO lines as needed.  Pass false to disable all timers and counters and set FIO lines to analog.</param>
		/// <returns>A negative error value on error, 0 on success.</returns>
		private int configIO_example(bool enable)
		{
			byte []sendBuff = new byte[12];
			byte []recBuff = new byte[12];
			ushort checksumTotal;
			byte timerCounterConfig;
			byte fioAnalog;

			if(enable == true)
			{
				timerCounterConfig = 101;	//Enabling Timer0 (bit 0), Counter0 (bit 2), 
											//and setting TimerCounterPinOffset (bits 4-7) to 6
				fioAnalog = 15;				//Setting FIO0-FIO3 (bits 0-3) to analog input  				 
			}
			else
			{
				timerCounterConfig = 64;	//Disabling timers (bits 0 and 1) and Counters, setting
											//TimerCounterPinOffset to 4 (bits 4-7)
				fioAnalog = 255;			//Setting all FIOs to analog 
			}			

			sendBuff[1] = (byte)(0xF8);		//Command byte
			sendBuff[2] = (byte)(0x03);		//Number of data words
			sendBuff[3] = (byte)(0x0B);		//Extended command number

			sendBuff[6] = 7;				//Writemask : Setting writemask for timerCounterConfig (bit 0), 
											//DAC1Enable (bit 1) and FIOAnalog (bit 2)
  
			sendBuff[7] = 0;				//Reserved
			sendBuff[8] = timerCounterConfig;  //TimerCounterConfig
			sendBuff[9] = 0;				//DAC1Enable : disabling (ignored in hardware versions >= 1.30)
			sendBuff[10] = fioAnalog;		//FIOAnalog
			sendBuff[11] = 0;				//EIOAnalog : Not setting anything                       
			U3.extendedChecksum(ref sendBuff,12);

			if( U3.LJUSB_BulkWrite(u3.handle, U3.U3_PIPE_EP1_OUT, sendBuff, 12) < 12)
				return -1;;
			
			if( U3.LJUSB_BulkRead(u3.handle, U3.U3_PIPE_EP1_IN, recBuff, 12) < 12)
				return -2;

			checksumTotal = U3.extendedChecksum16(recBuff, 12);
			if( (byte)((checksumTotal/256) & 0xff) != recBuff[5]
				|| (byte)(checksumTotal & 0xff) != recBuff[4])
				return -4;
			
			if( U3.extendedChecksum8(recBuff) != recBuff[0])
				return -3;

			if( recBuff[1] != (byte)(0xF8) || recBuff[2] != (byte)(0x03) || recBuff[3] != (byte)(0x0B) )
				return -6;

			if( recBuff[6] != 0)
			{
				u3.errorcode = recBuff[6];
				return -200;
			}
  
			if( recBuff[8] != timerCounterConfig)
			{
				error = "ConfigIO error : TimerCounterConfig did not get set correctly";
				return -100;
			}

			if( recBuff[10] != fioAnalog)
			{
				error = "ConfigIO error : FIOAnalog did not set correctly";
				return -100;
			}

			return 0;  
		}


		/// <summary>
		/// Configures the TimerClockBase to 48 MHz.
		/// </summary>
		/// <returns>Negative number on error, 0 on success.</returns>
		private int configTimerClock_example()
		{
			byte []sendBuff = new byte[10];
			byte []recBuff = new byte[10];
			ushort checksumTotal;

			sendBuff[1] = (byte)(0xF8);	//Command byte
			sendBuff[2] = (byte)(0x02);	//Number of data words
			sendBuff[3] = (byte)(0x0A);	//Extended command number

			sendBuff[6] = 0;			//Reserved
			sendBuff[7] = 0;			//Reserved

			sendBuff[8] = 130;			//TimerClockConfig : Configuring the clock (bit 7)
										//and setting the TimerClockBase (bits 0-2) to 48 MHz
			sendBuff[9] = 1;			//TimerClockDivisor : not using
			U3.extendedChecksum(ref sendBuff, 10);

			if( U3.LJUSB_BulkWrite(u3.handle, U3.U3_PIPE_EP1_OUT, sendBuff, 10) < 10)
				return -1;
		
			if( U3.LJUSB_BulkRead(u3.handle, U3.U3_PIPE_EP1_IN, recBuff, 10) < 10)
				return -2;

			checksumTotal = U3.extendedChecksum16(recBuff, 10);
			if( (byte)((checksumTotal >> 8) & 0xff) != recBuff[5]
				|| (byte)(checksumTotal & 0xff) != recBuff[4])
				return -4;
			
			if( U3.extendedChecksum8(recBuff) != recBuff[0])
				return -3;
			

			if( recBuff[1] != (byte)(0xF8) || recBuff[2] != (byte)(0x02) || recBuff[3] != (byte)(0x0A) )
				return -6;

			if( recBuff[6] != 0)
			{
				u3.errorcode = recBuff[6];
				return -200;
			}

			return 0;
		}


		/// <summary>
		///  Configures the timer 0 mode and its value using the Feedback low-level function.  
		///  The mode will be set to a system timer low. 
		/// </summary>
		/// <returns>A negative error value on error, 0 on success.</returns>
		private int timer0Config_example()
		{
			byte []sendBuff = new byte[12];
			byte []recBuff = new byte[10];
			ulong checksumTotal;
			uint recChar = 0;

			sendBuff[1] = (byte)(0xF8);	//Command byte
			sendBuff[2] = 3;			//Number of data words (.5 word for echo, 2 words for IOTypes 
										//and data, and .5 word for the extra byte)   
			sendBuff[3] = (byte)(0x00);	//Extended command number

			sendBuff[6] = 0;			//Echo

			sendBuff[7] = 43;			//IOType is Timer0Config
			sendBuff[8] = 10;			//TimerMode is System Timer Low Read (mode 10)
			sendBuff[9] = 0;			//Value LSB
			sendBuff[10] = 0;			//Value MSB

			sendBuff[11] = 0;

			U3.extendedChecksum(ref sendBuff, 12);

			if( U3.LJUSB_BulkWrite(u3.handle, U3.U3_PIPE_EP1_OUT, sendBuff, 12) < 12)
				return -1;
  
			if((recChar = U3.LJUSB_BulkRead(u3.handle, U3.U3_PIPE_EP1_IN, recBuff, 10)) < 9)
				return -2;
			  
			checksumTotal = U3.extendedChecksum16(recBuff, Convert.ToInt32(recChar));
			if( (byte)((checksumTotal/256) & 0xff) != recBuff[5]
				|| (byte)(checksumTotal & 0xff) != recBuff[4])
				return -4;
			
			if( U3.extendedChecksum8(recBuff) != recBuff[0])
				return -3;
			
			if( recBuff[1] != (byte)(0xF8) || recBuff[3] != (byte)(0x00) )
				return -6;

			if( recBuff[6] != 0)
			{
				error = "Feedback error: received errorcode " + recBuff[6] + " in frame " + recBuff[7];
				return -100;
			}

			if(recBuff[2] != 2)
				return -6;
  
			return 0;
		}


		/// <summary>
		/// Reads the voltages of AIN0-4, states of FIO5-6 and the count of counter0 using the Feedback
		/// low-level function.  Also, the direction and states of FIO5-6 and the voltage of DAC0 will
		/// be updated.
		/// </summary>
		/// <returns>A negative error value on error, 0 on success.</returns>
		private int feedbacklog_example()
		{
			ushort checksumTotal;
			uint recChars = 0;
			double voltage = 0;
			byte maskDirSt = 0;
			long tempTickCount = 0;
			int sendIndexAdjust = 0;
			uint sendBuffSize = 42;
			byte sendDataWords = 18;		//Number of data words (.5 word for echo, and 17.5 words
											//for IOTypes and data)

			//For hardware version >= 1.30, adjust Feedback command for DAC0 (16-bit) instead of DAC0 (8-bit)
			if(u3.calibrationInfo.hardwareVersion >= 1.30)
			{
				sendIndexAdjust = 1;
				sendBuffSize = 44;
				sendDataWords = 19;			//Number of data words (.5 word for echo, 18 words for
											//IOTypes and data, and .5 word for extra byte)
			}

			byte []sendBuff = new byte[sendBuffSize];
			byte []recBuff = new byte[28];

			/* Reading the AIN0-4, FIO5-6, timer0 and counter0 and updating the direction 
			 * and states of FIO5-6 and the voltage of DAC0 */
			sendBuff[1] = (byte)(0xF8);		//Command byte
			sendBuff[2] = sendDataWords;	//Number of data words
			sendBuff[3] = (byte)(0x00);		//Extended command number

			sendBuff[6] = 0;				//Echo

			sendBuff[7] = 29;				//IOType is PortDirWrite
			sendBuff[8] = 48;				//WriteMask - setting bits 4 and 5 (FIO4 and 5)
			sendBuff[9] = 0;				//WriteMask
			sendBuff[10] = 0;				//WriteMask

			if(fio4DirCB.Checked == true)
				maskDirSt += 16;			//setting (FIO4) as output
			if(fio5DirCB.Checked == true)
				maskDirSt += 32;			//setting (FIO5) of output
				
			sendBuff[11] = maskDirSt;		//Direction - Setting bits 4 and 5 (FIO4 and 5)
			sendBuff[12] = 0;				//Direction
			sendBuff[13] = 0;				//Direction
				
			sendBuff[14] = 27;				//IOType is PortStateWrite
			sendBuff[15] = maskDirSt;		//WriteMask
			sendBuff[16] = 0;				//WriteMask
			sendBuff[17] = 0;				//WriteMask

			maskDirSt = 0;
			if(fio4StateCB.Checked == true && fio4DirCB.Checked == true)
				maskDirSt += 16;			//setting (FIO4) as high
				
			if(fio5StateCB.Checked == true && fio5DirCB.Checked == true)
				maskDirSt += 32;			//setting (FIO5) of high
				
			sendBuff[18] = maskDirSt;		//State
			sendBuff[19] = 0;				//State
			sendBuff[20] = 0;				//State

			if(u3.calibrationInfo.hardwareVersion < 1.30)
			{
				sendBuff[21] = 34;			//IOType is DAC0 (8-bit)

				//Value of the DAC0 volts (in 8 bit binary form)
				u3.analogToCalibratedBinary8BitVoltage(0, dac0, ref sendBuff[22]);
			}
			else
			{
				ushort binVoltage16 = 0;

				sendBuff[21] = 38;			//IOType is DAC0 (16-bit)

				//Value of the DAC0 volts (in 16 bit binary form)
				u3.analogToCalibratedBinary16BitVoltage(0, dac0, ref binVoltage16);

				sendBuff[22] = (byte)(binVoltage16&255);			//Value LSB
				sendBuff[23] = (byte)((binVoltage16&65280)/256);	//Value MSB

				sendBuff[43] = 0;		//Extra byte
			}

			sendBuff[23+sendIndexAdjust] = 1;		//IOType is AIN
			sendBuff[24+sendIndexAdjust] = 0;		//Positive channel is 0
			sendBuff[25+sendIndexAdjust] = 31;		//Negative channel is SE (31)
  
			sendBuff[26+sendIndexAdjust] = 1;		//IOType is AIN
			sendBuff[27+sendIndexAdjust] = 1;		//Positive channel is 1
			sendBuff[28+sendIndexAdjust] = 31;		//Negative channel is SE (31)
  
			sendBuff[29+sendIndexAdjust] = 1;		//IOType is AIN
			sendBuff[30+sendIndexAdjust] = 2;		//Positive channel is 2
			sendBuff[31+sendIndexAdjust] = 31;		//Negative channel is SE (31)

			sendBuff[32+sendIndexAdjust] = 1;		//IOType is AIN
			sendBuff[33+sendIndexAdjust] = 3;		//Positive channel is 3
			sendBuff[34+sendIndexAdjust] = 31;		//Negative channel is SE (31)

			sendBuff[35+sendIndexAdjust] = 26;		//IOType is PortStateRead

			sendBuff[36+sendIndexAdjust] = 54;		//IOType is Counter0
			sendBuff[37+sendIndexAdjust] = Convert.ToByte(resetCounter);   //Reset (bit 0)

			sendBuff[38+sendIndexAdjust] = 42;		//IOType is Timer0
			sendBuff[39+sendIndexAdjust] = 1;		//UpdateReset
			sendBuff[40+sendIndexAdjust] = 0;		//Value LSB
			sendBuff[41+sendIndexAdjust] = 128;		//Value MSB

			U3.extendedChecksum(ref sendBuff, (int)sendBuffSize);

			if( U3.LJUSB_BulkWrite(u3.handle, U3.U3_PIPE_EP1_OUT, sendBuff, sendBuffSize) < sendBuffSize)
				return -1;
  
			recChars = U3.LJUSB_BulkRead(u3.handle, U3.U3_PIPE_EP1_IN, recBuff, 28);
			
			tempTickCount = U3.getTickCount();
			lastCount0 = count0;

			if(firstLog && logNow() && logging)
			{
				firstLog = false;
				startingTickCount = tempTickCount;
				tickCount = tempTickCount;
			}

			if(logging) {
				if(tempTickCount >= startingTickCount && tickCount < startingTickCount)
					tickCountOverlap++;

				tickCount = tempTickCount;
			}

			checksumTotal = U3.extendedChecksum16(recBuff, Convert.ToInt32(recChars));

			if( (byte)((checksumTotal/256) & 0xff) != recBuff[5]
				|| (byte)(checksumTotal & 0xff) != recBuff[4])
				return -4;
			
			if( U3.extendedChecksum8(recBuff) != recBuff[0])
				return -3;
			
			if( recBuff[1] != (byte)(0xF8) || recBuff[3] != (byte)(0x00) )
				return -6;
			
			if( recBuff[6] != 0)
			{
				error = "Feedback error: received errorcode " + recBuff[6] + " in frame " + recBuff[7];
				switch(recBuff[7])
				{
					case 1: error += "(AIN0)\n"; break;
					case 2: error += "(AIN1)\n"; break;
					case 3: error += "(AIN2)\n"; break;
					case 4: error += "(AIN3)\n"; break;					
					case 5: error += "(PortStateRead)\n"; break;
					case 6: error += "(Counter0)\n"; break;
					case 7: error += "(Timer0)\n"; break;
					default: error += "(Unknown)\n"; break;
				}
				return -100;
			}
			
			if( recBuff[2] != 11)
				return -6;

			updateFIODAC = false;

			//Converting AIN readings from 16 bit value to a voltage
			if(u3.calibrationInfo.hardwareVersion >= 1.30)
			{
				u3.binaryToCalibratedAnalogVoltage_hw130(0, 31, (ushort)(recBuff[9] + recBuff[10]*256), ref voltage); 
				ain0 = voltage;
				u3.binaryToCalibratedAnalogVoltage_hw130(1, 31, (ushort)(recBuff[11] + recBuff[12]*256), ref voltage);     
				ain1 = voltage;
				u3.binaryToCalibratedAnalogVoltage_hw130(2, 31, (ushort)(recBuff[13] + recBuff[14]*256), ref voltage); 
				ain2 = voltage;
				u3.binaryToCalibratedAnalogVoltage_hw130(3, 31, (ushort)(recBuff[15] + recBuff[16]*256), ref voltage);
				ain3 = voltage;
			}
			else
			{
				u3.binaryToCalibratedAnalogVoltage(false, 31, (ushort)(recBuff[9] + recBuff[10]*256), ref voltage); 
				ain0 = voltage;
				u3.binaryToCalibratedAnalogVoltage(false, 31, (ushort)(recBuff[11] + recBuff[12]*256), ref voltage);     
				ain1 = voltage;
				u3.binaryToCalibratedAnalogVoltage(false, 31, (ushort)(recBuff[13] + recBuff[14]*256), ref voltage); 
				ain2 = voltage;
				u3.binaryToCalibratedAnalogVoltage(false, 31, (ushort)(recBuff[15] + recBuff[16]*256), ref voltage);     
				ain3 = voltage;
			}

			//Checking if FIO4 (bit 4) state is high (1) or low (0)
			if(fio4DirCB.Checked == false)
			{
				if( (recBuff[17]&16) == 16 )
					fio4State = true;
				else
					fio4State = false;
			}

			//Checking if FIO5 (bit 5) state is high (1) or low (0)
			if(fio5DirCB.Checked == false)
			{
				if( (recBuff[17]&32) == 32 )
					fio5State = true;
				else
					fio5State = false;
			}

			//Getting the count of Counter0
			count0 = (uint)recBuff[20] + (uint)recBuff[21]*256 + (uint)recBuff[22]*65536 + (uint)recBuff[23]*16777216;

			lastU3SystemTimer = U3SystemTimer;
			U3SystemTimer = (uint)recBuff[24] + (uint)recBuff[25]*256 + (uint)recBuff[26]*65536 + (uint)recBuff[27]*16777216;

			if(logNow() && logging)
			{
				long tickCountElapsed = 0;

				if (tickCount < startingTickCount) 
				{
					tickCountElapsed = 2147483648 - startingTickCount + tickCount + tickCountOverlap*2147483648;
				}
				else
					tickCountElapsed = tickCount - startingTickCount + tickCountOverlap*2147483648;


				//Writing to file
				sWriter.WriteLine( tickCountElapsed + ";" + string.Format("{0:f3}", ain0) + ";" + string.Format("{0:f3}", ain1) + ";" + string.Format("{0:f3}", ain2) + ";" + string.Format("{0:f3}", ain3) + ";" + Convert.ToInt32(fio4DirCB.Checked) + ";" + Convert.ToInt32(fio4State) + ";" + Convert.ToInt32(fio5DirCB.Checked) + ";" + Convert.ToInt32(fio5State) + ";" + getCounter0Value().ToString());
				
				if(loggingTimeInterval == 0)
					loggingTimeTrack = 0;
				else
				{
					while(loggingTimeTrack >= loggingTimeInterval)
						loggingTimeTrack -= loggingTimeInterval;
				}
			}

			if(displayNow()) 
			{
				//Updating the values and check boxes of the application
				this.Invoke(new EventHandler(updateAINLabels));
				this.Invoke(new EventHandler(updateFIOStateCB));
				this.Invoke(new EventHandler(updateCount0Label));
				while(displayTimeTrack >= displayTimeInterval)
					displayTimeTrack -= displayTimeInterval;
			}

			if(resetCounter == true) 
			{
				resetCounter = false;
				count0 = 0;
			}
			return 0;
		}


		private uint getCounter0Value() 
		{
			if(hzCB.Checked == true) 
			{
				double countDiff = 0;
				double timeDiff = 0;
				if (count0 < lastCount0)
					countDiff = 4294967296 - lastCount0 + count0;
				else
					countDiff = count0 - lastCount0;
				
				if (U3SystemTimer < lastU3SystemTimer)
					timeDiff = 4294967296 - lastU3SystemTimer + U3SystemTimer;
				else
					timeDiff = U3SystemTimer - lastU3SystemTimer;

				//converting U3 system timer value difference to secs
				timeDiff = timeDiff * 0.00000025;

				if(timeDiff == 0)
					return 0;
				else
					return Convert.ToUInt32(countDiff/timeDiff); 
			}
			else 
				return count0;
		}


		private bool displayNow() 
		{
			if(displayTimeTrack >= displayTimeInterval) 
				return true;
			else return false;
		}


		private bool logNow() 
		{
			if(loggingTimeTrack >= loggingTimeInterval) 
				return true;
			else return false;

		}


		/// <summary>
		/// Gets the error message for the application (Does not translate U3 errorcodes)
		/// </summary>
		/// <param name="error">The error number.</param>
		/// <returns>The error message in string form.</returns>
		private string getError(int error)
		{
			switch(error)
			{
				case 0: return("");  //no errors
				case -1: return("error : send failed");
				case -2: return("error : receive failed");
				case -3: return("error : received bad checksum8");
				case -4: return("error : received bad checksum16");
				case -5: return("error : received bad checksum");
				case -6: return("error : received wrong command bytes");
				case -7: return("error : received buffer has wrong IOType");
				case -200: return("error : received error code " + u3.errorcode + " from the response"); 
				default: return("error");
			}
		}


		private void updateErrorBox(object sender, System.EventArgs e)
		{
			errorBox.Text = error;
		}


		private void updateAINLabels(object sender, System.EventArgs e)
		{
			ain0lbl.Text = string.Format("{0:f3}", ain0);
			ain1lbl.Text = string.Format("{0:f3}", ain1);
			ain2lbl.Text = string.Format("{0:f3}", ain2);
			ain3lbl.Text = string.Format("{0:f3}", ain3);
		}


		private void updateFIOStateCB(object sender, System.EventArgs e)
		{
			fio4StateCB.Checked = fio4State;
			fio5StateCB.Checked = fio5State;		
		}


		private void updateCount0Label(object sender, System.EventArgs e)
		{
			count0lbl.Text = getCounter0Value().ToString();
		}


		private void dac0Box_Validating(object sender, System.ComponentModel.CancelEventArgs e)
		{
			double dac0Conv = 0;
			try
			{
				if((dac0Conv = Convert.ToDouble(dac0Box.Text)) != dac0)
				{
					if(dac0Conv > 5.0 || dac0Conv < 0.0)
					{
						dac0Box.Text = string.Format("{0:f3}", dac0);
					}
					else
					{
						dac0 = dac0Conv;
						dac0Box.Text = string.Format("{0:f3}", dac0);
					}
				}
			}
			catch
			{
				dac0Box.Text = string.Format("{0:f3}", dac0);
			}
			
			updateFIODAC = true;
		}
		

		private void focusClick(object sender, System.EventArgs e)
		{
			this.Focus();
		}


		private void u3Logging_Closing(object sender, System.ComponentModel.CancelEventArgs e)
		{
			if(exitApp == false)
			{
				e.Cancel = true;
				exitApp = true;
			}
			else
				e.Cancel = false;
			//Thread.Sleep(1000);
		}


		private void fioDirCB_Click(object sender, System.EventArgs e)
		{
			updateFIODAC = true;
		}


		private void fio4StateCB_Click(object sender, System.EventArgs e)
		{
			
			if(fio4StateCB.Checked == true)
				fio4State = true;
			else
				fio4State = false;
			updateFIODAC = true;
		}


		private void fio5StateCB_Click(object sender, System.EventArgs e)
		{
			if(fio5StateCB.Checked == true)
				fio5State = true;
			else
				fio5State = false;
			updateFIODAC = true;
		}


		private void intervalBox_Validating(object sender, System.ComponentModel.CancelEventArgs e)
		{
			try
			{
				double dIntervalConv = System.Math.Round((Convert.ToDouble(intervalBox.Text)), 2);
				Int32 intervalConv = Convert.ToInt32(dIntervalConv*1000);
				if(intervalConv < 0)
					throw new Exception("Error: Interval value must be zero or greater");

				if(intervalConv > 100000)
					throw new Exception("Error: Interval value must be 1000.00 or less");

				loggingTimeInterval = intervalConv;
			}
			catch(Exception ex) {
				error = "Error: " + ex.Message;
				this.Invoke(new EventHandler(updateErrorBox));
			}
			
			intervalBox.Text = string.Format("{0:f2}",((double)loggingTimeInterval/1000.0));
		}


		private void startButton_Click(object sender, System.EventArgs e)
		{
			try 
			{
				if(logging == true)
				{
					logging = false;
					
					Thread.Sleep(backgroundTimeInterval +  200);
					try 
					{
						sWriter.Flush();
						sWriter.Close();
					}
					catch {}

					startButton.Text = "Start Logging";
					intervalBox.Enabled = true;
					intervalBox.ReadOnly = false;
				}
				else
				{
					string filePath = @Path.GetDirectoryName(Assembly.GetExecutingAssembly().GetModules()[0].FullyQualifiedName)+"\\"; //System.IO.Directory.GetCurrentDirectory();
					string fileName = "";
					uint count = 1;
					bool loop = true;

					//In loop to create new file dataX.txt in current directory.
					//Existing files will not be appended to or overwritten.
					while(loop == true) 
					{
						fileName = "data"+count+".txt";
						if(File.Exists(filePath+fileName) == false) 
						{
							sWriter = File.CreateText(filePath+fileName);
							loop = false;
						}
						count++;
					}

					DateTime date = DateTime.Now;
					sWriter.WriteLine("Date Created: " + date.ToShortDateString() + " " + date.Hour + ":" + date.Minute + ":" + date.Second);
					sWriter.WriteLine("TimeStamp(ms);AIN0;AIN1;AIN2;AIN3;FIO4Dir;FIO4State;FIO5Dir;FIO5State;Counter0");
					loggingTimeTrack = 0;
					
					startButton.Text = "Stop Logging";
					intervalBox.Enabled = false;
					intervalBox.ReadOnly = true;
					tickCountOverlap = 0;
					resetCounter = true;
					updateFIODAC = true;
					logging = true;
					firstLog = true;
				}
			}
			catch(Exception ex)
			{
				try 
				{
					sWriter.Flush();
					sWriter.Close();
				}
				catch(Exception ex2) {}

				logging = false;
				startButton.Text = "Start Logging";
				intervalBox.Enabled = true;
				intervalBox.ReadOnly = false;

				error += "Error: " + ex.Message;
				this.Invoke(new EventHandler(updateErrorBox));
			}
		}


		private void intervalBox_GotFocus(object sender, System.EventArgs e)
		{
			intervalBox.BackColor = System.Drawing.Color.FromArgb(0xFFFFA0);
		}


		private void intervalBox_LostFocus(object sender, System.EventArgs e)
		{
			intervalBox.BackColor = System.Drawing.Color.White;
		}


		private void dac0Box_GotFocus(object sender, System.EventArgs e)
		{
			dac0Box.BackColor = System.Drawing.Color.FromArgb(0xFFFFA0);
		}


		private void dac0Box_LostFocus(object sender, System.EventArgs e)
		{
			dac0Box.BackColor = System.Drawing.Color.White;
		}


		private void hzCB_CheckStateChanged(object sender, System.EventArgs e)
		{
			resetCounter = true;
			updateFIODAC = true;
		}
	}
}
