// labjackusb.c : Defines the entry point for the DLL application.
//

//#include "stdafx.h"
#include <windows.h>
//#include <Usbclient.h>
#include "labjackusb.h"
#include "labjackue9usb.h"
#include "labjacku3usb.h"
#include "labjack.h"

const BOOL LABJACKUSB_DEBUG = FALSE;

#define IOCTL_ABORT_PIPE		CTL_CODE(FILE_DEVICE_UNKNOWN, 0x800, METHOD_BUFFERED, FILE_ANY_ACCESS)
#define IOCTL_GET_PRODUCT_ID	CTL_CODE(FILE_DEVICE_UNKNOWN, 0x801, METHOD_BUFFERED, FILE_ANY_ACCESS)
#define IOCTL_GET_ACTIVE_PATH	CTL_CODE(FILE_DEVICE_UNKNOWN, 0x802, METHOD_BUFFERED, FILE_ANY_ACCESS)


BOOL
DllEntry( HANDLE hDllHandle, DWORD dwReason, LPVOID) 
{
    switch (dwReason) 
	{
        case DLL_PROCESS_ATTACH:
            break;

        case DLL_PROCESS_DETACH:
			break;
        default:
            break;
    }
    return TRUE ;
}

void
WriteToErrorLog(CHAR message[])
{
	if(LABJACKUSB_DEBUG == TRUE)
	{
		HANDLE logHandle = NULL;
		DWORD bytesWritten;
		CHAR time[32];
		CHAR date[32];
		WCHAR wTime[32];
		WCHAR wDate[32];	
		DWORD tLength;
		DWORD dLength;
		CHAR endLine[] = "\r\n\0";
		UINT i;

		logHandle = CreateFile(lpsLogName, GENERIC_WRITE, FILE_SHARE_WRITE, NULL, OPEN_ALWAYS, 
			FILE_ATTRIBUTE_NORMAL, NULL);

		SetFilePointer (logHandle, 0, NULL, FILE_END);
	
		if((dLength = GetDateFormat(LOCALE_SYSTEM_DEFAULT, 0, NULL,
			TEXT("MM'/'dd'/'yyyy"), wDate, 32)) != 0)
		{
			if((tLength = GetTimeFormat(LOCALE_SYSTEM_DEFAULT, TIME_FORCE24HOURFORMAT, NULL, 
				TEXT("HH':'mm':'ss"), wTime, 32)) != 0)
			{
				date[0] = '[';

				for(i = 0; i < (dLength - 1); i++)
					date[i + 1] = wDate[i];

				date[dLength] = ',';
				date[dLength + 1] = ' ';
				date[dLength + 2] = '\0';

				WriteFile(logHandle, date, strlen(date), &bytesWritten, NULL);

				for(i = 0; i < (tLength - 1); i++)
					time[i] = wTime[i];

				time[tLength - 1] = ']';
				time[tLength] = ' ';
				time[tLength + 1] = '\0';


				WriteFile(logHandle, time, strlen(time), &bytesWritten, NULL);
	
			}
		}
	
		WriteFile(logHandle, message, strlen(message), &bytesWritten, NULL);	
		WriteFile(logHandle, endLine, strlen(endLine), &bytesWritten, NULL);
	
		CloseHandle(logHandle);
	}
}

static void
SetupDriverSettingsUE9(USB_DRIVER_SETTINGS *pDriverSettings)
{
	pDriverSettings->dwCount = sizeof(USB_DRIVER_SETTINGS);

    pDriverSettings->dwVendorId          = LABJACK_VENDOR_ID;
    pDriverSettings->dwProductId         = UE9_PRODUCT_ID;
    pDriverSettings->dwReleaseNumber     = USB_NO_INFO;

    pDriverSettings->dwDeviceClass       = USB_NO_INFO;
    pDriverSettings->dwDeviceSubClass    = USB_NO_INFO;
    pDriverSettings->dwDeviceProtocol    = USB_NO_INFO;

    pDriverSettings->dwInterfaceClass    = USB_DEVICE_CLASS_VENDOR_SPECIFIC;
    pDriverSettings->dwInterfaceSubClass = USB_NO_INFO;
    pDriverSettings->dwInterfaceProtocol = USB_NO_INFO;
}

static void
SetupDriverSettingsU3(USB_DRIVER_SETTINGS *pDriverSettings)
{
	pDriverSettings->dwCount = sizeof(USB_DRIVER_SETTINGS);

    pDriverSettings->dwVendorId          = LABJACK_VENDOR_ID;
    pDriverSettings->dwProductId         = U3_PRODUCT_ID;
    pDriverSettings->dwReleaseNumber     = USB_NO_INFO;

    pDriverSettings->dwDeviceClass       = USB_NO_INFO;
    pDriverSettings->dwDeviceSubClass    = USB_NO_INFO;
    pDriverSettings->dwDeviceProtocol    = USB_NO_INFO;

    pDriverSettings->dwInterfaceClass    = USB_NO_INFO;//USB_DEVICE_CLASS_VENDOR_SPECIFIC;
    pDriverSettings->dwInterfaceSubClass = USB_NO_INFO;
    pDriverSettings->dwInterfaceProtocol = USB_NO_INFO;
}

BOOL 
USBInstallDriver(
    LPCWSTR szDriverLibFile)  // Contains client driver DLL name
{
	BOOL fRet = FALSE;
	HINSTANCE hInst = LoadLibrary(L"USBD.DLL");

	CHAR message[100];
	LONG iSuccess = 0;
	HKEY hKey;
	INT failCount = 0;

	if(LABJACKUSB_DEBUG)
	{
		strcpy(message, "USBInstallDriver\0");
		WriteToErrorLog(message);
	}

	if(hInst)
    {
		LPREGISTER_CLIENT_DRIVER_ID pRegisterId = 
        (LPREGISTER_CLIENT_DRIVER_ID) GetProcAddress(hInst, gcszRegisterClientDriverId);

        LPREGISTER_CLIENT_SETTINGS pRegisterSettings =
        (LPREGISTER_CLIENT_SETTINGS) GetProcAddress(hInst,gcszRegisterClientSettings);

        if (pRegisterId && pRegisterSettings) {
            USB_DRIVER_SETTINGS DriverSettingsUE9;
			USB_DRIVER_SETTINGS DriverSettingsU3;

            SetupDriverSettingsUE9(&DriverSettingsUE9);
			SetupDriverSettingsU3(&DriverSettingsU3);
			
			if(LABJACKUSB_DEBUG)
			{
				strcpy(message, "USBInstallDriver - Registering driverID.\0");
				WriteToErrorLog(message);
			}
            fRet = (*pRegisterId)(gcszDriverIdUE9);

            if(fRet)
            {
				//registering ue9 settings
                fRet = (*pRegisterSettings)(szDriverLibFile,gcszDriverIdUE9, 0, &DriverSettingsUE9);

                if(!fRet)
                {
                    LPUN_REGISTER_CLIENT_DRIVER_ID pUnregisterId =
                        (LPUN_REGISTER_CLIENT_DRIVER_ID) GetProcAddress(hInst,gcszUnRegisterClientDriverId);
                    if (pUnregisterId) 
                        (*pUnregisterId)(gcszDriverIdUE9);

					if(LABJACKUSB_DEBUG)
					{
						strcpy(message, "Error registering UE9 settings.\0");          
						WriteToErrorLog(message);
					}
                }
				else
				{
					fRet = (*pRegisterId)(gcszDriverIdU3);

					if(fRet)
					{
						//registering u3 settings
						fRet = (*pRegisterSettings)(szDriverLibFile,gcszDriverIdU3, 0, &DriverSettingsU3);

						if(!fRet)
						{
							LPUN_REGISTER_CLIENT_DRIVER_ID pUnregisterId =
							    (LPUN_REGISTER_CLIENT_DRIVER_ID) GetProcAddress(hInst,gcszUnRegisterClientDriverId);
							if (pUnregisterId) 
								(*pUnregisterId)(gcszDriverIdU3);

							if(LABJACKUSB_DEBUG)
							{
								strcpy(message, "Error registering U3 settings.\0");          
								WriteToErrorLog(message);
							}
						}
					}
				}
			}
		}
        else 
        {
			if(LABJACKUSB_DEBUG)
			{
				strcpy(message, "Error getting USBD function pointers during install.\0");          
				WriteToErrorLog(message);
			}
        }
        FreeLibrary(hInst);
    }

 
	//set registry value for ue9
	
	iSuccess = RegOpenKeyEx( HKEY_LOCAL_MACHINE, ue9SubKey, 0, 0, &hKey ); 

	if(iSuccess == ERROR_SUCCESS)
    {
		RegSetValueEx (hKey, TEXT("Prefix"), 0, REG_SZ,(CONST BYTE*) &(TEXT("UE9")), 8);
    }
	else
	{
		if(LABJACKUSB_DEBUG)
		{
			strcpy(message, "Error opening UE9 registry key during install.\0");
			WriteToErrorLog(message);
		}
		fRet = FALSE;
	}

	RegCloseKey(hKey);

	if(fRet)
	{
		//set registry value for u3 
		iSuccess = RegOpenKeyEx( HKEY_LOCAL_MACHINE, u3SubKey, 0, 0, &hKey ); 

		if(iSuccess == ERROR_SUCCESS)
		{
			RegSetValueEx (hKey, TEXT("Prefix"), 0, REG_SZ,(CONST BYTE*) &(TEXT("LU3")), 8);
		}
		else
		{
			if(LABJACKUSB_DEBUG)
			{
				strcpy(message, "Error opening U3 registry key during install.\0");
				WriteToErrorLog(message);
			}
			fRet = FALSE;
		}
		RegCloseKey(hKey);
	}

	return fRet;
}

BOOL 
USBUnInstallDriver()
{
	BOOL fRet = FALSE;
    HINSTANCE hInst = LoadLibrary(L"USBD.DLL");
	INT fRetCount = 0;
	CHAR message[50];

	if(LABJACKUSB_DEBUG)
	{
		strcpy(message, "USBUnInstallDriver.\0");
		WriteToErrorLog(message);
	}

	if(hInst)
    {
        LPUN_REGISTER_CLIENT_DRIVER_ID pUnRegisterId =
                (LPUN_REGISTER_CLIENT_DRIVER_ID) GetProcAddress(hInst, gcszUnRegisterClientDriverId);

        LPUN_REGISTER_CLIENT_SETTINGS pUnRegisterSettings =
                (LPUN_REGISTER_CLIENT_SETTINGS) GetProcAddress(hInst, gcszUnRegisterClientSettings);

        if(pUnRegisterSettings)
        {
            USB_DRIVER_SETTINGS DriverSettingsUE9;
            USB_DRIVER_SETTINGS DriverSettingsU3;
            SetupDriverSettingsUE9(&DriverSettingsUE9);
            SetupDriverSettingsU3(&DriverSettingsU3);

            fRet = (*pUnRegisterSettings)(gcszDriverIdUE9, NULL,&DriverSettingsUE9);
			if(!fRet)
				fRetCount++;

			fRet = (*pUnRegisterSettings)(gcszDriverIdU3, NULL,&DriverSettingsU3);
			if(!fRet)
				fRetCount++;
        }

        if(pUnRegisterId)
        {
            BOOL fRetTemp = (*pUnRegisterId)(gcszDriverIdUE9);
            fRet += fRet ? fRetTemp : fRet;
			fRetTemp = (*pUnRegisterId)(gcszDriverIdU3);
            fRet += fRet ? fRetTemp : fRet;
        }
        FreeLibrary(hInst);
    }

	if(fRetCount > 0)
		return FALSE;
	else 
		return fRet; 
}

/*
 * USBDeviceNotifications
 *
 *    Process notifications from USBD.  Currently, the only notification
 *    supported is a device removal.  
 */
BOOL 
USBDeviceNotifications(LPVOID lpvNotifyParameter, DWORD dwCode,
					   LPDWORD *, LPDWORD *, LPDWORD *, LPDWORD *)
{
	CHAR message[50];
	bool error = FALSE;
    PLABJACKUSB_CONTEXT pLJContext;
	INT i;

    switch(dwCode)
    {
        case USB_CLOSE_DEVICE:
			pLJContext = (PLABJACKUSB_CONTEXT)lpvNotifyParameter; 

			if(VALID_UE9_CONTEXT(pLJContext) || VALID_U3_CONTEXT(pLJContext))
			{
				TryEnterCriticalSection(&pLJContext->csLock);
				
				pLJContext->lpInterface = NULL;

				for(i = 0; i < 4; i++)
					pLJContext->UsbFuncs->lpClosePipe(pLJContext->hPipe[i]);

				pLJContext->ActivePath = NULL;
				pLJContext->UsbFuncs = NULL;
				pLJContext->ActivePath = NULL;
				LeaveCriticalSection(&pLJContext->csLock);
				DeleteCriticalSection(&pLJContext->csLock);
				pLJContext->bIsLocked = FALSE;
				
				DeactivateDevice(pLJContext->hStreamDevice);
				if(LABJACKUSB_DEBUG)
				{
					strcpy(message, "USBDeviceNotifications - Removed device.\0");
					WriteToErrorLog(message);
				}
	        }
			else
			{
				if(LABJACKUSB_DEBUG)
				{
					strcpy(message, "No device to remove.\0");
					WriteToErrorLog(message);
				}
			}
			
			return TRUE;
    }
    return FALSE;
}



BOOL 
USBDeviceAttach(
    USB_HANDLE hDevice,           // USB device handle
    LPCUSB_FUNCS lpUsbFuncs,      // Pointer to USBDI function table.
    LPCUSB_INTERFACE lpInterface, // If client is being loaded as an interface driver, contains a pointer to the USB_INTERFACE
                                  // structure that contains interface information. If client is not loaded for a specific interface,
                                  // this parameter will be NULL.
    LPCWSTR,					  // Contains client driver id string.
    LPBOOL fAcceptControl,        // Filled in with TRUE if we accept control of the device, 
                                  // or FALSE if USBD should continue to try to load client drivers.
    LPCUSB_DRIVER_SETTINGS,	// Contains pointer to USB_DRIVER_SETTINGS struct that indicates how we were loaded.

    DWORD)						  // Reserved for use with future versions of USBD
{
    LPCUSB_CONFIGURATION lpActiveConfig;
    LPCUSB_DEVICE lpDeviceInfo;
	CHAR message[100];
	bool error = FALSE;
	BOOL bRc = TRUE;
    PLABJACKUSB_CONTEXT  pLJContext = NULL;
	INT i = 0;
    HKEY hKey = NULL;
	unsigned long errCode = 0;

	*fAcceptControl = FALSE;
	
	if(LABJACKUSB_DEBUG)
	{
		strcpy(message, "USBDeviceAttach\0");
		WriteToErrorLog(message);
	}

	// Retrieve device descriptor information 
    lpDeviceInfo = (*lpUsbFuncs->lpGetDeviceInfo)(hDevice);
    if (lpDeviceInfo == NULL)
	{
		if(LABJACKUSB_DEBUG)
		{
			strcpy(message, "Error while attaching, LPCUSB_DEVICE = NULL\0");
			WriteToErrorLog(message);
		}
        goto Error;
	}

	if( ( (lpDeviceInfo->Descriptor.idProduct == UE9_PRODUCT_ID) || 
		(lpDeviceInfo->Descriptor.idProduct == U3_PRODUCT_ID) ) && 
		(lpDeviceInfo->Descriptor.idVendor == LABJACK_VENDOR_ID) )
	{
		if(LABJACKUSB_DEBUG)
		{
			sprintf(message, "USBDeviceAttach - recognized device %d\0", lpDeviceInfo->Descriptor.idProduct);
			WriteToErrorLog(message);
		}
		pLJContext = (PLABJACKUSB_CONTEXT)LocalAlloc( LPTR, sizeof(LABJACKUSB_CONTEXT) );
		if ( !pLJContext ) 
		{
			if(LABJACKUSB_DEBUG)
			{
				strcpy(message, "USBDeviceAttach - Error alloc. memory for context.\0"); 
				WriteToErrorLog(message);
			}
			goto Error;
		}

		lpActiveConfig = lpDeviceInfo->lpActiveConfig;

		if(  (int)lpActiveConfig->dwNumInterfaces == 1)
		{
			LPCUSB_INTERFACE lpIF = &lpActiveConfig->lpInterfaces[0];

			ASSERT(lpIF); 
			ASSERT(lpIF->Descriptor.bDescriptorType == USB_INTERFACE_DESCRIPTOR_TYPE);

			if(LABJACKUSB_DEBUG)
			{
				strcpy(message, "USBDeviceAttach - Setting up device context.\0"); 
				WriteToErrorLog(message);
			}

			InitializeCriticalSection(&pLJContext->csLock);
			pLJContext->VendorID = LABJACK_VENDOR_ID;
			pLJContext->ProductID = lpDeviceInfo->Descriptor.idProduct;
			pLJContext->UsbFuncs = lpUsbFuncs;
			pLJContext->hDevice = hDevice;
		
			i = 0;
			for(i; i < 4; i++)
			{
				if(lpDeviceInfo->Descriptor.idProduct == U3_PRODUCT_ID
					&& i == 2)
				{
					//skip since this is an unused/disabled endpoint and seems to cause problems in one case
					pLJContext->hPipe[i] = NULL;
				}
				else
				{
					SetLastError(0);
					if( (pLJContext->hPipe[i] = lpUsbFuncs->lpOpenPipe(hDevice, &lpIF->lpEndpoints[i].Descriptor)) == NULL )
					{
						errCode = GetLastError();

						if(LABJACKUSB_DEBUG)
						{

							sprintf(message, "USBDeviceAttach - cannot open pipe %d\0", i);
							WriteToErrorLog(message);

							sprintf(message, " WinError %lu.\0", errCode);
							WriteToErrorLog(message);
						}
						goto DeleteCont;
					}
				}
			}

			pLJContext->lpInterface = lpIF;
			pLJContext->bIsLocked = FALSE;

			*fAcceptControl = TRUE;
			if(LABJACKUSB_DEBUG)
			{
				strcpy(message, "USBDeviceAttach - driver accepting control\0");
				WriteToErrorLog(message);
			}
		}
		else 
		{
			if(LABJACKUSB_DEBUG)
			{
				sprintf(message, "USBDeviceAttach - invalid num interfaces %ld\0", lpActiveConfig->dwNumInterfaces);
				WriteToErrorLog(message);
			}
			goto DeleteContMem;
		}

		if(LABJACKUSB_DEBUG)
		{
			strcpy(message, "USBDeviceAttach - activating device.\0"); 
			WriteToErrorLog(message);
		}

		if(pLJContext->ProductID == UE9_PRODUCT_ID)
			pLJContext->hStreamDevice = ActivateDevice( ue9SubKey, (DWORD)pLJContext );
		else if(pLJContext->ProductID == U3_PRODUCT_ID)
			pLJContext->hStreamDevice = ActivateDevice( u3SubKey, (DWORD)pLJContext );
		else
		{
			if(LABJACKUSB_DEBUG)
			{
				strcpy(message, "USBDeviceAttach - Error finding prodID to ActivateDevice.\0");
				WriteToErrorLog(message);
			}
			goto DeleteCont;
		}

		if ( pLJContext->hStreamDevice ) 
		{
		     //
			 // register for USB callbacks
			//
			if(lpUsbFuncs->lpRegisterNotificationRoutine( hDevice,
	                                                     USBDeviceNotifications,
	                                                     pLJContext ) != TRUE)
			{
				if(LABJACKUSB_DEBUG)
				{
					strcpy(message, "USBDeviceAttach - Error registering de-attach.\0");
					WriteToErrorLog(message);
				}
				DeactivateDevice(pLJContext->hStreamDevice);  // inform the OS that we're going away
				goto DeleteCont;
			}
            
		} 
		else 
		{
			errCode = GetLastError();
			if(LABJACKUSB_DEBUG)
			{
				strcpy(message, "USBDeviceAttach - Error activating device.\0");
				WriteToErrorLog(message);

				sprintf(message, " WinError %lu.\0", errCode);
				WriteToErrorLog(message);
			}

			goto DeleteCont;
		}

		return TRUE;
	}
	else
	{
		if(LABJACKUSB_DEBUG)
		{
			strcpy(message, "USBDeviceAttach - Unknown productID/VendorID\0");
			WriteToErrorLog(message);
		}
		goto Error;
	}

DeleteCont:
	pLJContext->lpInterface = NULL;

	i = 0;

	for(i; i < 4; i++)
		lpUsbFuncs->lpClosePipe(pLJContext->hPipe[i]);

	pLJContext->ActivePath = NULL;
	pLJContext->UsbFuncs = NULL;
	pLJContext->bIsLocked = FALSE;
	pLJContext->hDevice = NULL;

    DeleteCriticalSection(&pLJContext->csLock);

	goto DeleteContMem;

DeleteContMem:
	LocalFree(pLJContext);
	goto Error;

Error:
	if(LABJACKUSB_DEBUG)
	{
		strcpy(message, "USBDeviceAttach - Error attaching device.\0");
		WriteToErrorLog(message);
	}	
	*fAcceptControl = FALSE;
    return FALSE;
}


HANDLE __stdcall
LJUSB_OpenDevice(UINT DevNum, DWORD dwReserved, ULONG ProductID)
{
	bool err = FALSE;
	UINT devCount = 0;
	INT i = 0;
	HANDLE hDevice = 0;
	ULONG maxDevs = 0;
	const LPTSTR *devName;
	CHAR message[50];

	if(LABJACKUSB_DEBUG)
	{
		strcpy(message, "OpenDevice.\0");
		WriteToErrorLog(message);
	}

	if(ProductID == UE9_PRODUCT_ID)
	{
		maxDevs = UE9_MAX_DEVICES;
		devName = ue9DevName;
	}
	else if(ProductID == U3_PRODUCT_ID)
	{
		maxDevs = U3_MAX_DEVICES;
		devName = u3DevName;
	}
	else
	{
		if(LABJACKUSB_DEBUG)
		{
			strcpy(message, "OpenDevice-Invalid productID\0");
			WriteToErrorLog(message);
		}
		goto Invalid;
	}
	if(DevNum > maxDevs || DevNum < 1)
		goto Invalid;

	for(i = 0; i < (INT)maxDevs; i++) 
	{
		if((hDevice= CreateFile(devName[i], (GENERIC_READ | GENERIC_WRITE), 
					(FILE_SHARE_READ | FILE_SHARE_WRITE), NULL, OPEN_EXISTING, 
					FILE_ATTRIBUTE_NORMAL, NULL)) != INVALID_HANDLE_VALUE )
		{
			devCount++;		
			
			if(devCount == DevNum)
				return hDevice;
			else
				CloseHandle(hDevice);
		}
		
	}

	if(LABJACKUSB_DEBUG)
	{
		sprintf(message, "OpenDevice - DevNum %u not found.\0", DevNum);
		WriteToErrorLog(message);
	}

Invalid:
	return INVALID_HANDLE_VALUE;
}

ULONG __stdcall
LJUSB_BulkRead(HANDLE hDevice, ULONG Pipe, BYTE *pBuff, ULONG Count)
{
	ULONG bytesRead = 0;
	bool err = FALSE;
	CHAR message[50];
	
	if(hDevice == INVALID_HANDLE_VALUE)
		return bytesRead;

	if(Pipe == UE9_PIPE_EP1_IN || Pipe == U3_PIPE_EP1_IN)
	{
		if(!ReadFile(hDevice, pBuff, Count, &bytesRead, NULL))
			goto Error;
	}
	else if(Pipe == UE9_PIPE_EP2_IN || Pipe == U3_PIPE_EP2_IN)
	{
		if(!ReadFile(hDevice, pBuff, Count + 32768, &bytesRead, NULL))
			goto Error;
	}
	else
	{
		return bytesRead;
	}

	return bytesRead;
Error:
	if(LABJACKUSB_DEBUG)
	{
		sprintf(message, "BulkRead error: WinError %u.\0", GetLastError());
		WriteToErrorLog(message);
	}
	return bytesRead;
}

ULONG __stdcall 
LJUSB_BulkWrite(HANDLE hDevice, ULONG Pipe, BYTE *pBuff, ULONG Count)
{
	ULONG bytesWrote = 0;
	bool err = FALSE;
	CHAR message[50];

	if(hDevice == INVALID_HANDLE_VALUE )
		return bytesWrote;

	if(Pipe == UE9_PIPE_EP1_OUT || Pipe == U3_PIPE_EP1_OUT)
	{
		if(!WriteFile(hDevice, pBuff, Count, &bytesWrote, NULL))
			goto Error;
	}
	else if(Pipe == UE9_PIPE_EP2_OUT)
	{
		if(!WriteFile(hDevice, pBuff, Count + 32768, &bytesWrote, NULL))
			goto Error;
	}
	else
	{
		return bytesWrote;
	}

	return bytesWrote;
Error:
	if(LABJACKUSB_DEBUG)
	{
		sprintf(message, "BulkWrite error: WinError %u.\0", GetLastError());
		WriteToErrorLog(message);
	}
	return bytesWrote;
}


void __stdcall
LJUSB_CloseDevice(HANDLE hDevice)
{
	CloseHandle(hDevice);
}


ULONG __stdcall
LJUSB_GetDevCount(ULONG ProductID)
{
	bool err = FALSE;
	INT i;
	INT count;
	HANDLE hDevice;
	ULONG maxDevs;
	const LPTSTR *devName;

	count = 0;

	if(ProductID == UE9_PRODUCT_ID)
	{
		maxDevs = UE9_MAX_DEVICES;
		devName = ue9DevName;

	}
	else if(ProductID == U3_PRODUCT_ID)
	{
		maxDevs = U3_MAX_DEVICES;
		devName = u3DevName;
	}
	else
		goto Count_ret;
	
	for(i = 0; i < (INT)maxDevs; i++)
	{
	
		if((hDevice = CreateFile(devName[i], (GENERIC_READ | GENERIC_WRITE), 
						(FILE_SHARE_READ | FILE_SHARE_WRITE), NULL, OPEN_EXISTING, 
						FILE_ATTRIBUTE_NORMAL, NULL)) != INVALID_HANDLE_VALUE)
		{
			count++;
			CloseHandle(hDevice);
		}
	}

Count_ret:
	return count;
}

BOOL __stdcall
LJUSB_AbortPipe(HANDLE hDevice, ULONG Pipe)
{
	BOOL sucess = FALSE;
	BYTE bPipe[1];
	CHAR message[50];
	DWORD bytesReturned = 0;
	bPipe[0] = (BYTE)Pipe;

	if( (sucess = DeviceIoControl(hDevice, IOCTL_ABORT_PIPE, &bPipe, 1, NULL, 0, &bytesReturned, NULL)) != TRUE)
	{
		if(LABJACKUSB_DEBUG)
		{
			sprintf(message, "AbortPipe error: WinError %u.\0", GetLastError() );
			WriteToErrorLog(message);
		}
	}

	return sucess;
}

//Not implemented
UINT _stdcall LJUSB_GetDevicePath(UINT DevNum, ULONG ProductID, char *Dest, UINT Length)
{
	return 0;
}

//Not implemented
HANDLE _stdcall LJUSB_OpenDevicePath(char *Path, DWORD dwFlagsAndAttributes)
{
	return INVALID_HANDLE_VALUE;
}


//////////////////////////////////////////////
//		Stream Interface Functions	- UE9	//
//////////////////////////////////////////////

DWORD
UE9_Init (DWORD dwCtx)
{
	CHAR message[50];
	LPTSTR ActivePath = (LPTSTR) dwCtx;
	HKEY hKey;
	long lStatus;
	PLABJACKUSB_CONTEXT pUE9 = NULL;

	if (ActivePath) 
	{
		lStatus = RegOpenKeyEx( HKEY_LOCAL_MACHINE,
                              ActivePath,
                              0,
                              0,
                              &hKey);

		if (lStatus==ERROR_SUCCESS) 
		{
			DWORD dwVal;
			DWORD dwType = REG_DWORD;
			DWORD dwValLen = sizeof(dwVal);
			lStatus = RegQueryValueEx( hKey,
                                    TEXT("ClientInfo"),
                                    NULL,
                                    &dwType,
                                    (LPBYTE)(&dwVal),
                                    &dwValLen);

			if (lStatus == ERROR_SUCCESS) 
			{
				pUE9 = (PLABJACKUSB_CONTEXT)dwVal;
			}

			RegCloseKey(hKey);
		}

    }
	else
	{
		return NULL;
	}

	if(VALID_UE9_CONTEXT(pUE9))
	{
		pUE9->ActivePath = ActivePath;
		return (DWORD)pUE9;
	}
	else
	{
		if(LABJACKUSB_DEBUG)
		{
			strcpy(message, "Init error: invalid context.\0");          			
			WriteToErrorLog(message);
		}

		return NULL;
	}	
}

BOOL
UE9_Deinit (DWORD dwCtx)
{
	PLABJACKUSB_CONTEXT pUE9;
	INT i;

	pUE9 = (PLABJACKUSB_CONTEXT) dwCtx;

	if(VALID_UE9_CONTEXT(pUE9))
	{
		EnterCriticalSection(&pUE9->csLock);
		DeactivateDevice(pUE9->hStreamDevice);
				
		pUE9->lpInterface = NULL;

		i = 0;

		for(i; i < 4; i++)
			pUE9->UsbFuncs->lpClosePipe(pUE9->hPipe[i]);

		pUE9->ActivePath = NULL;

		pUE9->UsbFuncs = NULL;

		LeaveCriticalSection(&pUE9->csLock);
		DeleteCriticalSection(&pUE9->csLock);
		
		LocalFree(pUE9);
		return TRUE;
	}
	else
		return FALSE;
}

DWORD
UE9_Open (DWORD dwCtx, DWORD, DWORD)
{
   	if( VALID_UE9_CONTEXT( ((PLABJACKUSB_CONTEXT)dwCtx)) )
		return dwCtx;
	else 
		return NULL; 
}

BOOL
UE9_Close (DWORD)
{
   return TRUE;
}

DWORD
UE9_Read (DWORD dwCtx, BYTE *pBuffer, DWORD count)
{
	PLABJACKUSB_CONTEXT pUE9;
	USB_TRANSFER hTransfer = NULL;
    DWORD dwWaitReturn = 0;
    BOOL  bRc = FALSE;
    DWORD dwErr = ERROR_SUCCESS;
	USB_PIPE hPipe;
	DWORD bytesTransferred = 0; 
	DWORD pUsbRc = USB_NO_ERROR; 
	
	pUE9 = (PLABJACKUSB_CONTEXT) dwCtx;

	EnterCriticalSection(&pUE9->csLock);
	pUE9->bIsLocked = TRUE;

    
    if ( VALID_UE9_CONTEXT(pUE9) )
	{

		if(count < 32768)
		{
			hPipe = pUE9->hPipe[UE9_PIPE_EP1_IN];

			hTransfer = pUE9->UsbFuncs->lpIssueBulkTransfer( hPipe,
															 NULL,
															 NULL,
															 USB_IN_TRANSFER | USB_SHORT_TRANSFER_OK,
															 count,
															 pBuffer,
															 0 );
		}
		else
		{
			count = count - 32768;

			hPipe = pUE9->hPipe[UE9_PIPE_EP2_IN];

			hTransfer = pUE9->UsbFuncs->lpIssueBulkTransfer( hPipe,
															 NULL,
															 NULL,
															 USB_IN_TRANSFER | USB_SHORT_TRANSFER_OK,
															 count,
															 pBuffer,
															 0 );
		}

        if ( hTransfer ) 
		{
            pUE9->UsbFuncs->lpGetTransferStatus(hTransfer, &bytesTransferred, &pUsbRc);
   			pUE9->UsbFuncs->lpCloseTransfer(hTransfer);
	    } 
		else 
		{
	        dwErr = GetLastError();
        }

    } 
	else 
	{
        dwErr = ERROR_INVALID_PARAMETER;
    }

    if ( ERROR_SUCCESS == dwErr && USB_NO_ERROR != pUsbRc) 
	{
        dwErr = pUsbRc;
    }

    if ( ERROR_SUCCESS != dwErr ) 
	{
        SetLastError(dwErr);
		bytesTransferred = 0;
    }

	
	LeaveCriticalSection(&pUE9->csLock);
	pUE9->bIsLocked = FALSE;

    return bytesTransferred;
}

DWORD
UE9_Write (DWORD dwCtx, BYTE *pBuffer,DWORD count)
{
	PLABJACKUSB_CONTEXT pUE9;
	USB_TRANSFER hTransfer = NULL;
    DWORD dwWaitReturn = 0;
    BOOL  bRc = FALSE;
    DWORD dwErr = ERROR_SUCCESS;
	USB_PIPE hPipe;
	DWORD bytesTransferred = 0; 
	DWORD pUsbRc = USB_NO_ERROR; 

	pUE9 = (PLABJACKUSB_CONTEXT) dwCtx;
	
	EnterCriticalSection(&pUE9->csLock);
   	pUE9->bIsLocked = TRUE;

    if ( VALID_UE9_CONTEXT(pUE9) ) 
	{
		if(count < 32768)
		{
			hPipe = pUE9->hPipe[UE9_PIPE_EP1_OUT];

			hTransfer = pUE9->UsbFuncs->lpIssueBulkTransfer( hPipe,
															 NULL,
															 NULL,
															 USB_OUT_TRANSFER,
															 count,
															 pBuffer,
															 0 );
		}
		else
		{
			count = count - 32768;

			hPipe = pUE9->hPipe[UE9_PIPE_EP2_OUT];

			hTransfer = pUE9->UsbFuncs->lpIssueBulkTransfer( hPipe,
															 NULL,
															 NULL,
															 USB_OUT_TRANSFER,
															 count,
															 pBuffer,
															 0 );
		}

		if ( hTransfer ) 
		{
            pUE9->UsbFuncs->lpGetTransferStatus(hTransfer, &bytesTransferred, &pUsbRc);
        	pUE9->UsbFuncs->lpCloseTransfer(hTransfer);
		} 
		else 
		{
		    dwErr = GetLastError();
        }
    } 
	else 
	{
        dwErr = ERROR_INVALID_PARAMETER;
    }

    if ( ERROR_SUCCESS == dwErr && USB_NO_ERROR != pUsbRc) 
	{
        dwErr = pUsbRc;
    }

    if ( ERROR_SUCCESS != dwErr ) 
	{
        SetLastError(dwErr);
		bytesTransferred = 0;
    }
	
	LeaveCriticalSection(&pUE9->csLock);
	pUE9->bIsLocked = FALSE;
	
    return bytesTransferred;
}

void 
UE9_PowerDown(DWORD)
{
	SetLastError(ERROR_NOT_SUPPORTED);
}

void 
UE9_PowerUp(DWORD)
{
	SetLastError(ERROR_NOT_SUPPORTED);
}

DWORD
UE9_Seek (DWORD,long,DWORD)
{
   SetLastError(ERROR_NOT_SUPPORTED);
   return -1;
}

BOOL
UE9_IOControl (DWORD dwCtx, DWORD dwCode, 
			   PBYTE pBufIn, DWORD dwLenIn, 
			   PBYTE pBufOut, DWORD dwLenOut, 
			   PDWORD pdwActualOut)
{
	PLABJACKUSB_CONTEXT pUE9;
	INT pipe = 5;
	BOOL halted = false;
	UINT i = 0;
	
	pUE9 = (PLABJACKUSB_CONTEXT) dwCtx;
						
	if ( VALID_UE9_CONTEXT(pUE9) ) 
	{
		switch(dwCode)
		{
			case IOCTL_ABORT_PIPE:
				if(dwLenIn >= 1)
				{
					pipe = (INT)pBufIn[0];

					if(pipe < 4 && pipe >= 0)
					{
						if(pUE9->UsbFuncs->lpAbortPipeTransfers(pUE9->hPipe[pipe], 0) == TRUE)
							return TRUE;
					}
				}
				break;
			case IOCTL_GET_PRODUCT_ID:
				if(dwLenOut >= 1)
				{
					pBufOut[0] = (BYTE)pUE9->ProductID;
					*pdwActualOut = 1;
					return TRUE;
				}
				break;
			case IOCTL_GET_ACTIVE_PATH:
				if(sizeof(pUE9->ActivePath)/2 <= dwLenOut)
				{
					*pdwActualOut = sizeof(pUE9->ActivePath)/2;
					for(i = 0; i < *pdwActualOut; i++)
						pBufOut[0] = pUE9->ActivePath[0];
						
					return TRUE;
				}
			default:
				break;
		}
	}

	return FALSE;
}


//////////////////////////////////////////////
//		Stream Interface Functions	- U3	//
//////////////////////////////////////////////

DWORD
LU3_Init (DWORD dwCtx)
{
	
	CHAR message[50];
	LPTSTR ActivePath = (LPTSTR) dwCtx;
	HKEY hKey;
	long lStatus;
	PLABJACKUSB_CONTEXT pU3 = NULL;

	if (ActivePath) 
	{
		lStatus = RegOpenKeyEx( HKEY_LOCAL_MACHINE,
                              ActivePath,
                              0,
                              0,
                              &hKey);

		if (lStatus==ERROR_SUCCESS) 
		{
			DWORD dwVal;
			DWORD dwType = REG_DWORD;
			DWORD dwValLen = sizeof(dwVal);
			lStatus = RegQueryValueEx( hKey,
                                    TEXT("ClientInfo"),
                                    NULL,
                                    &dwType,
                                    (LPBYTE)(&dwVal),
                                    &dwValLen);

			if (lStatus == ERROR_SUCCESS) 
			{
				pU3 = (PLABJACKUSB_CONTEXT)dwVal;
			}

			RegCloseKey(hKey);
		}

    }
	else
	{
		if(LABJACKUSB_DEBUG)
		{
			strcpy(message, "Init error: invalid active path.\0");          			
			WriteToErrorLog(message);
		}
		return NULL;
	}

	if(VALID_U3_CONTEXT(pU3))
	{
		pU3->ActivePath = ActivePath;
		return (DWORD)pU3;
	}
	else
	{
		if(LABJACKUSB_DEBUG)
		{
			strcpy(message, "Init error: invalid context.\0");          			
			WriteToErrorLog(message);
		}

		return NULL;
	}
	
	return NULL;
}

BOOL
LU3_Deinit (DWORD dwCtx)
{
	PLABJACKUSB_CONTEXT pU3;
	INT i;

	pU3 = (PLABJACKUSB_CONTEXT) dwCtx;

	if(VALID_U3_CONTEXT(pU3))
	{
		EnterCriticalSection(&pU3->csLock);
		DeactivateDevice(pU3->hStreamDevice);
				
		pU3->lpInterface = NULL;

		i = 0;

		for(i; i < 4; i++)
			pU3->UsbFuncs->lpClosePipe(pU3->hPipe[i]);

		pU3->ActivePath = NULL;

		pU3->UsbFuncs = NULL;

		LeaveCriticalSection(&pU3->csLock);
		DeleteCriticalSection(&pU3->csLock);
		
		LocalFree(pU3);
		return TRUE;
	}
	else
		return FALSE;
}

DWORD
LU3_Open (DWORD dwCtx, DWORD, DWORD)
{
   	if( VALID_U3_CONTEXT( ((PLABJACKUSB_CONTEXT)dwCtx)) )
		return dwCtx;
	else 
		return NULL; 
}

BOOL
LU3_Close (DWORD)
{
   return TRUE;
}

DWORD
LU3_Read (DWORD dwCtx, BYTE *pBuffer, DWORD count)
{
	PLABJACKUSB_CONTEXT pU3;
	USB_TRANSFER hTransfer = NULL;
    DWORD dwWaitReturn = 0;
    BOOL  bRc = FALSE;
    DWORD dwErr = ERROR_SUCCESS;
	USB_PIPE hPipe;
	DWORD bytesTransferred = 0; 
	DWORD pUsbRc = USB_NO_ERROR; 
	CHAR message[50];
	long time0 = 0, time1 = 0;

	pU3 = (PLABJACKUSB_CONTEXT) dwCtx;

	EnterCriticalSection(&pU3->csLock);
	pU3->bIsLocked = TRUE;

    if ( VALID_U3_CONTEXT(pU3) )
	{
		if(count < 32768)
		{
			hPipe = pU3->hPipe[U3_PIPE_EP1_IN];
			time0 = GetTickCount();

			hTransfer = pU3->UsbFuncs->lpIssueBulkTransfer( hPipe,
															 NULL,
															 NULL,
															 USB_IN_TRANSFER | USB_SHORT_TRANSFER_OK,
															 count,
															 pBuffer,
															 0 );

			time1 = GetTickCount();

		}
		else
		{
			count = count - 32768;

			hPipe = pU3->hPipe[U3_PIPE_EP2_IN];

			hTransfer = pU3->UsbFuncs->lpIssueBulkTransfer( hPipe,
															 NULL,
															 NULL,
															 USB_IN_TRANSFER | USB_SHORT_TRANSFER_OK,
															 count,
															 pBuffer,
															 0 );
		}

        if ( hTransfer ) 
		{
			pU3->UsbFuncs->lpGetTransferStatus(hTransfer, &bytesTransferred, &pUsbRc);
   			pU3->UsbFuncs->lpCloseTransfer(hTransfer);
	    } 
		else 
		{
	        dwErr = GetLastError();
        }
    } 
	else 
	{
        dwErr = ERROR_INVALID_PARAMETER;
    }

    if ( ERROR_SUCCESS == dwErr && USB_NO_ERROR != pUsbRc) 
	{
        dwErr = pUsbRc;
    }

    if ( ERROR_SUCCESS != dwErr ) 
	{
        SetLastError(dwErr);
		bytesTransferred = 0;
    }

	
	LeaveCriticalSection(&pU3->csLock);
	pU3->bIsLocked = FALSE;

	if(LABJACKUSB_DEBUG)
	{
		sprintf(message, "read : error = %d, count = %d.\0", dwErr, bytesTransferred );
		WriteToErrorLog(message);
		sprintf(message, "    time = %ld.\0", (time1 - time0) );
		WriteToErrorLog(message);
	}

    return bytesTransferred;
}

DWORD
LU3_Write (DWORD dwCtx, BYTE *pBuffer,DWORD count)
{
	PLABJACKUSB_CONTEXT pU3;
	USB_TRANSFER hTransfer = NULL;
    DWORD dwWaitReturn = 0;
    BOOL  bRc = FALSE;
    DWORD dwErr = ERROR_SUCCESS;
	USB_PIPE hPipe;
	DWORD bytesTransferred = 0; 
	DWORD pUsbRc = USB_NO_ERROR; 
	CHAR message[50];
	long time0 = 0, time1 = 0;

	pU3 = (PLABJACKUSB_CONTEXT) dwCtx;
	
	EnterCriticalSection(&pU3->csLock);
   	pU3->bIsLocked = TRUE;

    if ( VALID_U3_CONTEXT(pU3) ) 
	{
		time0 = GetTickCount();
		hPipe = pU3->hPipe[U3_PIPE_EP1_OUT];

			hTransfer = pU3->UsbFuncs->lpIssueBulkTransfer( hPipe,
															 NULL,
															 NULL,
															 USB_OUT_TRANSFER,
															 count,
															 pBuffer,
															 0 );
		time1 = GetTickCount();

		if ( hTransfer ) 
		{
            pU3->UsbFuncs->lpGetTransferStatus(hTransfer, &bytesTransferred, &pUsbRc);
        	pU3->UsbFuncs->lpCloseTransfer(hTransfer);
		} 
		else 
		{
		    dwErr = GetLastError();
        }
    } 
	else 
	{
        dwErr = ERROR_INVALID_PARAMETER;
    }

    if ( ERROR_SUCCESS == dwErr && USB_NO_ERROR != pUsbRc) 
	{
        dwErr = pUsbRc;
    }

    if ( ERROR_SUCCESS != dwErr ) 
	{
        SetLastError(dwErr);
		bytesTransferred = 0;
    }

	LeaveCriticalSection(&pU3->csLock);
	pU3->bIsLocked = FALSE;

	if(LABJACKUSB_DEBUG)
	{
		sprintf(message, "write : error = %d, count = %d.\0", dwErr, bytesTransferred );
		WriteToErrorLog(message);
		sprintf(message, "    time = %ld.\0", (time1 - time0) );
		WriteToErrorLog(message);
	}

    return bytesTransferred;
}

void 
LU3_PowerDown(DWORD)
{
	SetLastError(ERROR_NOT_SUPPORTED);
}

void 
LU3_PowerUp(DWORD)
{
	SetLastError(ERROR_NOT_SUPPORTED);
}

DWORD
LU3_Seek (DWORD,long,DWORD)
{
   SetLastError(ERROR_NOT_SUPPORTED);
   return -1;
}

BOOL
LU3_IOControl (DWORD dwCtx, DWORD dwCode, 
			   PBYTE pBufIn, DWORD dwLenIn, 
			   PBYTE pBufOut, DWORD dwLenOut, 
			   PDWORD pdwActualOut)
{
	PLABJACKUSB_CONTEXT pU3;
	INT pipe = 5;
	BOOL halted = false;
	UINT i = 0;
	
	pU3 = (PLABJACKUSB_CONTEXT) dwCtx;
	
	if ( VALID_U3_CONTEXT(pU3) ) 
	{
		switch(dwCode)
		{
			case IOCTL_ABORT_PIPE:
				if(dwLenIn >= 1)
				{
					pipe = (INT)pBufIn[0];

					if(pipe < 4 && pipe >= 0)
					{
						if(pU3->UsbFuncs->lpAbortPipeTransfers(pU3->hPipe[pipe], 0) == TRUE)
							return TRUE;
					}
				}
				break;
			case IOCTL_GET_PRODUCT_ID:
				if(dwLenOut >= 1)
				{
					pBufOut[0] = (BYTE)pU3->ProductID;
					*pdwActualOut = 1;
					return TRUE;
				}
				break;
			case IOCTL_GET_ACTIVE_PATH:
				if(sizeof(pU3->ActivePath)/2 <= dwLenOut)
				{
					*pdwActualOut = sizeof(pU3->ActivePath)/2;
					for(i = 0; i < *pdwActualOut; i++)
						pBufOut[0] = pU3->ActivePath[0];
						
					return TRUE;
				}
			default:
				break;
		}
	}

	return FALSE;
}

//EOF