﻿using System;
using System.Diagnostics;
using System.IO;
using System.Text;
using System.Threading;
using LabJack;

namespace LoggerModbusTCPExample
{
    /// <summary>
    /// Class containing example for logging LabJack readings using Modbus over
    /// TCP.
    /// </summary>
    class Program
    {
        static void Main(string[] args)
        {
            try
            {
                Program example = new Program();
                example.ReadAndLog();
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.ToString());
            }

            Console.WriteLine("\nPress Enter to exit.");
            Console.ReadLine();
        }

        private DateTime _epoch;  // The epoch time.
        private int _logInterval = 0;  // The log interval, in milliseconds
        private long _startTimeTickMS = 0;  // The tick start time, in milliseconds.
        private long _nextTimeTickMS = 0;  // The next interval tick time, in milliseconds.
        private long _startTimeEpochMS = 0;  //The start time from the epoch, in milliseconds.

        /// <summary>
        /// Example that performs LabJack device readings using Modbus over
        /// TCP and logs to a to a file.
        /// By default, AIN0-AIN3 are logged to a log#.csv file at a 500 ms
        /// interval. Data delimiter is the comma character. The timestamp is
        /// the UNIX/POSIX epoch (milliseconds elapsed from January 1, 1970 at
        /// 00:00:00 GMT).
        /// </summary>
        public void ReadAndLog()
        {
            // Configuration
            string ipAddr = "192.168.1.166";  // LabJack device IP address.

            string fileName = "log";  // Log file name.
            string fileExt = "csv";  // Log file extension.
            string delimiter = ",";  // Delimiter to use between data

            ushort startAddress = 0;  // Start register address. 0 = AIN0
            int numAINs = 4;  // Number of analog inputs

            this._logInterval = 500;  // Log interval, in ms.

            // Epoch time used for timestamps. Currently set to UNIX/POSIX epoch.
            this._epoch = new DateTime(1970, 1, 1, 0, 0, 0, DateTimeKind.Utc);

            // Connect to the device
            Console.WriteLine("Connecting to device with IP address " + ipAddr);
            ModbusTCPClient mb = new ModbusTCPClient(ipAddr);

            try
            {
                string fullFileName = "";
                // Getting number to append to fileName.
                for (int i = 0; ; i++)
                {
                    fullFileName = fileName + i + "." + fileExt;
                    if (!File.Exists(fullFileName))
                        break;
                }
                // Create file to log to.
                StreamWriter file = new StreamWriter(fullFileName, false, Encoding.ASCII);

                try
                {
                    // Write header information
                    string header = "TIME";
                    for (int i = 0; i < numAINs; i++)
                        header += delimiter + "AIN" + (int)((startAddress + i * 2) / 2);
                    file.WriteLine(header);

                    Console.WriteLine("Logging to file " + fullFileName);
                    Console.WriteLine("Press q to stop");

                    float[] ains = new float[numAINs];
                    string readings = "";
                    bool run = true;
                    long timeStamp = 0;

                    SetStartTime();

                    while (run)
                    {
                        // Reading consecutive analog inputs.
                        mb.Read(startAddress, ains);

                        // Get Timestamp
                        timeStamp = this.GetTime();

                        // Readings to be logged.
                        readings = "";
                        foreach (float value in ains)
                            readings += delimiter + value.ToString("F6");

                        // Write to file timestamp and readings
                        file.WriteLine(timeStamp + readings);
                        //Console.WriteLine(timeStamp + readings);  // Console display off by default for better performance

                        // Check if user pressed q to quit logging.
                        while (Console.KeyAvailable)
                        {
                            if (Console.ReadKey(false).Key == ConsoleKey.Q)
                                run = false;
                        }

                        // Delay for next log interval
                        Thread.Sleep(GetNextSleepInterval());
                    }
                }
                catch (Exception ex)
                {
                    Console.WriteLine(ex.ToString());
                }

                Console.WriteLine("\nLogging Done");
                // Close file
                file.Close();
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.ToString());
            }
            // Close connection
            mb.Close();
        }

        /// <summary>
        /// Gets the Stopwatch timestamp.
        /// </summary>
        /// <returns>The Stopwatch timestamp in milliseconds.</returns>
        private long GetStopwatchTimestampMS()
        {
            return (long)(Stopwatch.GetTimestamp() / (Stopwatch.Frequency / 1000.0));
        }

        /// <summary>
        /// Set the start times for logging. Automatically uses current times.
        /// </summary>
        private void SetStartTime()
        {
            this._startTimeEpochMS = (long)(DateTime.Now - this._epoch).TotalMilliseconds;
            this._startTimeTickMS = GetStopwatchTimestampMS();
            this._nextTimeTickMS = this._startTimeTickMS + this._logInterval;
        }

        /// <summary>
        /// Get the current time. Set _epoch field and call SetStartTime before
        /// using.
        /// </summary>
        /// <returns>The UNIX/POSIX epoch in milliseconds (elapsed from
        /// January 1, 1970 at 00:00:00 GMT).</returns>
        private long GetTime()
        {
            if (this._epoch == null)
                throw new Exception("_epoch field needs to be set.");

            return (GetStopwatchTimestampMS() - this._startTimeTickMS) + this._startTimeEpochMS;
        }

        /// <summary>
        /// Calculate the sleep delay for the next interval reading. Set the
        /// _logInterval field and call SetStartTime before using.
        /// </summary>
        /// <returns>The sleep time for the next interval, in
        /// milliseconds.</returns>
        private int GetNextSleepInterval()
        {
            if (this._logInterval <= 0)
                throw new Exception("_logInterval field needs to be a value > 1.");

            long curTime = GetStopwatchTimestampMS();
            while (this._nextTimeTickMS <= curTime)
                this._nextTimeTickMS += this._logInterval;
            return (int)(this._nextTimeTickMS - curTime);
        }
    }
}
