﻿using System;
using LabJack;  //For ModbusTCPClient

namespace T4ModbusTCPExamples
{
    /// <summary>
    /// Class containing LabJack T4 Modbus over TCP examples.
    /// </summary>
    class Program
    {
        static void Main(string[] args)
        {
            ModbusTCPClient mb = null;
            string ipAddr = "192.168.1.105";  //Change to your T4's IP address

            try
            {
                // Connect to the T4
                Console.WriteLine("Connecting to a device with IP address " + ipAddr + ".\n");
                mb = new ModbusTCPClient(ipAddr);

                Program example = new Program();
                example.ReadAllAnalogExample(mb);
                example.ReadAllDigitalIOExample(mb);
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.ToString());
            }

            if (mb != null)
            {
                // Close the T4 connection
                mb.Close();
            }

            Console.WriteLine("\nPress Enter to exit.");
            Console.ReadLine();
        }

        /// <summary>
        /// Displays "AINxx : Values" readings.
        /// </summary>
        /// <param name="startingAddress">Starting Modbus address of
        /// readings.</param>
        /// <param name="values">Float analog input (AIN) readings.</param>
        public void DisplayAnalogInputReadings(ushort startingAddress, float[] values)
        {
            for (int i = 0; i < values.Length; i++)
            {
                Console.WriteLine("AIN" + (startingAddress + i * 2) / 2 + " : " + values[i] + " V");
            }
        }

        /// <summary>
        /// Displays all digital I/O readings.
        /// </summary>
        /// <param name="directions">Reading from T4 Modbus address
        /// 2850 (DIO_DIRECTION).</param>
        /// <param name="states">Reading from T4 Modbus address
        /// 2800 (DIO_STATE).</param>
        public void DisplayDigitalIOReadings(uint directions, uint states)
        {
            string fioDirs = "";
            string fioStates = "";
            string eioDirs = "";
            string eioStates = "";
            string cioDirs = "";
            string cioStates = "";

            for (int i = 4; i < 8; i++)
            {
                fioDirs += Convert.ToString((directions >> i) & 1);
                fioStates += Convert.ToString((states >> i) & 1);
            }
            Console.WriteLine("FIO4-FIO7 directions = " + fioDirs + ", states = " + fioStates);

            for (int i = 8; i < 16; i++)
            {
                eioDirs += Convert.ToString((directions >> i) & 1);
                eioStates += Convert.ToString((states >> i) & 1);
            }
            Console.WriteLine("EIO0-EIO7 directions = " + eioDirs + ", states = " + eioStates);

            for (int i = 16; i < 20; i++)
            {
                cioDirs += Convert.ToString((directions >> i) & 1);
                cioStates += Convert.ToString((states >> i) & 1);
            }
            Console.WriteLine("CIO0-CIO3 directions = " + cioDirs + ", states = " + cioStates);
        }

        /// <summary>
        /// Configures analog enable, resolution index and settling time for all
        /// analog inputs.
        /// DIO_ANALOG_ENABLE, AIN_ALL_RESOLUTION, and AIN_ALL_SETTLING_US
        /// registers/settings documented here:
        /// https://labjack.com/support/datasheets/t-series/ain
        /// </summary>
        /// <param name="mb">The ModbusTCPClient to the connected T4.</param>
        /// <param name="analogEnable">DIO_ANALOG_ENABLE settings.</param>
        /// <param name="resolutionIndex">AIN_ALL_RESOLUTION_INDEX setting.</param>
        /// <param name="settling">AIN_ALL_SETTLING_US setting.</param>
        public void ConfigureAllAnalog(ModbusTCPClient mb, uint analogEnable, ushort resolutionIndex, float settling)
        {
            ushort address;
            ushort uint16Value;
            uint uint32Value;
            float float32Value;

            // Configure all flexible lines to analog inputs or digital I/O.
            address = 2900;  // 2900 = DIO_INHIBIT
            uint32Value = 0x000;  //Setting bits 0-11 to all 0. 0 = Affected.
            mb.Write(address, uint32Value);

            address = 2880;  // 2880 = DIO_ANALOG_ENABLE
            uint32Value = analogEnable;
            mb.Write(address, uint32Value);

            // Configure all analog input resolution indexes.
            address = 43903;  // 43903 = AIN_ALL_RESOLUTION_INDEX
            uint16Value = resolutionIndex;
            mb.Write(address, uint16Value);

            // Configure all analog input settling times.
            address = 43904;  // 43904 = AIN_ALL_SETTLING_US
            float32Value = settling;
            mb.Write(address, float32Value);
        }

        /// <summary>
        /// Example that configures, reads and displays all the analog
        /// inputs (AIN0-AIN11) on the T4.
        /// Analog inputs (AIN) registers used are documented here:
        /// https://labjack.com/support/datasheets/t-series/ain
        /// </summary>
        /// <param name="mb">The ModbusTCPClient to the connected T4.</param>
        public void ReadAllAnalogExample(ModbusTCPClient mb)
        {
            Console.WriteLine("Reading AIN0-AIN11.");

            // Configure all analog inputs.
            // DIO Analog Enable = 0xFFF = b111111111111 (Enable AIN0-AIN11)
            // Resolution Indexes = 5
            // Settlings = 0 (auto)
            ConfigureAllAnalog(mb, 0xFFF, 5, 0);

            // Read all 12 analog inputs.
            ushort startAddress = 0;  // 0 = AIN0
            float[] ains = new float[12];  // 12 analog input readings
            mb.Read(startAddress, ains);
            DisplayAnalogInputReadings(startAddress, ains);

            Console.WriteLine("");
        }

        /// <summary>
        /// Example that reads and displays all the digital I/O (FIOs, EIOs,
        /// CIOs) on the T4.
        /// Digital I/O registers used are documented here:
        /// https://labjack.com/support/datasheets/t-series/digital-io
        /// </summary>
        /// <param name="mb">The ModbusTCPClient to the connected T4.</param>
        public void ReadAllDigitalIOExample(ModbusTCPClient mb)
        {
            ushort address = 0;
            uint directions = 0;
            uint states = 0;

            Console.WriteLine("Reading FIOs, EIOs and CIOs directions and states.");

            // Configure all flexible I/O lines to digital I/O (FIO4-7, EIO0-3).
            // DIO Analog Enable = 0x00F = b000000001111
            // Resolution Indexes = 0
            // Settlings = 0 (auto)
            ConfigureAllAnalog(mb, 0x00F, 0, 0);

            // Read all digital I/O directions and states.
            address = 2850;  // 2850 = DIO_DIRECTION
            mb.Read(address, ref directions);
            address = 2800;  // 2800 = DIO_STATE
            mb.Read(address, ref states);

            DisplayDigitalIOReadings(directions, states);

            Console.WriteLine("");
        }
    }
}
