﻿using System;
using LabJack;  //For ModbusTCPClient

namespace T7ModbusTCPExamples
{
    /// <summary>
    /// Class containing LabJack T7 Modbus over TCP examples.
    /// </summary>
    class Program
    {
        static void Main(string[] args)
        {
            ModbusTCPClient mb = null;
            string ipAddr = "192.168.1.166";  //Change to your T7's IP address

            try
            {
                // Connect to the T7
                Console.WriteLine("Connecting to a device with IP address " + ipAddr + ".");
                mb = new ModbusTCPClient(ipAddr);

                Program example = new Program();
                example.ReadAllAnalog(mb);
                example.ReadAllDigitalIO(mb);

                // Uncomment below if you have a Mux80 connected
                //example.ReadAllAnalogMux80(mb);
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.ToString());
            }

            if (mb != null)
            {
                // Close the T7 connection
                mb.Close();
            }

            Console.WriteLine("\nPress Enter to exit.");
            Console.ReadLine();
        }

        /// <summary>
        /// Displays "AINxx : Values" readings.
        /// </summary>
        /// <param name="startingAddress">Starting Modbus address of 
        /// readings.</param>
        /// <param name="values">Float analog input (AIN) readings.</param>
        public void DisplayAnalogInputReadings(ushort startingAddress, float[] values)
        {
            for (int i = 0; i < values.Length; i++)
            {
                Console.WriteLine("AIN" + (startingAddress + i * 2) / 2 + " : " + values[i] + " V");
            }
        }

        /// <summary>
        /// Displays all digital I/O readings.
        /// </summary>
        /// <param name="directions">Reading from T7 Modbus address
        /// 2850 (DIO_DIRECTION).</param>
        /// <param name="states">Reading from T7 Modbus address
        /// 2800 (DIO_STATE).</param>
        public void DisplayDigitalIOReadings(uint directions, uint states)
        {
            string fioDirs = "";
            string fioStates = "";
            string eioDirs = "";
            string eioStates = "";
            string cioDirs = "";
            string cioStates = "";
            string mioDirs = "";
            string mioStates = "";

            for (int i = 0; i < 8; i++)
            {
                fioDirs += Convert.ToString((directions >> i) & 1);
                fioStates += Convert.ToString((states >> i) & 1);
            }
            Console.WriteLine("FIO0-FIO7 directions = " + fioDirs + ", states = " + fioStates);

            for (int i = 8; i < 16; i++)
            {
                eioDirs += Convert.ToString((directions >> i) & 1);
                eioStates += Convert.ToString((states >> i) & 1);
            }
            Console.WriteLine("EIO0-EIO7 directions = " + eioDirs + ", states = " + eioStates);

            for (int i = 16; i < 20; i++)
            {
                cioDirs += Convert.ToString((directions >> i) & 1);
                cioStates += Convert.ToString((states >> i) & 1);
            }
            Console.WriteLine("CIO0-CIO3 directions = " + cioDirs + ", states = " + cioStates);

            for (int i = 20; i < 23; i++)
            {
                mioDirs += Convert.ToString((directions >> i) & 1);
                mioStates += Convert.ToString((states >> i) & 1);
            }
            Console.WriteLine("MIO0-MIO2 directions = " + mioDirs + ", states = " + mioStates);
        }

        /// <summary>
        /// Configures range, negative channel, resolution index and settling
        /// time for all analog inputs.
        /// AIN_ALL_RANGE, AIN_ALL_NEGATIVE_CH, AIN_ALL_RESOLUTION_INDEX, and
        /// AIN_ALL_SETTLING_US registers/settings are documented here:
        /// https://labjack.com/support/datasheets/t-series/ain
        /// </summary>
        /// <param name="mb">The ModbusTCPClient to the connected T7 </param>
        /// <param name="range">AIN_ALL_RANGE setting.</param>
        /// <param name="negativeChannel">AIN_ALL_NEGATIVE_CH setting.</param>
        /// <param name="resolutionIndex">AIN_ALL_RESOLUTION_INDEX setting.</param>
        /// <param name="settling">AIN_ALL_SETTLING_US setting.</param>
        public void ConfigureAllAnalog(ModbusTCPClient mb, float range, ushort negativeChannel, ushort resolutionIndex, float settling)
        {
            ushort address;
            ushort uint16Value;
            float float32Value;

            // Configure all analog input ranges.
            address = 43900;  // 43900 = AIN_ALL_RANGE
            float32Value = range;
            mb.Write(address, float32Value);

            // Configure all analog input negative channels.
            address = 43902;  // 43902 = AIN_ALL_NEGATIVE_CH 
            uint16Value = negativeChannel;
            mb.Write(address, uint16Value);

            // Configure all analog input resolution indexes.
            address = 43903;  // 43903 = AIN_ALL_RESOLUTION_INDEX
            uint16Value = resolutionIndex;
            mb.Write(address, uint16Value);

            // Configure all analog input settling times.
            address = 43904;  // 43904 = AIN_ALL_SETTLING_US
            float32Value = settling;
            mb.Write(address, float32Value);
        }

        /// <summary>
        /// Example that configures, reads and displays all the analog
        /// inputs (AIN0-AIN13) on the T7.
        /// Analog inputs (AIN) registers used are documented here:
        /// https://labjack.com/support/datasheets/t-series/ain
        /// </summary>
        /// <param name="mb">The ModbusTCPClient to the connected T7.</param>
        public void ReadAllAnalog(ModbusTCPClient mb)
        {
            Console.WriteLine("Reading AIN0-AIN13.");

            // Configure all analog inputs.
            // Ranges = +/-10 to.
            // Negative Channels = 199 (single-ended)
            // Resolution Indexes = 8
            // Settlings = 0 (auto)
            ConfigureAllAnalog(mb, 10.0f, 199, 8, 0);

            // Read all 14 analog inputs.
            ushort startAddress = 0;  // 0 = AIN0
            float[] ains = new float[14];  // 14 analog input readings
            mb.Read(startAddress, ains);
            DisplayAnalogInputReadings(startAddress, ains);

            Console.WriteLine("");
        }

        /// <summary>
        /// Example that reads and displays all the digital I/O (FIOs, EIOs,
        /// CIOs, MIOs) on the T7.
        /// Digital I/O registers used are documented here:
        /// https://labjack.com/support/datasheets/t-series/digital-io
        /// </summary>
        /// <param name="mb">The ModbusTCPClient to the connected T7.</param>
        public void ReadAllDigitalIO(ModbusTCPClient mb)
        {
            Console.WriteLine("Reading FIOs, EIOs, CIOs and MIO directions and states.");
            ushort address = 0;
            uint directions = 0;
            uint states = 0;

            // Read all digital I/O directions and states.
            address = 2850;  // 2850 = DIO_DIRECTION
            mb.Read(address, ref directions);
            address = 2800;  // 2800 = DIO_STATE
            mb.Read(address, ref states);

            DisplayDigitalIOReadings(directions, states);

            Console.WriteLine("");
        }

        /// <summary>
        /// Example that configures, reads and displays all the analog
        /// inputs on the T7 with a Mux80 (AIN0-AIN3, AIN48-AIN127).
        /// Analog inputs (AIN) registers used are documented here:
        /// https://labjack.com/support/datasheets/t-series/ain
        /// Extended channels AIN48+ are further documented here:
        /// https://labjack.com/support/datasheets/t-series/ain/extended-channels
        /// Mux80 datasheet can be found here:
        /// https://labjack.com/support/datasheets/accessories/mux80
        /// </summary>
        /// <param name="mb">The ModbusTCPClient to the connected T7.</param>
        public void ReadAllAnalogMux80(ModbusTCPClient mb)
        {

            // Many registers to channnels are incorrect. Check with Steve.

            Console.WriteLine("Reading AIN0-AIN3, AIN48-AIN127.");

            // Configure all analog inputs.
            // Ranges = +/-10 to.
            // Negative Channels = 199 (single-ended)
            // Resolution Indexes = 1
            // Settlings = 0 (auto)
            ConfigureAllAnalog(mb, 10.0f, 199, 1, 0);

            //Reading from 84 analog inputs with the Mux80.
            ushort startAddress;
            float[] ains;

            // Read from AIN0-AIN3 on the T7 terminals.
            startAddress = 0;  // 0 = AIN0
            ains = new float[4];  // 4 analog input readings.
            mb.Read(startAddress, ains);
            DisplayAnalogInputReadings(startAddress, ains);

            // Read from AIN48-AIN87 on Mux80.
            startAddress = 96;  // 96 = AIN48
            ains = new float[40];  // 40 analog input readings
            mb.Read(startAddress, ains);
            DisplayAnalogInputReadings(startAddress, ains);

            // Read from AIN88-AIN127 on Mux80.
            startAddress = 176;  // 176 = AIN88
            ains = new float[40];  // 40 analog input readings
            mb.Read(startAddress, ains);
            DisplayAnalogInputReadings(startAddress, ains);

            Console.WriteLine("");
        }
    }
}
