﻿using System;
using LabJack;  //For ModbusTCPClient

namespace UE9ModbusTCPExamples
{
    /// <summary>
    /// Class containing LabJack UE9 Modbus over TCP examples.
    /// </summary>
    class Program
    {
        static void Main(string[] args)
        {
            ModbusTCPClient mb = null;
            string ipAddr = "192.168.1.199";  //Change to your UE9's IP address

            try
            {
                // Connect to the UE9
                Console.WriteLine("Connecting to a device with IP address " + ipAddr + ".");
                mb = new ModbusTCPClient(ipAddr);

                Program example = new Program();
                example.ReadAllAnalog(mb);
                example.ReadAllDigitalIO(mb);

                // Uncomment below if you have a Mux80 connected
                //example.ReadAllAnalogMux80(mb);
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.ToString());
            }

            if (mb != null)
            {
                // Close the UE9 connection
                mb.Close();
            }

            Console.WriteLine("\nPress Enter to exit.");
            Console.ReadLine();
        }

        /// <summary>
        /// Displays "AINxx : Values" readings.
        /// </summary>
        /// <param name="startingAddress">Starting Modbus address of 
        /// readings.</param>
        /// <param name="values">Float analog input (AIN) readings.</param>
        public void DisplayAnalogInputReadings(ushort startingAddress, float[] values)
        {
            for (int i = 0; i < values.Length; i++)
            {
                Console.WriteLine("AIN" + (startingAddress + i * 2) / 2 + " : " + values[i] + " V");
            }
        }

        /// <summary>
        /// Displays all digital I/O readings.
        /// </summary>
        /// <param name="directions">Readings from UE9 Modbus addresses 6750,
        /// 6751, 6752 and 6753 (FIO, EIO, CIO, MIO Directions). 4 element
        /// array.</param>
        /// <param name="states">Readings from UE9 Modbus addresses 6700, 6701,
        /// 6702 and 6703 (FIO, EIO, CIO, MIO States). 4 element array.</param>
        public void DisplayDigitalIOReadings(ushort[] directions, ushort[] states)
        {
            string fioDirs = "";
            string fioStates = "";
            string eioDirs = "";
            string eioStates = "";
            string cioDirs = "";
            string cioStates = "";
            string mioDirs = "";
            string mioStates = "";

            for (int i = 0; i < 8; i++)
            {
                fioDirs += Convert.ToString((directions[0] >> i) & 1);
                fioStates += Convert.ToString((states[0] >> i) & 1);

                eioDirs += Convert.ToString((directions[1] >> i) & 1);
                eioStates += Convert.ToString((states[1] >> i) & 1);

                if (i < 4)
                {
                    cioDirs += Convert.ToString((directions[2] >> i) & 1);
                    cioStates += Convert.ToString((states[2] >> i) & 1);
                }
                if (i < 3)
                {
                    mioDirs += Convert.ToString((directions[3] >> i) & 1);
                    mioStates += Convert.ToString((states[3] >> i) & 1);
                }
            }
            Console.WriteLine("FIO0-FIO7 directions = " + fioDirs + ", states = " + fioStates);
            Console.WriteLine("EIO0-EIO7 directions = " + eioDirs + ", states = " + eioStates);
            Console.WriteLine("CIO0-CIO3 directions = " + cioDirs + ", states = " + cioStates);
            Console.WriteLine("MIO0-MIO2 directions = " + mioDirs + ", states = " + mioStates);
        }

        /// <summary>
        /// Configures resolution, bipolar and gain, settling time for all
        /// analog inputs.
        /// Modbus registers are documented here:
        /// https://labjack.com/support/software/api/modbus/ud-modbus
        /// </summary>
        /// <param name="mb">The ModbusTCPClient to the connected UE9 </param>
        /// <param name="resolution">The resolution (0-18) setting. Resolution
        /// values are documented here:
        /// https://labjack.com/support/datasheets/ue9/hardware-description/ain
        /// </param>
        /// <param name="bipGain">The bipolar and gain setting. BipGain values
        /// are 0x00, 0x01, 0x02, 0x03 or 0x088. For a table of these values and
        /// the mapped Unipolar, Bipolar, and Gain settings, refer to
        /// table 5.3.3-2:
        /// https://labjack.com/support/datasheets/ue9/low-level-function-reference/control-functions/feedback
        /// UE9-Pro hi resolution 18 only supports Gain 1 (BipGain 0x00 and 0x08).
        /// </param>
        /// <param name="settling">The settling time setting. This value
        /// multiplied by 5 microseconds is the additional settling time.</param>
        /// <param name="configMux80">Configure the settings for the Mux80's
        /// AIN48-127.</param>
        public void ConfigureAllAnalog(ModbusTCPClient mb, ushort resolution, ushort bipGain, ushort settling, bool configMux80)
        {
            ushort address;
            ushort[] values = new ushort[14];
            ushort[] mux80Values = new ushort[40];

            // Configure all analog input resolution indexes.
            address = 1000;  // 1000 = AIN Res
            for (int i = 0; i < values.Length; i++)
                values[i] = resolution;  // Resolution
            mb.Write(address, values);
            if (configMux80)
            {
                // Configure for AIN48-127.
                for (int i = 0; i < mux80Values.Length; i++)
                    mux80Values[i] = resolution;  // Resolution
                mb.Write((ushort)(address + 48), mux80Values);
                mb.Write((ushort)(address + 88), mux80Values);
             }

            // Configure all analog input BipGains.
            address = 1500;  // 1500 = AIN Range (BipGain)
            for (int i = 0; i < values.Length; i++)
                values[i] = bipGain;  // BipGain
            mb.Write(address, values);
            if (configMux80)
            {
                // Configure for AIN48-127.
                for (int i = 0; i < mux80Values.Length; i++)
                    mux80Values[i] = bipGain;  // BipGain
                mb.Write((ushort)(address + 48), mux80Values);
                mb.Write((ushort)(address + 88), mux80Values);
            }

            // Configure all settling times.
            address = 2000;  // 2000 = AIN Settling
            for (int i = 0; i < values.Length; i++)
                values[i] = settling;  // Settling
            mb.Write(address, values);
            if (configMux80)
            {
                // Configure for AIN48-127.
                for (int i = 0; i < mux80Values.Length; i++)
                    mux80Values[i] = settling;  // Settling
                mb.Write((ushort)(address + 48), mux80Values);
                mb.Write((ushort)(address + 88), mux80Values);
            }
        }

        /// <summary>
        /// Example that configures, reads and displays all the analog
        /// inputs (AIN0-AIN13) on the UE9.
        /// Analog inputs (AIN) registers and general documentation are here:
        /// https://labjack.com/support/software/api/modbus/ud-modbus
        /// https://labjack.com/support/datasheets/ue9/hardware-description/ain
        /// </summary>
        /// <param name="mb">The ModbusTCPClient to the connected UE9.</param>
        public void ReadAllAnalog(ModbusTCPClient mb)
        {
            ushort startAddress = 0;
            float[] ains;

            Console.WriteLine("Reading AIN0-AIN13.");

            // Configure all analog inputs.
            // Resolutions = 12 (12-bits)
            // bipGain (range) = 0x08 (bipolar 5 V)
            // Settlings = 0 (* 5us)
            ConfigureAllAnalog(mb, 12, 0x08, 0, false);

            // Read first 10 analog inputs, AIN0-AIN9.
            startAddress = 0;  // 0 = AIN0
            ains = new float[10];  // 10 analog input readings
            mb.Read(startAddress, ains);
            DisplayAnalogInputReadings(startAddress, ains);

            // Read next 4 analog inputs, AIN10-13
            startAddress = 20;  // 20 = AIN10
            ains = new float[4];  // 4 analog input readings
            mb.Read(startAddress, ains);
            DisplayAnalogInputReadings(startAddress, ains);

            Console.WriteLine("");
        }

        /// <summary>
        /// Example that reads and displays all the digital I/O (FIOs, EIOs,
        /// CIOs, MIOs) on the UE9.
        /// Digital I/O registers and general documentation can be found here:
        /// https://labjack.com/support/software/api/modbus/ud-modbus
        /// https://labjack.com/support/datasheets/ue9/hardware-description/dio
        /// </summary>
        /// <param name="mb">The ModbusTCPClient to the connected UE9.</param>
        public void ReadAllDigitalIO(ModbusTCPClient mb)
        {
            ushort startAddress = 0;
            ushort[] directions = new ushort[4];
            ushort[] states = new ushort[4];

            // Read all digital I/O directions and states.
            Console.WriteLine("Reading FIOs, EIOs, CIOs and MIO directions and states.");

            // Address 6700 = FIO States
            startAddress = 6700;
            // states = FIO States (6700), EIO States (6701), CIO States (6702), MIO States (6703)
            states = new ushort[4];
            mb.Read(startAddress, states);

            // Address 6750 = FIO Directions
            startAddress = 6750;
            //directions = FIO Dirs (6750), EIO Dirs (6751), CIO Dirs (6752), MIO Dirs (6753)
            directions = new ushort[4];
            mb.Read(startAddress, directions);

            DisplayDigitalIOReadings(directions, states);

            Console.WriteLine("");
        }

        /// <summary>
        /// Example that configures, reads and displays all the analog
        /// inputs on the UE9 with a Mux80 (AIN0-AIN3, AIN48-AIN127).
        /// Analog inputs (AIN) registers and general documentation are here:
        /// https://labjack.com/support/software/api/modbus/ud-modbus
        /// https://labjack.com/support/datasheets/ue9/hardware-description/ain
        /// Mux80 datasheet can be found here:
        /// https://labjack.com/support/datasheets/accessories/mux80
        /// </summary>
        /// <param name="mb">The ModbusTCPClient to the connected UE9.</param>
        public void ReadAllAnalogMux80(ModbusTCPClient mb)
        {
            Console.WriteLine("Reading AIN0-AIN3, AIN48-AIN127.");

            // Configure all analog inputs.
            // Resolutions = 12 (12-bits)
            // bipGain (range) = 0x08 (bipolar 5 V)
            // Settlings = 0 (* 5us)
            ConfigureAllAnalog(mb, 12, 0x08, 0, true);

            //Reading from 84 analog inputs with a Mux80 connected.
            ushort startAddress;
            float[] ains;

            //Read from AIN0-AIN3 on the UE9 terminals.
            startAddress = 0;  // 0 = AIN0
            ains = new float[4];  // 4 analog input readings.
            mb.Read(startAddress, ains);
            DisplayAnalogInputReadings(startAddress, ains);

            // Read from AIN48-AIN127, 10 analog inputs per Modbus read.
            // Keep the analog inputs reads per packet under 13 (26 registers)
            // to keep Modbus packets within 64 bytes.
            ushort startAIN = 48;
            ushort endAIN = 127;
            ushort ainPerRead = 10;
            ains = new float[ainPerRead];

            for (int i = startAIN; i < endAIN; i += ainPerRead)
            {
                startAddress = (ushort)(i * 2);  // i * 2 = AINi
                mb.Read(startAddress, ains);
                DisplayAnalogInputReadings(startAddress, ains);
            }

            Console.WriteLine("");
        }
    }
}
