//----------------------------------------------------------------------
//
//  ue9calibrate.cpp
//
//
//    UE9 User Calibration Program V0.92
//
//    The LabJack UE9 is shipped from the factory fully calibrated.  This
//    program allows you to recalibrate a UE9 at a later time if needed.  A
//    jumper from DAC0 to AIN0 is used to calibrate the analog inputs and 
//    DAC0.  The analog inputs on the UE9 are multiplexed, so the same 
//    calibration constants apply for all channels.  DAC0 is simply used to
//    create different voltages, and the accuracy of DAC0 is not important
//    for the analog input calibration.  A reference meter must be connected
//    to provide the actual voltage on DAC0/AIN0.  The calibration will only 
//    be as accurate as the reference meter.
//
//    A jumper from DAC1 to AIN1 is used to calibrate DAC1, and the jumper
//    from DAC0 to AIN0 is used to calibrate DAC0.  The analog inputs are
//    used to measure the actual voltages (no reference meter required), so
//    the analog inputs must be calibrated before calibrating the DACs.
//    
//    The program calibrates the first found UE9 over USB or specified
//    (IP Address) UE9 over Ethernet.  LabJackUD driver 2.48 or later is 
//    required.   
//
//  support@labjack.com
//  December 9, 2011
//----------------------------------------------------------------------
//

#include <stdio.h>
#include <windows.h>
#include <math.h>
#include "c:\program files\labjack\drivers\LabJackUD.h"
//The project must also know where to find labjackud.lib.  Here we do
//that by putting the lib file in the file view to the left.  The relative
//path stored by Visual Studio might not be the same on your machine, so
//you will probably have to delete the lib in the file view and right-click
//to re-add it from the install location ...\labjack\drivers\labjackud.lib.
//Another option, besides using the .lib, is dynamic linking.  Some
//compilers might not be able to link the .lib, and dynamic linking
//is usually the solution in those cases.


//calibration option names
const char *caliOpts[10] = { "-5V to +5V", 
							 "-5V to +5V (UE9 Pro)", 
							 "0V to +5V", 
							 "0V to +5V (UE9 Pro)",
							 "0V to +2.5V", 
							 "0V to +1.25V",
							 "0V to +0.625V", 
							 "DAC0",
							 "DAC1", 
							 "Temperature" };


//This is a simple error handling function that is called after every UD
//function call.  This function displays the errorcode and string description
//of the error.  It also has a line number input that can be used with the macro 
//__LINE__ to display the line number in the source code that called the error handler.
//Returns -1 if an error is found and 0 if an error was not found.  
int LJErrorHandler (LJ_ERROR lngErrorcode, long lngLineNumber)
{
	char err[255];

	if (lngErrorcode != LJE_NOERROR)
	{
		ErrorToString (lngErrorcode,err);
		printf ("Error number = %d\n",lngErrorcode);
		printf ("Error string = %s\n",err);
		printf ("Source line number = %d\n",lngLineNumber);
		return -1;
	}
	return 0;
}


//Finds the calibration slope and offset values from the voltageReadings 2D array using linear regression.
//The voltageReadings parameter is a 2D array which contains binary voltage and decimal voltage data for a 
//set of data.  Array elements [x][0] and [x][1] should contain binary voltage decimal voltage readings.  
//The [x][0] voltage is the binary AIN voltage read (Gain calibration) or the decimal voltage read from AIN 
//(DAC calibration), and the [x][1] voltage is the decimal voltage read from the user's meter (Gain calibration) 
//or binary DAC voltage (DAC calibration) set.
//The numReadings parameter is the number of binary voltage and decimal voltage readings in the voltageReadings 
//array.
//The slope parameter is an output pointer to the calculated slope from the voltageReadings data.   
//The offset parameter is an output pointer to the calculated offset from the voltageReadings data.       
//void linearRegression(const struct voltage voltageReadings[], int numReadings, double *slope, double *offset) 
void linearRegression (double voltageReadings[][2], int numReadings, double *slope, double *offset) 
{
	int i = 0;
	double sumY = 0, sumX = 0;
	double sumXY = 0, sumXX = 0;
	
	for (i = 0; i < numReadings; i++) 
	{
		sumY = sumY + voltageReadings[i][1];
		sumX = sumX + voltageReadings[i][0];
		sumXY = sumXY + voltageReadings[i][0] * voltageReadings[i][1];
		sumXX = sumXX + voltageReadings[i][0] * voltageReadings[i][0];
	}

	*slope = numReadings * sumXY -  sumX * sumY;
	*slope = *slope / (numReadings * sumXX - sumX * sumX);
	*offset = (sumY - *slope * sumX) / numReadings;
}


//Performs a specified number of reads on the specified AIN channel and averages the the voltages read.
//The lngHandle parameter is the handle to a UE9 device.
//The numOfReads parameter specifies the number of AIN reads to perform and average from.
//The ainChannel specifies the AIN channel to perform all the reads on.
//The slope pointer returns the average voltage read.  This voltage can be either a binary or decimal
//voltage value, depending on how AINs were set up.
//Returns 0 if there were no errors, and -1 if there were.
int ainReadAverage (LJ_HANDLE lngHandle, int numOfReads, int ainChannel, double *voltage)
{
	int lngErrorcode;
	double dblBinaryAINVolt = 0;
	double dblBinaryAINAvg = 0;
	int i = 0;

	for (i = 0; i < numOfReads; i++)
	{
		lngErrorcode = eGet (lngHandle, LJ_ioGET_AIN, ainChannel, &dblBinaryAINVolt, 0);
		if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;
		
		dblBinaryAINAvg += dblBinaryAINVolt;
	}
	
	*voltage = dblBinaryAINAvg/numOfReads;
	return 0;	
}


//This function calculates the new calibration slope and offset values for specified gain.  
//The lngHandle parameter is the handle to a UE9 device.  Call the UD function OpenLabJack to 
//get this handle.
//The isUE9Pro parameter specifies if the UE9 device with the lngHandle is a UE9 pro or not.
//Check your UE9 by making a request with the LJ_ioGET_CONFIG and LJ_chUE9_PRO parameters,
//The option parameter is the calibration option you want to perform.  
//Below is a list of the options:
//  1 - -5V to +5V 
//  2 - -5V to +5V (Pro)
//  3 - 0V to +5V 
//  4 - 0V to +5V (Pro)
//  5 - 0V to +2.5V 
//  6 - 0V to 1.25V 
//  7 - 0V to 0.625V
//The slope and offset pointers return the newly calculated slope and offset. 
//Returns 0 if success, -1 on failure.
int calcGainCalibration (LJ_HANDLE lngHandle, int isUE9Pro, int option, double *slope, double *offset)
{
	LJ_ERROR lngErrorcode = 0;
	int i = 0, j = 0;
	double ainResolution = 17; 
	double dblBinaryDACVolt = 0;
	double dblDecimalDACVolt;				//Decimal DAC voltage 
	long lngAINRange;						//UD constant for specified AIN rainge
	double dblRange = 0;					//Voltage range value for specified gain
	double dblMeterVolt = 0;				//Voltage from user's voltage meter
	double dblBinaryAINAvg = 0;				//Binary AIN voltage average
	double dblDecimalAINAvg = 0;			//Decimal AIN voltage average
	long curAINChannel;						//Current AIN channel to read from.  This is set as ainChannel or
											//136 if dblMinPercentDACVolt is 0.
	const long dacChannel = 0;				//The DAC channel to use
	const long ainChannel = 0;				//The AIN channel used to read from DAC
	const int numAINReadings = 32;			//Number of AIN readings to take the average from
	const double dblMinPercentDACVolt = 0;	//The minimum volt % the DAC will be set to.  If set to 0, 
											//then reading from 0V from GND)
	const double dblMaxPercentDACVolt = 80; //The maximum volt %  DAC will be set to. 
	const int numVoltSteps = 5;				//Number of voltage steps to calculate slope and offset from
	double dblMinDecimalDACVolt = 0;		//The minumum voltage DAC will be set to 
											//(calculated from dblMinPercentDACVolt)
	double dblMaxDecimalDACVolt = 0;		//The maximum voltage DAC will be set to 
											//(calculated from dblMaxPercentDACVolt)
	double dblIncDecimalDACVolt = 0;		//The incremental voltage the DAC will be set to when being 
											//set from dblMinDecimalDACVolt V to dblMaxDecimalDACVolt V 
											//in numVoltSteps steps
	int verify = 0;
	int isUE9ProError = 0;	
	double voltReadings[numVoltSteps][2];	//array that will contain the decimal and binary voltages used 
											//to calculate the new slope and offset
	double adblBinaryDACVolts[numVoltSteps];//array that will contain binary DAC voltages (uncalibrated)

	//setting max and min DAC voltages and AIN volt ranges
	switch (option)
	{
		case 1:  
		case 2:
			//if UE9 is not pro, then option 2 is not available
			if(option == 2 && isUE9Pro == 0)
				isUE9ProError = 1;

			dblMaxDecimalDACVolt = 5.0 * dblMaxPercentDACVolt / 100.0;
			dblMinDecimalDACVolt = 5.0 * dblMinPercentDACVolt / 100.0;
			lngAINRange = LJ_rgBIP5V;
			dblRange = 10;
			break;
		case 3: 
		case 4:
			//if UE9 is not pro, then option 4 is not available
			if(option == 4 && isUE9Pro == 0)
				isUE9ProError = 1;

			dblMaxDecimalDACVolt = 5.0 * dblMaxPercentDACVolt / 100.0;
			dblMinDecimalDACVolt = 5.0 * dblMinPercentDACVolt / 100.0;
			lngAINRange = LJ_rgUNI5V;
			dblRange = 5;
			break;
		case 5:
			dblMaxDecimalDACVolt = 2.5 * dblMaxPercentDACVolt / 100.0;
			dblMinDecimalDACVolt = 2.5 * dblMinPercentDACVolt / 100.0;
			lngAINRange = LJ_rgUNI2P5V;
			dblRange = 2.5;
			break;
		case 6: 
			dblMaxDecimalDACVolt = 1.25 * dblMaxPercentDACVolt / 100.0;
			dblMinDecimalDACVolt = 1.25 * dblMinPercentDACVolt / 100.0;
			lngAINRange = LJ_rgUNI1P25V;
			dblRange = 1.25;
			break;
		case 7:
			dblMaxDecimalDACVolt = 0.625 * dblMaxPercentDACVolt / 100.0;
			dblMinDecimalDACVolt = 0.625 * dblMinPercentDACVolt / 100.0;
			lngAINRange = LJ_rgUNIP625V;
			dblRange = 0.625;
			break;
		default:
			printf ("Invalid option\n\n");
			return -1;
	}

	if (isUE9ProError != 0)
	{
		printf ("Cannot calibrate for this option,  This option is only available for \na UE9 Pro\n\n");
		return -1;
	}

	//setting incremental DAC voltage
	dblIncDecimalDACVolt = (dblMaxDecimalDACVolt - dblMinDecimalDACVolt)/(numVoltSteps - 1);

	//If UE9 is a Pro, and option is 2 or 4, then set AIN resolution to 18
	if (isUE9Pro == 1 && (option == 2 || option == 4) )
		ainResolution = 18;

	//Configuring AIN resolution as 17-bit or 18-bit (UE9 Pro only) 
	lngErrorcode = AddRequest (lngHandle, LJ_ioPUT_CONFIG, LJ_chAIN_RESOLUTION, ainResolution, 0, 0);
	if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;

	//Setting the AIN range for AIN, which we are going to use for calibration  
	lngErrorcode = AddRequest (lngHandle, LJ_ioPUT_AIN_RANGE, 0, lngAINRange, 0, 0);
	if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;
	
	//Setting the AIN range for GRN, which we are going to use for calibration  
	lngErrorcode = AddRequest (lngHandle, LJ_ioPUT_AIN_RANGE, 136, lngAINRange, 0, 0);
	if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;

	//Enabling a DAC just in case both DAC0 and DAC1 are disabled
	lngErrorcode = AddRequest (lngHandle, LJ_ioPUT_DAC_ENABLE, dacChannel, 1, 0, 0);
	if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;

	//Setting DAC to accept binary voltage (uncalibrated voltage)
	lngErrorcode = AddRequest (lngHandle, LJ_ioPUT_CONFIG, LJ_chDAC_BINARY, 1, 0, 0);
	if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;

	//Setting AIN readings as binary (uncalibrated voltage)
	lngErrorcode = AddRequest (lngHandle, LJ_ioPUT_CONFIG, LJ_chAIN_BINARY, 1, 0, 0);
	if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;

	//sending requests
	lngErrorcode = GoOne (lngHandle);
	if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;
	
	//Setting uncalibrated DAC voltages and reading from uncalibrated AIN and volt meter
	for (i = 0; i < numVoltSteps; i++)
	{
		//Setting DAC voltage
		if (i == 0)
		{
			//voltReadings[i].decimalVoltage = 0.0;
			if (dblMinDecimalDACVolt == 0)
				voltReadings[i][1] = 0.0;
			else
				dblDecimalDACVolt = dblMinDecimalDACVolt;
		}
		else if (i == numVoltSteps - 1)
			dblDecimalDACVolt = dblMaxDecimalDACVolt;
		else
			dblDecimalDACVolt = dblIncDecimalDACVolt*i;				

		if( (i == 0 && dblMinDecimalDACVolt !=  0) || i != 0)
		{
			printf ("Setting DAC0 to %.3f (uncalibrated)\n", dblDecimalDACVolt);
			dblBinaryDACVolt = (dblDecimalDACVolt / 4.86) * 4096.0;
			//Setting DAC voltage on UE9
			lngErrorcode = ePut (lngHandle, LJ_ioPUT_DAC, dacChannel, dblBinaryDACVolt, 0);
			if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;
				
			//Getting the voltage from the user's meter
			printf ("Please enter the voltage your meter reads: ");
			fflush (stdin);
			if (scanf ("%lf",&dblMeterVolt) != 1)
			{
				printf ("Invalid meter voltage.  Aborting calibration.\n");
				return -1;
			}
				
			voltReadings[i][1] = dblMeterVolt;
		}

		adblBinaryDACVolts[i] = dblBinaryDACVolt;


		if (i == 0 && dblMinDecimalDACVolt ==  0)	//reading voltage from channel 136 (GND)
			curAINChannel = 136;
		else		//reading voltage from AIN
			curAINChannel = ainChannel;

		//Performing 32 reads on AIN which the set DAC should be connected to and 
		//calculating average voltage read from DAC
		if ( (lngErrorcode = ainReadAverage (lngHandle, numAINReadings, curAINChannel, &dblBinaryAINAvg)) == -1) 
			return -1;
		
		if (option == 2 || option == 4)
			voltReadings[i][0] = dblBinaryAINAvg/256;
		else
			voltReadings[i][0] = dblBinaryAINAvg;
	}

	//Performing linear regression on AIN voltages collected to calculate new slope and offset
	linearRegression (voltReadings, numVoltSteps, slope, offset);

	printf ("\nNew calibration values: \nSlope = %.10lf, Offset = %.10lf\n\n", *slope, *offset);

	//Displaying error of AIN readings with new calibration constants with regards to meter values
	printf ("Error when using new calibration values: \n");
	for (i = 0; i < numVoltSteps; i++)
	{
		dblDecimalDACVolt = (adblBinaryDACVolts[i] * 4.86) / 4096.0;
		printf (" %lf V : %.10lf V error, %.10lf %% error\n", dblDecimalDACVolt, 
			fabs (voltReadings[i][1] - (voltReadings[i][0] * (*slope) + *offset)), 
			fabs (voltReadings[i][1] - (voltReadings[i][0] * (*slope) + *offset)) / dblRange * 100.0);
	}
	printf("\n");

	//Setting AIN readings as decimal voltage
	lngErrorcode = ePut (lngHandle, LJ_ioPUT_CONFIG, LJ_chAIN_BINARY, 0, 0);
	if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;

	//Getting and displaying error of AIN readings with current calibration values with regards to meter values
	printf ("Error when using current calibration values: \n");
	for (i = 0; i < numVoltSteps; i++)
	{
		if (i == 0 && dblMinDecimalDACVolt ==  0)
		{
			//reading voltage from channel 136 (GND)
			curAINChannel = 136;
		}
		else		
		{
			//reading voltage from AIN
			lngErrorcode = ePut (lngHandle, LJ_ioPUT_DAC, dacChannel, adblBinaryDACVolts[i], 0);
			if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;

			curAINChannel = ainChannel;
		}

		if( (lngErrorcode = ainReadAverage (lngHandle, numAINReadings, curAINChannel, &dblDecimalAINAvg)) == -1) 
			return -1;

		dblDecimalDACVolt = (adblBinaryDACVolts[i] * 4.86) / 4096.0;
		printf (" %lf V : %.10lf V error, %.10lf %% error\n", dblDecimalDACVolt, 
			fabs (voltReadings[i][1] - dblDecimalAINAvg), 
			fabs (voltReadings[i][1] - dblDecimalAINAvg) / dblRange * 100.0);
	}
	printf ("\n");

	//Setting DAC to accept decimal voltage
	lngErrorcode = ePut (lngHandle, LJ_ioPUT_CONFIG, LJ_chDAC_BINARY, 0, 0);
	if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;

	return 0;
}


//This function calculates the new calibration slope and offset values for the specified DAC.  
//The lngHandle parameter is the handle to a UE9 device.  Call the UD function OpenLabJack to 
//get this handle.
//The dacNumber parameter specifies which dac you want to calibrate.
//The ainSlope and ainOffset are the slope and offset to use to calculate AIN voltage.  You
//should pass the slope and offset for the 0V to +5V range (gain = 1).
//The slope and offset pointers return the newly calculated slope and offset. 
//Returns 0 if success, -1 on failure.
int calcDACCalibration (LJ_HANDLE lngHandle, int dacNumber, double ainSlope, double ainOffset, double *slope, double *offset)
{
	LJ_ERROR lngErrorcode = 0;
	int i = 0, j = 0;
	double dblBinaryDACVolt = 0;			//Binary AIN voltage
	double dblBinaryAINAvg = 0;				//Binary AIN voltage average
	long ainChannel = 0;					//The AIN channel used to read DAC output
	const int numAINReadings = 32;			//Number of AIN readings to take the average from
	const double dblMinPercentDACVolt = 10; //The minimum % voltage DAC will be set to.
	const double dblMaxPercentDACVolt = 90; //The maximum % voltage DAC will be set to.
	const int numVoltSteps = 5;				//Number of voltage steps to calculate slope and offset from
	double dblMinDecimalDACVolt = 0;		//The minumum voltage DAC will be set to 
											//(calculated from dblMinPercentDACVolt)
	double dblMaxDecimalDACVolt = 0;		//The maximum voltage DAC will be set to 
											//(calculated from dblMaxPercentDACVolt)
	double dblIncDecimalDACVolt = 0;		//The incremental voltage the DAC will be set to when being 
											//set from dblMinDecimalDACVolt V to dblMaxDecimalDACVolt V 
											//in numVoltSteps steps
	double voltReadings[numVoltSteps][2];   //array that will contain the decimal and binary voltages used 
											//to calculate the new slope and offset
	double adblDecimalDACVolts[numVoltSteps]; //array that will contain Decimal DAC voltages 

	if (dacNumber != 0 && dacNumber != 1)
	{
		printf ("Invalid DAC number\n");
		return -1;
	}

	ainChannel = dacNumber;		//read DAC0 output from AIN0 and DAC1 output from AIN1

	//setting min, max and incremental DAC voltages
	dblMaxDecimalDACVolt = 5.0 * dblMaxPercentDACVolt / 100.0;
	dblMinDecimalDACVolt = 5.0 * dblMinPercentDACVolt / 100.0;
	dblIncDecimalDACVolt = (dblMaxDecimalDACVolt - dblMinDecimalDACVolt)/(numVoltSteps - 1);
	
	//Configuring AIN resolution as 17-bit
	lngErrorcode = AddRequest (lngHandle, LJ_ioPUT_CONFIG, LJ_chAIN_RESOLUTION, 17, 0, 0);
	if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;

	//Setting the AIN range to unipolar 5V, which we are going to use for calibration  
	lngErrorcode = AddRequest (lngHandle, LJ_ioPUT_AIN_RANGE, ainChannel, LJ_rgUNI5V, 0, 0);
	if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;
	
	//Enabling a DAC just in case both DAC0 and DAC1 are disabled
	lngErrorcode = AddRequest (lngHandle, LJ_ioPUT_DAC_ENABLE, dacNumber, 1, 0, 0);
	if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;

	//Setting DAC to accept binary voltage
	lngErrorcode = AddRequest (lngHandle, LJ_ioPUT_CONFIG, LJ_chDAC_BINARY, 1, 0, 0);
	if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;

	//Setting AIN to read binary voltage
	lngErrorcode = AddRequest (lngHandle, LJ_ioPUT_CONFIG, LJ_chAIN_BINARY, 1, 0, 0);
	if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;

	//Sending requests
	lngErrorcode = GoOne (lngHandle);
	if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;
	
	
	//Setting uncalibrated DAC voltages and reading from calibrated AIN
	for (i = 0; i < numVoltSteps; i++)
	{
		//Setting DAC voltage
		if (i == 0)
			adblDecimalDACVolts[i] = dblMinDecimalDACVolt;
		else if(i == numVoltSteps - 1)
			adblDecimalDACVolts[i] = dblMaxDecimalDACVolt;
		else
			adblDecimalDACVolts[i] = dblIncDecimalDACVolt*i;
		
		printf ("Setting DAC%d to %.3lf (uncalibrated)\n", dacNumber, adblDecimalDACVolts[i]);
		dblBinaryDACVolt = (adblDecimalDACVolts[i] / 4.86) * 4096.0;

		//setting DAC voltage on UE9
		lngErrorcode = ePut (lngHandle, LJ_ioPUT_DAC, dacNumber, dblBinaryDACVolt, 0);
		if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;
			
		voltReadings[i][1] = dblBinaryDACVolt;

		//Performing 32 reads on AIN which the set DAC should be connected to and 
		//calculating average voltage read from DAC
		if( (lngErrorcode = ainReadAverage (lngHandle, numAINReadings, ainChannel, &dblBinaryAINAvg)) == -1) 
			return -1;

		voltReadings[i][0] = dblBinaryAINAvg*ainSlope + ainOffset;
	}

	//Performing linear regression on DAC and AIN voltages collected to calculate new slope and offset
	linearRegression (voltReadings, numVoltSteps, slope, offset);
		
	printf ("\nNew calibration values: \nSlope = %.10lf, Offset = %.10lf\n\n", *slope, *offset);
	

	//Getting and displaying DAC error with new and current calibration values
	for(j = 0; j < 2; j++)
	{
		if(j == 1)
		{
			//Setting DAC to accept decimal voltage
			lngErrorcode = ePut (lngHandle, LJ_ioPUT_CONFIG, LJ_chDAC_BINARY, 0, 0);
			if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;

			printf ("Error when using current calibration values: \n");
		}
		else
			printf ("Error when using new calibration values: \n");

		for(i = 0; i < numVoltSteps; i++)
		{		
			

			if(j == 1)
				lngErrorcode = ePut (lngHandle, LJ_ioPUT_DAC, dacNumber, adblDecimalDACVolts[i], 0);
			else
			{
				dblBinaryDACVolt = (adblDecimalDACVolts[i] * (*slope)) + *offset;
				lngErrorcode = ePut (lngHandle, LJ_ioPUT_DAC, dacNumber, dblBinaryDACVolt, 0);
			}
			if(LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;
			
			//Performing 32 reads on AIN which the set DAC should be connected to and 
			//calculating average voltage read from DAC
			if( (lngErrorcode = ainReadAverage (lngHandle, numAINReadings, ainChannel, &dblBinaryAINAvg)) == -1) 
				return -1;

			//Displaying error of AIN readings with new calibrations with regards to read AIN values
			printf (" %lf V : %.10lf V error, %.10lf %% error\n", adblDecimalDACVolts[i], 
				fabs (adblDecimalDACVolts[i] - (dblBinaryAINAvg * ainSlope + ainOffset)), 
				fabs (adblDecimalDACVolts[i] - (dblBinaryAINAvg * ainSlope + ainOffset)) / 5 * 100.0);
			
		}
		printf("\n");
	}

	//Setting AIN to read decimal voltage
	lngErrorcode = AddRequest (lngHandle, LJ_ioPUT_CONFIG, LJ_chAIN_BINARY, 0, 0, 0);
	if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;

	return 0;
}


//This function calculates the new calibration slope value the internal temperature sensor. 
//The lngHandle parameter is the handle to a UE9 device.  Call the UD function OpenLabJack to 
//get this handle.
//The slope pointer returns the newly calculated slope. 
//Returns 0 if success, -1 on failure.
int calcTempCalibration (LJ_HANDLE lngHandle, double *slope)
{
	LJ_ERROR lngErrorcode = 0;
	int i = 0, j = 0;
	double dblUserTemp = 0;				//temperature from user in degrees C
	double dblTempAvg = 0;				//average temperature
	const int numTempReadings = 32;		//The number of temperature readings to perform
	
	//Configuring AIN resolution as 17-bit 
	lngErrorcode = AddRequest (lngHandle, LJ_ioPUT_CONFIG, LJ_chAIN_RESOLUTION, 17, 0, 0);
	if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;

	//Setting AIN readings as binary
	lngErrorcode = AddRequest (lngHandle, LJ_ioPUT_CONFIG, LJ_chAIN_BINARY, 1, 0, 0);
	if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;

	//sending requests
	lngErrorcode = GoOne (lngHandle);
	if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return -1;
	
	//Getting temperature from user
	printf ("Please enter the current temperature degrees Celsius: ");
	fflush (stdin);
	if (scanf ("%lf",&dblUserTemp) != 1)
	{
		printf ("Invalid meter voltage.  Aborting calibration.\n");
		return -1;
	}

	//Performing 32 binary reads from the temp sensor (channel 133) and calculating the average
	if ( (lngErrorcode = ainReadAverage(lngHandle, numTempReadings, 133, &dblTempAvg)) == -1) 
			return -1;
	
	//converting C temp to Kelvins and calculating new slope for temperature
	*slope = (dblUserTemp + 273.15)/dblTempAvg;
	printf ("\nNew calibration value: \nSlope = %.10lf\n\n", *slope);//dblSlope);

	//Getting and displaying temperature error with new and current calibration values with regard
	//to user inputted temperature.
	for (j = 0; j < 2; j++)
	{
		if(j == 1)
		{
			//Setting AIN readings as decimal (calibrated)
			lngErrorcode = ePut (lngHandle, LJ_ioPUT_CONFIG, LJ_chAIN_BINARY, 0, 0);
			if (LJErrorHandler(lngErrorcode, __LINE__) == -1) return -1;

			printf ("Error when using current calibration value: \n");
		}
		else
			printf ("Error when using new calibration value: \n");

		//Performing 32 reads from the temp sensor (channel 133) and calculating the average
		if( (lngErrorcode = ainReadAverage (lngHandle, numTempReadings, 133, &dblTempAvg)) == -1) 
			return -1;

		if(j != 1)
			dblTempAvg = dblTempAvg * (*slope);  //converting binary read to decimal

		printf (" %.10lf degrees C error\n\n", fabs (dblUserTemp - (dblTempAvg - 273.15)), 0); 
	}
	return 0;
}


//This function determines what to calibrate and reading/writing calibration constants to the UE9's memory.
//The option parameter specifies what to calibrate.  Options 1 - 7 will calibrate gains, 8-9 calibrates the DACs
//and 10 calibrates the temperature.  Below is the list of the options:
//1 - -5V to +5V 
//2 - -5V to +5V (UE9 Pro) 
//3 - 0V to +5V 
//4 - 0V to +5V (UE9 Pro)
//5 - 0V to +2.5V 
//6 - 0V to +1.25V,
//7 - 0V to +0.625V 
//8 - DAC0
//9 - DAC1 
//10 - Temperature
void calibrate (int option, int cOption, char addr[])
{
	LJ_HANDLE lngHandle;
	LJ_ERROR lngErrorcode = 0;
	double slope, offset;
	double isUE9Pro = 0;
	int i = 1, j;
	char acceptCali;		
	int acceptAnswer;		
	double adblCaliMem[128];	//Array that contains the calibration memory 
	int writeToMem = 0;			//Stays at 0 if not writing out to UE9 memory.  Incremnts to >=1 if writing to memory. 

    if(cOption == 2)
    {
        printf("Ethernet\n");
        //Opening specified UE9 over Ethernet
	    lngErrorcode = OpenLabJack (LJ_dtUE9, LJ_ctETHERNET, addr, 0, &lngHandle);
	    if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return;
    }
    else
    {
        printf("USB\n");
        //Opening first found UE9 over USB
	    lngErrorcode = OpenLabJack (LJ_dtUE9, LJ_ctUSB, "1", 1, &lngHandle);
	    if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return;
    }
	
	//Checking if UE9 is a Pro
	lngErrorcode = eGet (lngHandle, LJ_ioGET_CONFIG, LJ_chUE9_PRO, &isUE9Pro, 0);
	if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return; 

	for (j = 0; j < 128; j++)
		adblCaliMem[j] = 0;

	//Getting calibration constants array from UE9's memory
	lngErrorcode = eGet (lngHandle, LJ_ioGET_CAL_CONSTANTS, 0, 0, (long)adblCaliMem);
	if (LJErrorHandler (lngErrorcode, __LINE__) == -1) return; 

	/*//Prints the UE9's calibration constants from memory
	for (int k = 0; k < 128; k++)
		printf("%.10lf\n", adblCaliMem[k]);
	*/

	if (option != 11)
		i = option;

	for (i; i < 11; i++)
	{
		acceptAnswer = 0;
	
		//skip over this loop iteration since UE9 is not a Pro and options 2 and 4 do not apply
		if(option == 11 && isUE9Pro == 0 && (i == 2 || i == 4))
			continue;

		if(i <= 7)	//Gets new calibration values for a gain
		{
			printf ("\nCalibrating %s range :\n", caliOpts[i-1]);		
			
			if(calcGainCalibration (lngHandle, (int)isUE9Pro, i, &slope, &offset) == -1)
				return; 
		}
			
		if (i == 8 || i == 9) //Gets new calibration values for a DAC
		{
			printf ("\nCalibrating %s :\n", caliOpts[i-1]);
		
			if (calcDACCalibration (lngHandle, i - 8, adblCaliMem[0], adblCaliMem[1], &slope, &offset) == -1)
				return;
		}		
		
		if (i == 10)	//Gets new calibration value for internal temperature
		{
			if (calcTempCalibration (lngHandle, &slope) == -1)
				return;
		}

		//Loops until the user gives an acceptable answer of 'y' or 'n'
		while (!acceptAnswer)
		{
			//Asking if user wants to accept the new calibration value
			if (i == 10)
				printf ("Do you want to accept the new calibration slope value? (y or n) : ");
			else
				printf ("Do you want to accept the new calibration slope and offset values? (y or n) : ");
			
			fflush (stdin);

			//Getting answer from user
			if (scanf ("%c",&acceptCali) != 1 || (acceptCali != 'y' && acceptCali != 'n'))
			{
				//Invalid answer. will loop and ask again
				printf ("Invalid answer.\n");
			}
			else
			{
				//Answer from user was acceptable.  
				acceptAnswer = 1;
	
				if (acceptCali == 'y')
				{
					//User accepted new calibration constants.  Putting new calibration constants 
					//into the array that will be written to the UE9's memory.
					writeToMem++;					
					
					switch (i)
					{
						case 1:
							adblCaliMem[16] = slope;
							adblCaliMem[17] = offset;
							break;
						case 2:
							adblCaliMem[64] = slope;
							adblCaliMem[65] = offset;
							break;
						case 3:
							adblCaliMem[0] = slope;
							adblCaliMem[1] = offset;
							break;
						case 4:
							adblCaliMem[48] = slope;
							adblCaliMem[49] = offset;
							break;
						case 5:
							adblCaliMem[2] = slope;
							adblCaliMem[3] = offset;
							break;
						case 6:
							adblCaliMem[4] = slope;
							adblCaliMem[5] = offset;
							break;
						case 7:
							adblCaliMem[6] = slope;
							adblCaliMem[7] = offset;
							break;
						case 8:
							adblCaliMem[32] = slope;
							adblCaliMem[33] = offset;
							break;
						case 9:
							adblCaliMem[34] = slope;
							adblCaliMem[35] = offset;
							break;
						case 10:
							adblCaliMem[36] = slope;
							adblCaliMem[37] = offset;
							break;
						default:
							return;
					}
				}
				else
				{
					//User did not accept new calibration constants
					printf ("\nNot accepting the new calibration values.");  
					if(option == 11)
						printf ("  Previously accepted calibration values will still be applied.");
					printf("\n\n");
				}
			}

		}

		if (option != 11)
			i = 11; //breaking for loop
	}

	if (writeToMem >= 1)
	{
		//Writing new calibration constants array to UE9's memory.  For this request, the 
		//channel and value parameters are bytes 6 and 7 of the EraseMem low-level function.
		lngErrorcode = ePut (lngHandle, LJ_ioPUT_CAL_CONSTANTS, 0x4C, 0x4A, (long)adblCaliMem);
		LJErrorHandler (lngErrorcode, __LINE__);

		printf("\nDone calibrating.  Calibration values for the UE9 were written to memory.\n"); 
	}
}

void exit_prog()
{
	char ch;

	fflush (stdin);
	printf ("\nExiting program.  Press the Enter key to quit.");
	ch = getchar ();
	exit(0);
}

int main (int argc, char* argv[])
{
	int option = 0;
	char ch;
	double udVersion;
    int cOption = 1;  //Connection option
    char addr[16] = "0";  //IP Address
    int test;

	fflush (stdin);

	udVersion = GetDriverVersion ();

	if (udVersion < 2.48)
	{
		printf ("The current LabJackUD driver version is %.2lf.  Driver version 2.48 or later\n", udVersion);
		printf ("is required for the UE9 User Calibration Program.  For the latest driver,\n");
		printf ("go to http://labjack.com/support/windows-ud.\n");
		exit_prog();
	}

	printf ("UE9 User Calibration Program V0.92\nDecember 9th, 2011\nsupport@labjack.com\n\n");
	printf ("The LabJack UE9 is shipped from the factory fully calibrated.  This\n");
	printf ("program allows you to recalibrate a UE9 at a later time if needed.  A\n");
	printf ("jumper from DAC0 to AIN0 is used to calibrate the analog inputs and\n"); 
	printf ("DAC0.  The analog inputs on the UE9 are multiplexed, so the same\n"); 
	printf ("calibration constants apply for all channels.  DAC0 is simply used to\n");
	printf ("create different voltages, and the accuracy of DAC0 is not important\n");
	printf ("for the analog input calibration.  A reference meter must be connected\n");
	printf ("to provide the	actual voltage on DAC0/AIN0.  The calibration will only\n"); 
	printf ("be as accurate as the reference meter.\n\n");

	printf ("A jumper from DAC1 to AIN1 is used to calibrate DAC1, and the jumper\n");
	printf ("from DAC0 to AIN0 is used to calibrate DAC0.  The analog inputs are\n");
	printf ("used to measure the actual voltages (no reference meter required), so\n");
	printf ("the analog inputs must be calibrated before calibrating the DACs.\n\n");
	
	printf ("Press the Enter key to continue.\n");
	ch = getchar ();

   	while (1)
    {
        //Connection types menu
	    printf ("\nChoose your connection type: \n");
        printf ("1 - USB, first found\n");
        printf ("2 - Ethernet\n");
        printf ("\nEnter your choice : ");

        fflush (stdin);
        if(scanf ("%d",&cOption) != 1 || cOption < 1 || cOption > 2)
        {
            printf("Invalid connection type\n");
        }
        else
        {
            if(cOption == 2)
            {
                //Ethernet type selected
                while(1)
                {
                    printf("Enter your UE9's IP address: ");

                    fflush (stdin);
                    if( (test = scanf ("%15s",addr)) != 1)
                    {
                        printf("Invalid IP address string\n");
                    }
                    else
                    {
                        break;
                    }
                }
            }
            break;
        }
    }

	while (option != 12)
	{
		//Calibration options menu
		printf ("\nUE9 calibration options: \n");
		printf ("1 - %s\n", caliOpts[0]);
		printf ("2 - %s\n", caliOpts[1]);
		printf ("3 - %s\n", caliOpts[2]);
		printf ("4 - %s\n", caliOpts[3]);
		printf ("5 - %s\n", caliOpts[4]);
		printf ("6 - %s\n", caliOpts[5]);
		printf ("7 - %s\n", caliOpts[6]);
		printf ("8 - %s\n", caliOpts[7]);
		printf ("9 - %s\n", caliOpts[8]);
		printf ("10 - %s\n", caliOpts[9]);
		printf ("11 - All\n");
		printf ("12 - Exit program\n");
		printf ("\nEnter your choice : ");
		
		fflush (stdin);
		if(scanf ("%d",&option) != 1 || option < 1 || option > 12)
		{
			printf ("Invalid choice\n");
			option = 0;
		}
		else
		{
			if(option != 12)
				calibrate (option, cOption, addr);
		}
	}

	exit_prog();
	return 0;
}
