/******************************************************************************************************************************/
// AISample
//
// Reads the voltage from 1,2, or 4 analog inputs.  Also controls/reads the 4 IO Ports.  Execution time for this function is 50 
// ms or less
//
// MATLAB Syntax:
//		[voltages stateIO overVoltage errorcode Idnum] = AISample(idnum, demo, stateIO, updateIO, ledOn, numChannels, channels, 
//																gains, diableCal)
// Input:
//		idnum - Local Id, serial number, or -1 for first LJ found
//		demo - 0 for normal operation, >0 for demo mode.  Demo mode allows this function to be called    without a LabJack.
//		stateIO - Output states for IO3 - IO4 (Decimal value).
//		updateeIO - If > 0, sate values will be written.  Otherwise, just a ead is performed.
//		ledOn - if > 0, the LabJack LED is turned on.
//		numChannels - Number of analog input channels to read (1,2, or 4).  
//		channels -  Array of channel commands with at least numChannels elements. Each channel command is 0-7.for single-ended, 
//					or 8-11 for differential.
//		gains - Array of gain commands with at least numChannel elements.  Gain commands are 0=1, 1=2, ..., 7=20.  This 
//				amplification is available only for differential channels. disableCal - If >0, voltages returned will be raw 
//				readings that are note corrected using calibration constants.
//
// Output:
//		Voltages - Aray where voltage readings are returned.  Same length as numChannels.
//		stateIO = Returns input states for IO0 - IO3.  Decimal value ranging from 0 - 15.
//		OverVoltage = if > 0 over voltage has been detected on one of the selected analog inputs
//		errorcode = LabJack error codes or 0 for no error.
//		idnum = Local ID number of Labjack, or -1 if no LabJack is found.
//
/********************************************************************************************************************************/

#include "mex.h"
#include <wtypes.h>
#include "ljackuw.h"

void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[] )
{
  double  inp1,inp2,inp3,inp4,inp5,inp6,*inp7,*inp8,inp9,*out1,*out2,*out3,*out4,*out5;
  int i, m, n;
  long idnum, demo, stateIO, updateIO, ledOn, numChannels, disableCal, errorcode, overvoltage;
  long channels[4], gains[4];
  float Voltages[4] = {0,0,0,0};


  /* Check for proper number of arguments. */
  if(nrhs != 9) 
  {
    mexErrMsgTxt("Input Error --> (idnum, demo, stateIO, updateIO, ledOn, numChannels, channels, gains, disableCal)");
  } 
  else if(nlhs > 5) 
  {
    mexErrMsgTxt("Output Error --> [voltages stateIO overVoltage errorcode Idnum]");
  }

  /* Get dimensions of inputs */
  //for (i=0;i<=9;i++)
  //{
//	 m = mxGetM(prhs[i]);
//	 n = mxGetN(prhs[i]);
	 
//	 if(m*n != 1 & i==)
//	 {
//		 mexErrMsgTxt("Inputs must to be scalar");
//	 }
//  }
  
  /* Retrive Matlab input values from function call. */
  inp1 = mxGetScalar(prhs[0]);
  inp2 = mxGetScalar(prhs[1]);
  inp3 = mxGetScalar(prhs[2]);
  inp4 = mxGetScalar(prhs[3]);
  inp5 = mxGetScalar(prhs[4]);
  inp6 = mxGetScalar(prhs[5]);
  inp7 = mxGetPr(prhs[6]);
  inp8 = mxGetPr(prhs[7]);
  inp9 = mxGetScalar(prhs[8]);

  /* Create matrix for the return arguments. */
  plhs[0] = mxCreateDoubleMatrix(inp6,1, mxREAL);
  plhs[1] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[2] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[3] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[4] = mxCreateDoubleMatrix(1,1, mxREAL);

  /* Get pointer to real part of return argument */
  out1 = mxGetPr(plhs[0]);
  out2 = mxGetPr(plhs[1]);
  out3 = mxGetPr(plhs[2]);
  out4 = mxGetPr(plhs[3]);
  out5 = mxGetPr(plhs[4]);
  
  /* Recast Matlab inputs */
  idnum = (long)inp1;
  demo = (long)inp2;
  stateIO = (long)inp3;
  updateIO = (long)inp4;
  ledOn = (long)inp5;
  numChannels = (long)inp6;
  disableCal = (long)inp9;



  for(i=0;i<=3;i++)
	{
		channels[i] = (long)*inp7;	//Fill channel array
		gains[i] = (long)*inp8;		//Fill gain array
		inp7++;						//Increment pointer to channel array from Matlab
		inp8++;						//Increment pointer to gains array from matlab
	}

  /* Call the AISample function. */
  errorcode = AISample(&idnum, demo, &stateIO, updateIO, ledOn, numChannels, &channels[0], &gains[0], disableCal, &overvoltage, &Voltages[0]);
  
  /* Matlab outputs equal to returned LJ values */
  for(i=0;i<=3;i++)
	{
	out1[i] = (double)Voltages[i];	//Fill voltage output array
	}
  out2[0] = (double)stateIO;
  out3[0] = (double)overvoltage;
  out4[0] = (double)errorcode;
  out5[0] = (double)idnum;
}
