/******************************************************************************************************************************/
// AIStreamStart
//
// Starts hardware timed continuous acquisition where data is sampled and stored in the LabJack RAM buffer, and can be 
// simultaneously transferred out of the RAM buffer to the PC application.  A call to this function should be followed by 
// periodic calls to AIStreamRead , and eventually a call to AIStream Clear.
//
// If the LED is enabled (ledOn>0), it will be toggled every 40 samples during acquisition and turn on when the stream operation 
// stops.
//
// MATLAB Syntax:
//		[scanRate errorcode idnum] = AIStreamStart(idnum, demo, stateIOin, updateIO, ledOn, numChannels, channels, gains, 
//													scanRate, disableCal, readCount)
//
// Inputs:
//		idnum - Local ID, serial number, or -1 for first LabJack found.
//		demo - Send 0 for normal operation, >0 for demo mode.  Demo mode allows this function to be called without a LabJack.
//		stateIOin - Output state for IO0 - IO3.
//		updateIO - If>0, state values will be written.  Otherwise, just a read is performed.
//		ledOn - If>0, the LabJack LED is turned on.
//		numChannels - Number of analog input channels to read (1,2, or 4).  If readCount is >0, numChannels should be 4.
//		channels - Array of channel commands with at least numChannels elements.  Each channel command is 0-7 for single-0ended, 
//				   or 8-11 for differential.
//		gains - Array of gain commands with at least numChannels elements.  Gain commands are 0=1,1=2,...7=20.  This amplification 
//			    is only available for differential channels.
//		scanRate - Scans acquired per second.  A scan is a reading from every channel (1,2,or 4).  The sample rate 
//			      (scanRate*numChannels) must be between 200 and 1200.
//		disableCal - If>0, voltages returned will be raw readings that are not corrected using calibration constants. 
//		readCount  - If>0, the current count (CNT) is returned instead of the 2nd, 3rd, and 4th analog input channels.  2nd 
//					 channel is bits 0-11,  3rd channel is bists 12-23.  4th channel is bits 24-31.  This feature was added 
//					 to the LabJack U12 starting with firmware version 1.03, and this input has no effect with earlier firmware 
//					 versions. 
//
// Outputs:
//		scanRate - Returns the actual scan rate, which due to clock resolution is not always exactly the same as the desired 
//				   scan rate.
//		errorcode = LabJack error codes or 0 for no error.
//		idnum = Local ID number of LabJack, or -1 if no LabJack is found.
//
// 
/********************************************************************************************************************************/

#include "mex.h"
#include <wtypes.h>
#include "ljackuw.h"

void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[] )
{
  double  inp1,inp2,inp3,inp4,inp5,inp6,*inp7,*inp8,inp9, inp10, inp11,*out1,*out2,*out3;
  int i, m, n;
  long idnum, demo, stateIOin, updateIO, ledOn, numChannels, reserved1, readCount, disableCal, errorcode, overvoltage;
  long channels[4], gains[4];
  float scanRate;


  /* Check for proper number of arguments. */
  if(nrhs != 11) 
  {
    mexErrMsgTxt("Input Error --> (idnum, demo, stateIOin, updateIO, ledOn, numChannels, channels, gains, scanRate, disableCal, readCount)");
  } 
  else if(nlhs > 3) 
  {
    mexErrMsgTxt("Output Error --> [scanRate Errorcode Idnum]");
  }

  /* Get dimensions of inputs */
  //for (i=0;i<=9;i++)
  //{
//	 m = mxGetM(prhs[i]);
//	 n = mxGetN(prhs[i]);
	 
//	 if(m*n != 1 & i==)
//	 {
//		 mexErrMsgTxt("Inputs must to be scalar");
//	 }
//  }
  
  /* Retrive Matlab input values from function call. */
  inp1 = mxGetScalar(prhs[0]);
  inp2 = mxGetScalar(prhs[1]);
  inp3 = mxGetScalar(prhs[2]);
  inp4 = mxGetScalar(prhs[3]);
  inp5 = mxGetScalar(prhs[4]);
  inp6 = mxGetScalar(prhs[5]);
  inp7 = mxGetPr(prhs[6]);
  inp8 = mxGetPr(prhs[7]);
  inp9 = mxGetScalar(prhs[8]);
  inp10 = mxGetScalar(prhs[9]);
  inp11 = mxGetScalar(prhs[10]);

  /* Create matrix for the return arguments. */
  plhs[0] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[1] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[2] = mxCreateDoubleMatrix(1,1, mxREAL);

  /* Get pointer to real part of return argument */
  out1 = mxGetPr(plhs[0]);
  out2 = mxGetPr(plhs[1]);
  out3 = mxGetPr(plhs[2]);
  
  /* Recast Matlab inputs */
  idnum = (long)inp1;
  demo = (long)inp2;
  stateIOin = (long)inp3;
  updateIO = (long)inp4;
  ledOn = (long)inp5;
  numChannels = (long)inp6;
  scanRate = (float)inp9;
  disableCal = (long)inp10;
  reserved1 = 0;
  readCount = (long)inp11;


  for(i=0;i<=3;i++)
	{
		channels[i] = (long)*inp7;	//Fill channel array
		gains[i] = (long)*inp8;		//Fill gain array
		inp7++;						//Increment pointer to channel array from Matlab
		inp8++;						//Increment pointer to gains array from matlab
	}

  /* Call the EAnalogin function. */
  errorcode = AIStreamStart(&idnum, demo, stateIOin, updateIO, ledOn, numChannels, &channels[0], &gains[0], &scanRate, disableCal, reserved1, readCount);
  
  /* Matlab outputs equal to returned LJ values */
  out1[0] = (double)scanRate;
  out2[0] = (double)errorcode;
  out3[0] = (double)idnum;
}