/********************************************************************************************************************************/
// BitsToVolts
//
// Converts a 12-bit (0-4095) binary value into a Labjack voltage.  
// Volts=((2*Bits*Vmax/4096)-Vmax)/Gain where Vmax=10 for SE, 20 for Diff.
//
// MATLAB Syntax:
//		[volts Errorcode] = bitstovolts(chnum, chgain, bits)
//
// Input:
//		chnum - channel idex.  0-7 SE, 8-11 Diff.
//		chgain - Gain index. 0=1, 1=2, .., 7=20.
//		bits - Binary value from 0-4096.
// Output:
//		volts - Voltage.
// 
/********************************************************************************************************************************/

#include "mex.h"
#include <wtypes.h>
#include "ljackuw.h"

void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[] )
{
  double  inp1,inp2,inp3,*out1,*out2;
  int i, m, n;
  long errorcode, chnum, chgain, bits;
  float volts = 0;

  /* Check for proper number of arguments. */
  if(nrhs != 3) 
  {
    mexErrMsgTxt("Input Error --> (chnum, chgain, bits)");
  } 
  else if(nlhs > 2) 
  {
    mexErrMsgTxt("Output Error --> [volts errorcode]");
  }

  /* Get dimensions of inputs */
  for (i=0;i<=2;i++)
  {
	 m = mxGetM(prhs[i]);
	 n = mxGetN(prhs[i]);
	 
	 if(m*n != 1)
	 {
		 mexErrMsgTxt("Inputs must to be scalar");
	 }
  }
  
  /* Retrive Matlab input values from function call. */
  inp1 = mxGetScalar(prhs[0]);
  inp2 = mxGetScalar(prhs[1]);
  inp3 = mxGetScalar(prhs[2]);

  /* Create matrix for the return arguments. */
  plhs[0] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[1] = mxCreateDoubleMatrix(1,1, mxREAL);

  /* Get pointer to real part of return argument */
  out1 = mxGetPr(plhs[0]);
  out2 = mxGetPr(plhs[1]);
  
  /* Recast Matlab inputs */
  chnum = (long)inp1;
  chgain = (long)inp2;
  bits = (long)inp3;

  /* Call the BitsToVolts function. */
  errorcode = BitsToVolts(chnum, chgain, bits, &volts);
  
  /* Set Matlab outputs equal to returned LJ values */
  out1[0] = (double)volts;
  out1[1] = (double)errorcode;
}
