/*************************************************************************************************/
// EDigitalIn
//
// Easy function.  This is a simplified version of DigitalIO that reads the state of one digital 
// input.  Also configures the requested pin to input and leave it that way.  Execution time for 
// this function is 50 ms or less.  
//
// Note that this is a simplified version of the lower level function DigitalIO, which operates on 
// all 20 digital lines.  The DLL (ljackuw) attempts to keep track of the current direction and 
// output state of all lines, so that this easy function can operate on a single line without changing 
// the others.  When the DLL is first loaded, though, it does not know the direction and state for the 
// lines and assumes all directions are inputs and output stares are low.
//
// MATLAB Syntax:  
//		[state Errorcode Idnum] = EDigitalIn(Idnum, demo, channel, readD)
//
// Inputs:
//		idnum - Local Id, serial number, or -1 for first LabJack found
//		demo - 0 for normal operation, >0 for demo mode.  Demo mode allows this function to be called     without a LabJack.
//		channel - Line to read.  0-3 for IO or 0-15 for D.
//		readD - If > 0, a D line is read as opposed to an IO line.
//
// Outputs:
//		state - The selected line is TRUE/Set if > 0.  FALSE/Clear if 0.
//		errorcode - LabJack error code or 0 for no error.
//		idnum - Returns the local ID or -1 if no LabJack is found.
//
/**************************************************************************************************/

#include "mex.h"
#include <wtypes.h>
#include "ljackuw.h"

void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[] )
{
  double  inp1,inp2,inp3,inp4,*out1,*out2,*out3;
  int i, m, n;
  long idnum, demo, channel, readD, state, errorcode;

  /* Check for proper number of arguments. */
  if(nrhs != 4) 
  {
    mexErrMsgTxt("Input Error --> (Idnum, demo, channel, readD)");
  } 
  else if(nlhs > 3) 
  {
    mexErrMsgTxt("Output Error --> [state, Errorcode, Idnum]");
  }

  /* Get dimensions of inputs */
  for (i=0;i<=3;i++)
  {
	 m = mxGetM(prhs[i]);
	 n = mxGetN(prhs[i]);
	 
	 if(m*n != 1)
	 {
		 mexErrMsgTxt("Inputs must to be scalar");
	 }
  }
  
  /* Create matrix for the return arguments. */
  plhs[0] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[1] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[2] = mxCreateDoubleMatrix(1,1, mxREAL);
  
  /* Retrive Matlab input values from function call. */
  inp1 = mxGetScalar(prhs[0]);
  inp2 = mxGetScalar(prhs[1]);
  inp3 = mxGetScalar(prhs[2]);
  inp4 = mxGetScalar(prhs[3]);

  /* Get pointer to real part of return argument */
  out1 = mxGetPr(plhs[0]);
  out2 = mxGetPr(plhs[1]);
  out3 = mxGetPr(plhs[2]);
  
  /* Recast Matlab inputs */
  idnum = (long)inp1;
  demo = (long)inp2;
  channel = (long)inp3;
  readD = (long)inp4;

  /* Call the EDigitalIn function. */
  errorcode = EDigitalIn(&idnum, demo, channel, readD, &state);
  
  /* Matlab outputs equal to returned LJ values */
  out1[0] = (double)state;
  out2[0] = (double)errorcode;
  out3[0] = (double)idnum;

}
