/********************************************************************************************************************************/
// ReadMem
//
// Reads 4 bytes from a specified address in the LabJack's nonvolatile memory.  Execution time for this function is 50 ms or less.
//
// MATLAB Syntax:
//		[data3 data2 data1 data0 idnum errorcode] = GetErrorString(idnum address)
//
// Input:
//		idnum - Local ID, serial number, or -1 for first LabJack found.
//		address - Starting address of data to read (0-8188)	
//
// Output:
//		data3 - Byte at address.
//		data2 - Byte at address+1.
//		data1 - Byte at address+2.
//		data0 - Byte at address+3.
//		idnum - Returns the local ID or -1 if no LabJack is found.
//		errorcode - LabJack error codes or 0 for no error.
//
//*******************************************************************************************************************************/

#include "mex.h"
#include "matrix.h"
#include <wtypes.h>
#include "ljackuw.h"

void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[] )
{
  double  inp1, inp2, *out1, *out2, *out3, *out4, *out5, *out6;
  int i, m, n;
  long address, data3, data2, data1, data0, idnum, errorcode;

  /* Check for proper number of arguments. */
  if(nrhs != 2) 
  {
    mexErrMsgTxt("Input Error --> (idnum address)");
  } 
  else if(nlhs > 6) 
  {
    mexErrMsgTxt("Output Error --> [data3 data2 data1 data0 idnum errorcode]");
  }

  /* Get dimensions of inputs */
  for (i=0;i<=1;i++)
  {
	 m = mxGetM(prhs[i]);
	 n = mxGetN(prhs[i]);
	 
	 if(m*n != 1)
	 {
		 mexErrMsgTxt("Input must to be scalar");
	 }
  }

  /* Retrive Matlab input values from function call. */
  inp1 = mxGetScalar(prhs[0]);
  inp2 = mxGetScalar(prhs[1]);

  /* Create matrix for the return arguments based upon the number of LabJacks present */
  plhs[0]= mxCreateDoubleMatrix(1,1,mxREAL);
  plhs[1]= mxCreateDoubleMatrix(1,1,mxREAL);
  plhs[2]= mxCreateDoubleMatrix(1,1,mxREAL);
  plhs[3]= mxCreateDoubleMatrix(1,1,mxREAL);
  plhs[4]= mxCreateDoubleMatrix(1,1,mxREAL);
  plhs[5]= mxCreateDoubleMatrix(1,1,mxREAL);

  /* Get pointer to real part of return argument */
  out1 = mxGetPr(plhs[0]);
  out2 = mxGetPr(plhs[1]);
  out3 = mxGetPr(plhs[2]);
  out4 = mxGetPr(plhs[3]);
  out5 = mxGetPr(plhs[4]);
  out6 = mxGetPr(plhs[5]);

  /* Recast Matlab inputs */
  idnum = long(inp1);
  address = long(inp2);

  /* Call the ReadMem function. */
  errorcode = ReadMem(&idnum, address, &data3, &data2, &data1, &data0);

  /* Matlab outputs equal to returned LJ values */
  out1[0] = (double)data3;
  out2[0] = (double)data2;
  out3[0] = (double)data1;
  out4[0] = (double)data0;
  out5[0] = (double)errorcode;
  out6[0] = (double)idnum;

}
