/******************************************************************************************************************************/
// SHTComm:		Low-level public function to send and receive up to 4 bytes
//				to from an SHT1X sensor.  Data rate is about 2 kbps
//				with firmware V1.09 or higher (hardware communication).
//				If firmware is less than V1.09, or TRUE is passed for
//				softComm, data rate is about 20 bps.
//
//				DATA = IO0
//				SCK = IO1
//
//				The EI-1050 has an extra enable line that allows multiple
//				probes to be connected at the same time using only the one
//				line for DATA and one line for SCK.  This function does not
//				control the enable line.
//
//				This function automatically configures IO0 has an input
//				and IO1 as an output.
//
//				Note that internally this function operates on the state and
//				direction of IO0 and IO1, and to operate on any of the IO
//				lines the LabJack must operate on all 4.  The DLL keeps track
//				of the current direction and output state of all lines, so that
//				this function can operate on IO0 and IO1 without changing
//				IO2 and IO3.  When the DLL is first loaded,
//				though, it does not know the direction and state of
//				the lines and assumes all directions are input and
//				output states are low.
//
//MATLAB Syntax:
//		[errorcode idnum datarx] = SHTComm(idnum, softComm, waitMeas, serialReset,
//										 dataRate, numWrite, numRead, datatx, datarx)
//
//	Returns:	LabJack errorcodes or 0 for no error (I32).
//
//	Inputs:		*idnum		-Local ID, Serial Number, or -1 for first
//							 found (I32).
//				softComm	-If >0, forces software based communication.  Otherwise
//							 software communication is only used if the LabJack U12
//							 firmware version is less than V1.09.
//				waitMeas	-If >0, this is a T or RH measurement request.
//				serialReset	-If >0, a serial reset is issued before sending and
//							 receiving bytes.
//				dataRate	-0=no extra delay (default),1=medium delay,2=max delay.
//				numWrite	-Number of bytes to write (0-4,I32).
//				numRead		-Number of bytes to read (0-4,I32).
//				*datatx		-Array of 0-4 bytes to send.  Make sure you pass at least
//							 numWrite number of bytes (U8).
//	Outputs:	*idnum		-Returns the Local ID or -1 if no LabJack is
//							 found (I32).
//				*datarx		-Returns 0-4 read bytes as determined by numRead (U8).
//
//	Time:		About 20 ms plus SHT1X measurement time for hardware comm.
//				Default measurement time is 210 ms for temp and 55 ms for RH.
//				About 2 s per measurement for software comm.
/********************************************************************************************************************************/

#include "mex.h"
#include <wtypes.h>
#include "ljackuw.h"

void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[] )
{
  double  inp1,inp2,inp3,inp4,inp5,inp6,inp7,*inp8,*inp9;
  double  *out1,*out2,*out3;
  int i, m, n;
  long  idnum, softComm, waitMeas, serialReset, dataRate, numWrite, numRead, errorcode;
  unsigned char datatx[4] = {0};
  unsigned char datarx[4] = {0};

  /* Check for proper number of arguments. */
  if(nrhs != 9) 
  {
    mexErrMsgTxt("Input Error --> (idnum, softComm, waitMeas, serialReset, dataRate, numWrite, numRead, datatx, datarx)");
  } 
  else if(nlhs > 3) 
  {
    mexErrMsgTxt("Output Error --> [errorcode idnum datarx]");
  }

  /* Get dimensions of inputs */
  for (i=0;i<7;i++)
  {
	 m = mxGetM(prhs[i]);
	 n = mxGetN(prhs[i]);
	 
	 if(m*n != 1)
	 {
		 mexErrMsgTxt("Inputs must be scalar:");
	 }
  }
	
  /* Retrive Matlab input values from function call. */
  inp1 = mxGetScalar(prhs[0]);
  inp2 = mxGetScalar(prhs[1]);
  inp3 = mxGetScalar(prhs[2]);
  inp4 = mxGetScalar(prhs[3]);
  inp5 = mxGetScalar(prhs[4]);
  inp6 = mxGetScalar(prhs[5]);
  inp7 = mxGetScalar(prhs[6]);
  inp8 = mxGetPr(prhs[7]);
  inp9 = mxGetPr(prhs[8]);

  /* Create matrix for the return arguments. */
  plhs[0] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[1] = mxCreateDoubleMatrix(1,1, mxREAL);
  plhs[2] = mxCreateDoubleMatrix(4,1, mxREAL);

  /* Get pointer to real part of return argument */
  out1 = mxGetPr(plhs[0]);
  out2 = mxGetPr(plhs[1]);
  out3 = mxGetPr(plhs[2]);

  /* Recast Matlab inputs */
  idnum = (long)inp1;
  softComm = (long)inp2;
  waitMeas = (long)inp3;
  serialReset = (long)inp4;
  dataRate = (long)inp5;
  numWrite = (long)inp6;
  numRead = (long)inp7;

  for(i=0;i<inp6;i++)
	{
		datatx[i] = (long)*inp8;	//Fill data array
		inp8++;						//Increment pointer to data array from Matlab
	}
  for(i=0;i<inp7;i++)
	{
		datarx[i] = (long)*inp9;	//Fill data array
		inp9++;						//Increment pointer to data array from Matlab
	}



  /* Call the SHTComm function. */
 errorcode = SHTComm(&idnum, softComm, waitMeas, serialReset, 
					dataRate, numWrite, numRead, datatx, datarx);

  /* Matlab outputs equal to returned LJ values */

	  out1[0] = (double)errorcode;
	  out2[0] = (long)idnum;	
	  
	for(i=0;i<inp7;i++)
 		{
			out3[i] = (long)datarx[i];	//Fill data output array
		}


}
