package com.teravation.labjack;
/**
 * This file is part of LabJackJava.  Copyright 2003-2004, Teravation.  All rights reserved. 
 * 
 * LabJackJava is free software; you can redistribute it and/or modify it under the terms of the GNU
 * General Public License as published by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 * 
 * LabJackJava is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; 
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with LabJackJava;
 * if not, write to:
 *    Free Software Foundation, Inc.
 *    59 Temple Place, Suite 330
 *    Boston, MA  02111-1307  USA
 * 
 * In addition, as a special exception, Teravation gives permission to link the code of this program with
 * the LabJack C driver (or with modified versions of the LabJack C driver that use the same license),
 * and distribute linked combinations including the two.  You must obey the GNU General Public License in all
 * respects for all of the code used other than the LabJack C driver.  If you modify this file, you may 
 * extend this exception to your version of the file, but you are not obligated to do so.  If you do not wish to
 * do so, delete this exception statement from your version.
 * 
 */

/**
 * This is the "platform-specific" implementation of the {@link LabJack LabJack} interface, and is dependent on 
 * the underlying JNI library, which accesses the platform-specific implementation of the LabJack C driver.
 *
 * <p />Copyright &copy; 2003-2004 <a href="http://www.teravation.com">Teravation</a>. All rights reserved.
 * 
 * @created (12/30/2002 2:25:49 PM)
 * @version 4.0
 * @author Chris Reigrut 
 */
public class LabJackDriver extends LabJackImpl implements LabJack {
	private int pulseB1 = 0;
	private int pulseC1 = 0;
	private int pulseB2 = 0;
	private int pulseC2 = 0;

	static private Class thisClass; // For synchronization of API calls
	static {
		thisClass = LabJackDriver.class;
		// Load the underlying JNI driver
		//    Windows:  ljackj.dll
		//    Mac OSX:  libljackj.jnilib
		//    Linux:    libljackj.so
		System.loadLibrary("ljackj");
	}

	static private LabJack[] labjacks;

	/**
	 * Standard no-argument constructor
	 */
	private LabJackDriver() {
	}

	/**
	 * Constructor, taking the LabJack's localid, serial number and product id
	 * 
	 * Creation date: (1/10/2003 1:29:45 PM)
	 * @author: Chris Reigrut
	 * 
	 * @param localId int
	 * @param serialNumber int
	 * @param productId int
	 */
	private LabJackDriver(int localId, int serialNumber, int productId) {
		this();
		this.setLocalId(localId);
		this.setSerialNumber(serialNumber);
		this.setProductId(productId);
	}

	/**
	 * Insert the method's description here.
	 * @created (2/7/2003 12:01:08 PM)
	 * @param idnum int
	 * @param stateIO long
	 * @param updateIO long
	 * @param ledOn long
	 * @param numChannels long
	 * @param channels long[]
	 * @param gains long[]
	 * @param disableCal long
	 * @return float[]
	 * @exception com.teravation.labjack.LabJackException The exception description.
	 */
	private native float[] apiAISample(int idnum, long stateIO, long updateIO, long ledOn, long numChannels, long[] channels, long[] gains, long disableCal) throws LabJackException;

	/**
	 * Insert the method's description here.
	 * @created (2/7/2003 12:01:08 PM)
	 * @param idnum int
	 * @param stateIOIn long
	 * @param updateIO long
	 * @param ledOn long
	 * @param numChannels long
	 * @param channels long[]
	 * @param gains long[]
	 * @param scanRate float
	 * @param disableCal long
	 * @param triggerIO long
	 * @param triggerState long
	 * @param numScans long
	 * @param timeout long
	 * @param transferMode long
	 * @param scanObject LabJackScan
	 * @exception com.teravation.labjack.LabJackException The exception description.
	 */
	private native float apiAIBurst(
		int idnum,
		long stateIOIn,
		long updateIO,
		long ledOn,
		long numChannels,
		long[] channels,
		long[] gains,
		float scanRate,
		long disableCal,
		long triggerIO,
		long triggerState,
		long numScans,
		long timeout,
		long transferMode,
		LabJackScan scanObject)
		throws LabJackException;

	/**
	 * Insert the method's description here.
	 * Creation date: (1/31/2003 1:33:10 PM)
	 *
	 * @param idnum int
	 * @param trisD long
	 * @param trisIO long
	 * @param stateD long
	 * @param stateIO long
	 * @param updateDigital long
	 * @param resetCounter long
	 * @param analogOut0 float
	 * @param analogOut1 float
	 * @return long
	 * @exception com.teravation.labjack.LabJackException The exception description.
	 */
	private native long apiAOUpdate(int idnum, long trisD, long trisIO, long stateD, long stateIO, long updateDigital, long resetCounter, float analogOut0, float analogOut1) throws LabJackException;

	/**
	 * Insert the method's description here.
	 * 
	 * Creation date: (1/10/2003 2:57:58 PM)
	 * @author: 
	 * 
	 * @param idnum int
	 * @param channel int
	 * @param gain int
	 * @return float
	 * @exception com.teravation.labjack.LabJackException The exception description.
	 */
	private native float apiEAnalogIn(int idnum, int channel, int gain) throws LabJackException;

	/**
	 * Insert the method's description here.
	 * 
	 * Creation date: (1/10/2003 2:57:58 PM)
	 * @author: 
	 * 
	 * @param idnum int
	 * @param analogOut0 float
	 * @param analogOut1 float
	 * @exception com.teravation.labjack.LabJackException The exception description.
	 */
	private native void apiEAnalogOut(int idnum, float analogOut0, float analogOut1) throws LabJackException;

	/**
	 * Insert the method's description here.
	 * 
	 * Creation date: (1/20/2003 12:50:06 PM)
	 * @author: 
	 * 
	 * @param idnum int
	 * @param reset boolean
	 * @return long
	 * @exception com.teravation.labjack.LabJackException The exception description.
	 */
	private native long apiECount(int idnum, boolean resetCounter) throws LabJackException;

	/**
	 * Insert the method's description here.
	 * 
	 * Creation date: (1/10/2003 2:13:56 PM)
	 * @author: 
	 * 
	 * @param idnum int
	 * @param channel int
	 * @param readD boolean
	 * @return boolean
	 * @exception com.teravation.labjack.LabJackException The exception description.
	 */
	private native boolean apiEDigitalIn(int idnum, int channel, boolean readD) throws LabJackException;

	/**
	 * Insert the method's description here.
	 * 
	 * Creation date: (1/10/2003 1:51:17 PM)
	 * @author: 
	 * 
	 * @param idnum int
	 * @param channel int
	 * @param writeD boolean
	 * @param state boolean
	 * @exception com.teravation.labjack.LabJackException The exception description.
	 */
	private native void apiEDigitalOut(int idnum, int channel, boolean writeD, boolean state) throws LabJackException;

	/**
	 * Insert the method's description here.
	 * 
	 * Creation date: (1/10/2003 11:39:27 AM)
	 * @author: 
	 * 
	 * @return com.teravation.labjack.LabJack[]
	 * @throws com.teravation.labjack.LabJackException
	 */
	private native static LabJack[] apiListAll() throws LabJackException;

	/**
	 * Insert the method's description here.
	 * @created (2/6/2003 3:47:08 PM)
	
	 * @param idnum int
	 */
	private native void apiSHT1X(int idnum);

	/**
	 * Insert the method's description here.
	 * Creation date: (3/3/2004 2:18:08 PM)
	 *
	 * @param idnum int
	 * @param lowFirst int
	 * @param bitSelect int
	 * @param numPulses int
	 * @param timeB1 int
	 * @param timeC1 int
	 * @param timeB2 int
	 * @param timeC2 int
	 * @exception com.teravation.labjack.LabJackException The exception description.
	 */
	private native void apiPulseOut(int idnum, int lowFirst, int bitSelect, int numPulses, int timeB1, int timeC1, int timeB2, int timeC2) throws LabJackException;

	/**
	 * Insert the method's description here.
	 * Creation date: (3/3/2004 2:18:08 PM)
	 *
	 * @param idnum int
	 * @param lowFirst int
	 * @param bitSelect int
	 * @param numPulses int
	 * @param timeB1 int
	 * @param timeC1 int
	 * @param timeB2 int
	 * @param timeC2 int
	 * @exception com.teravation.labjack.LabJackException The exception description.
	 */
	private native void apiPulseOutStart(int idnum, int lowFirst, int bitSelect, int numPulses, int timeB1, int timeC1, int timeB2, int timeC2) throws LabJackException;

	/**
	 * Insert the method's description here.
	 * Creation date: (3/3/2004 2:18:08 PM)
	 *
	 * @param idnum int
	 * @param timeoutMS long
	 * @exception com.teravation.labjack.LabJackException The exception description.
	 */
	private native void apiPulseOutFinish(int idnum, int timeoutMS) throws LabJackException;

	/**
	 * Insert the method's description here.
	 * Creation date: (3/3/2004 2:18:08 PM)
	 *
	 * @param idnum int
	 * @exception com.teravation.labjack.LabJackException The exception description.
	 */
	private native void apiReEnum(int idnum) throws LabJackException;

	/**
	 * Insert the method's description here.
	 * Creation date: (3/3/2004 2:18:08 PM)
	 *
	 * @param idnum int
	 * @exception com.teravation.labjack.LabJackException The exception description.
	 */
	private native void apiReset(int idnum) throws LabJackException;

	/**
	 * Insert the method's description here.
	 * Creation date: (3/3/2004 2:18:08 PM)
	 *
	 * @param idnum int
	 * @exception com.teravation.labjack.LabJackException The exception description.
	 */
	private native float apiGetFirmwareVersion(int idnum) throws LabJackException;

	/**
	 * Insert the method's description here.
	 * Creation date: (3/3/2004 2:18:08 PM)
	 *
	 * @exception com.teravation.labjack.LabJackException The exception description.
	 */
	private static native float apiGetDriverVersion() throws LabJackException;

	/**
	 * Insert the method's description here.
	 * Creation date: (3/3/2004 2:18:08 PM)
	 *
	 * @param idnum int
	 * @param localID int
	 * @exception com.teravation.labjack.LabJackException The exception description.
	 */
	private native void apiLocalID(int idnum, int localID) throws LabJackException;

	/**
	 * Insert the method's description here.
	 * @created (3/14/2003 11:48:15 PM)
	 * @param idnum int
	 * @param stateIO long
	 * @param updateIO long
	 * @param ledOn long
	 * @param numChannels long
	 * @param channels long[]
	 * @param gains long[]
	 * @param disableCal long
	 * @return float
	 * @exception com.teravation.labjack.LabJackException The exception description.
	 */
	private native float apiAIStreamStart(int idnum, long stateIO, long updateIO, long ledOn, long numChannels, long[] channels, long[] gains, float scanRate, long disableCal, long readCount) throws LabJackException;

	/**
	 * Insert the method's description here.
	 * @created (2/7/2003 12:01:08 PM)
	 * @param idnum int
	 * @param numScans long
	 * @param timeout long
	 * @param scanObject LabJackScan
	 * @exception com.teravation.labjack.LabJackException The exception description.
	 */
	private native void apiAIStreamRead(int idnum, long numScans, long timeout, LabJackScan scanObject) throws LabJackException;

	/**
	 * Insert the method's description here.
	 * @created (3/14/2003 11:48:15 PM)
	 * @param idnum int
	 * @exception com.teravation.labjack.LabJackException The exception description.
	 */
	private native void apiAIStreamClear(int idnum) throws LabJackException;

	/**
	 * Insert the method's description here.
	 * 
	 * Creation date: (1/10/2003 11:36:52 AM)
	 * @author: 
	 * 
	 * @return com.teravation.labjack.LabJack[]
	 * @throws com.teravation.labjack.LabJackException
	 */
	static LabJack[] getAll() throws LabJackException {
		synchronized (thisClass) {
			return apiListAll();
		}
	}
	
  /**
	* Insert the method's description here.
	* 
	* Creation date: (1/10/2003 11:36:52 AM)
	* @author: 
	* 
	* @return com.teravation.labjack.LabJack[]
	* @throws com.teravation.labjack.LabJackException
	*/
  static float getDriverVersion() throws LabJackException {
	  synchronized (thisClass) {
		  return apiGetDriverVersion();
	  }
  }

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#updateAI(int)
	 */
	public void updateAI(int channel) throws LabJackException {
		if (channel < 0 || channel >= LabJackConstants.ANALOG_SINGLE_INPUT_CHANNELS) {
			throw new LabJackException(21);
		}

		synchronized (this.getClass()) {
			float value = this.apiEAnalogIn(this.getSerialNumber(), channel, 0);
			this.analogInputValues[channel] = value;
		}
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#()
	 */
	public void updateAll() throws LabJackException {
		this.updateAll(false);
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#updateAll(boolean)
	 */
	public void updateAll(boolean resetCounter) throws LabJackException {
		this.updateAllAOsAndDigitals(resetCounter);
		this.updateAllAIs();
		this.updateAllDifferentialAIs();
		this.updateEnvironmentals();
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#updateAllAIs()
	 */
	public void updateAllAIs() throws LabJackException {
		this.updateLowAIs();
		this.updateHighAIs();
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#updateAllAOsAndDigitals()
	 */
	public void updateAllAOsAndDigitals() throws LabJackException {
		this.updateAllAOsAndDigitals(false);
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#updateAllAOsAndDigitals(boolean)
	 */
	public void updateAllAOsAndDigitals(boolean resetCounter) throws LabJackException {
		synchronized (thisClass) {
			long allDIOValues =
				apiAOUpdate(getSerialNumber(), digitalDDirections, digitalIODirections, digitalDValues, digitalIOValues, 1, (resetCounter ? 1 : 0), analogOutputValues[0], analogOutputValues[1]);
			digitalIOValues = (digitalIOValues & digitalIODirections) | (~digitalIODirections & (allDIOValues & ((1 << LabJackConstants.IO_CHANNELS) - 1)));
			digitalDValues = (digitalDValues & digitalDDirections) | (~digitalDDirections & (allDIOValues >> LabJackConstants.IO_CHANNELS));
		}
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#updateAllDifferentialAIs()
	 */
	public void updateAllDifferentialAIs() throws LabJackException {
		synchronized (thisClass) {
			long[] channels = { 8, 9, 10, 11 };
			float[] voltages = this.apiAISample(getSerialNumber(), 0, 0, 1, 4, channels, this.analogInputGains, 0);
			for (int i = 0; i < channels.length; i++) {
				this.analogInputValues[(int) channels[i]] = voltages[i];
			}
		}
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#updateAO(int)
	 */
	public void updateAO(int channel) throws LabJackException {
		if (channel < 0 || channel >= LabJackConstants.ANALOG_OUTPUT_CHANNELS) {
			throw new LabJackException(21);
		}

		synchronized (thisClass) {
			this.apiEAnalogOut(this.getSerialNumber(), this.analogOutputValues[0], this.analogOutputValues[1]);
		}
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#updateCounter()
	 */
	public void updateCounter() throws LabJackException {
		this.updateCounter(false);
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#updateCounter(boolean)
	 */
	public void updateCounter(boolean resetCounter) throws LabJackException {
		synchronized (thisClass) {
			this.counter = this.apiECount(this.getSerialNumber(), resetCounter);
		}
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#updateD(int)
	 */
	public void updateD(int channel) throws LabJackException {
		if (channel < 0 || channel >= LabJackConstants.D_CHANNELS) {
			throw new LabJackException(21);
		}

		synchronized (thisClass) {
			if (isDForOutput(channel)) {
				this.apiEDigitalOut(this.getSerialNumber(), channel, true, this.getDigitalDValue(channel));
			} else {
				boolean value = this.apiEDigitalIn(this.getSerialNumber(), channel, true);
				this.setDigitalDValue(channel, value);
			}
		}
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#updateDifferentialAI(int)
	 */
	public void updateDifferentialAI(int channel) throws LabJackException {
		if (channel < 0 || channel >= LabJackConstants.ANALOG_DIFFERENTIAL_INPUT_CHANNELS) {
			throw new LabJackException(21);
		}

		synchronized (thisClass) {
			float value = this.apiEAnalogIn(this.getSerialNumber(), LabJackConstants.ANALOG_SINGLE_INPUT_CHANNELS + channel, this.getDifferentialAIGain(channel));
			this.analogInputValues[LabJackConstants.ANALOG_SINGLE_INPUT_CHANNELS + channel] = value;
		}
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#updateEnvironmentals()
	 */
	public void updateEnvironmentals() throws LabJackException {
		synchronized (thisClass) {
			this.apiSHT1X(this.getSerialNumber());
		}
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#updateHighAIs()
	 */
	public void updateHighAIs() throws LabJackException {
		synchronized (thisClass) {
			long[] channels = { 4, 5, 6, 7 };
			long[] gains = { 0, 0, 0, 0 };
			float[] voltages = this.apiAISample(getSerialNumber(), 0, 0, 1, 4, channels, gains, 0);
			for (int i = 0; i < channels.length; i++) {
				this.analogInputValues[(int) channels[i]] = voltages[i];
			}
		}
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#updateIO(int)
	 */
	public void updateIO(int channel) throws LabJackException {
		if (channel < 0 || channel >= LabJackConstants.IO_CHANNELS) {
			throw new LabJackException(21);
		}

		synchronized (thisClass) {
			if (isIOForOutput(channel)) {
				this.apiEDigitalOut(this.getSerialNumber(), channel, false, this.getDigitalIOValue(channel));
			} else {
				boolean value = this.apiEDigitalIn(this.getSerialNumber(), channel, false);
				this.setDigitalIOValue(channel, value);
			}
		}
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#updateLowAIs()
	 */
	public void updateLowAIs() throws LabJackException {
		synchronized (thisClass) {
			long[] channels = { 0, 1, 2, 3 };
			long[] gains = { 0, 0, 0, 0 };
			float[] voltages = this.apiAISample(getSerialNumber(), 0, 0, 1, 4, channels, gains, 0);
			for (int i = 0; i < channels.length; i++) {
				this.analogInputValues[(int) channels[i]] = voltages[i];
			}
		}
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#updateScanFromBurst(long[], float, int, int, boolean, boolean, boolean, long, boolean)
	 */
	public float updateScanFromBurst(long[] channels, float scanRate, int scans, int timeout, boolean updateIO, boolean triggerIO, boolean triggerState, long transferMode, boolean ledOn)
		throws LabJackException {
	   float actualScanRate;
		synchronized (thisClass) {
			LabJackScan newScan = new LabJackScan();
			actualScanRate = this.apiAIBurst(
				getSerialNumber(),
				(this.digitalIOValues & this.digitalIODirections),
				(updateIO ? 1 : 0),
				(ledOn ? 1 : 0),
				channels.length,
				channels,
				this.getGains(channels),
				scanRate,
				0,
				(triggerIO ? 1 : 0),
				(triggerState ? 1 : 0),
				scans,
				timeout,
				transferMode,
				newScan);
			this.scan = newScan;
		}
		return actualScanRate;
	}

	private long[] getGains(long[] channels) {
		long[] gains = new long[channels.length];
		for (int i = 0; i < channels.length; i++) {
			if (channels[i] >= LabJackConstants.ANALOG_SINGLE_INPUT_CHANNELS) {
				gains[i] = this.analogInputGains[(int) channels[i] - LabJackConstants.ANALOG_SINGLE_INPUT_CHANNELS];
			}
		}
		return gains;
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#updateScanFromBurst(long[], float, int scans)
	 */
	public float updateScanFromBurst(long[] channels, float scanRate, int scans) throws LabJackException {
		return this.updateScanFromBurst(channels, scanRate, scans, 10, false, false, false, 0, true);
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#updateIO(int)
	 */
	public void updateLocalId() throws LabJackException {
		synchronized (thisClass) {
			this.apiLocalID(this.getSerialNumber(), this.getLocalId());
		}
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#reset()
	 */
	public void reset() throws LabJackException {
		synchronized (thisClass) {
			this.apiReset(this.getSerialNumber());
		}
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#reenumerate()
	 */
	public void reenumerate() throws LabJackException {
		synchronized (thisClass) {
			this.apiReEnum(this.getSerialNumber());
		}
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#updateFirmwareVersion()
	 */
	public void updateFirmwareVersion() throws LabJackException {
		synchronized (thisClass) {
			this.setFirmwareVersion(this.apiGetFirmwareVersion(this.getSerialNumber()));
		}
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#setPulseWidth(float)
	 */
	public void setPulseWidth(float pulseWidth) {
		float bestGuess = Float.MAX_VALUE;
		for (int b = 1; b < 256; b++) {
			for (int c = 1; c < 256; c++) {
				float thisGuess = this.calcPulseWidth(b, c);
				if (Math.abs(pulseWidth - thisGuess) < bestGuess) {
					this.pulseB1 = b;
					this.pulseC1 = c;
					bestGuess = Math.abs(pulseWidth - thisGuess);
				}
			}
		}
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#setPulseSpace(float)
	 */
	public void setPulseSpace(float pulseSpace) {
		float bestGuess = Float.MAX_VALUE;
		for (int b = 1; b < 256; b++) {
			for (int c = 1; c < 256; c++) {
				float thisGuess = this.calcPulseSpace(b, c);
				if (Math.abs(pulseSpace - thisGuess) < bestGuess) {
					this.pulseB2 = b;
					this.pulseC2 = c;
					bestGuess = Math.abs(pulseSpace - thisGuess);
				}
			}
		}
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#getPulseWidth()
	 */
	public float getPulseWidth() {
		return this.calcPulseWidth(this.pulseB1, this.pulseC1);
	}

	/* (non-Javadoc)
	 * @see com.teravation.labjack.LabJack#getPulseSpace()
	 */
	public float getPulseSpace() {
		return this.calcPulseSpace(this.pulseB2, this.pulseC2);
	}

	private float calcPulseWidth(int b, int c) {
		return 17.0f + (0.83f * c) + (20.17f * b * c);
	}

	private float calcPulseSpace(int b, int c) {
		return 12.0f + (0.83f * c) + (20.17f * b * c);
	}

	private int getBitSelect(boolean[] dChannels) {
		int bitSelect = 0;
		for (int i = 0; i < Math.min(8, dChannels.length); i++) {
			bitSelect <<= 1;
			if (dChannels[i]) {
				bitSelect += 1;
			}
		}
		return bitSelect;
	}

	private void updatePulse(int numPulses, int bitSelect, boolean lowFirst) throws LabJackException {
		synchronized (thisClass) {
			this.apiPulseOut(this.getSerialNumber(), lowFirst ? 0 : 1, bitSelect, numPulses, this.pulseB1, this.pulseC1, this.pulseB2, this.pulseC2);
		}
	}

	public void updatePulse(int dChannel, int numPulses) throws LabJackException {
		this.updatePulse(numPulses, 1 << dChannel, true);
	}

	public void updatePulse(boolean[] dChannels, int numPulses) throws LabJackException {
		this.updatePulse(numPulses, this.getBitSelect(dChannels), true);
	}

	public void updatePulse(boolean[] dChannels, int numPulses, boolean lowFirst) throws LabJackException {
		this.updatePulse(numPulses, this.getBitSelect(dChannels), lowFirst);
	}

	private void updatePulseAsynch(int numPulses, int bitSelect, boolean lowFirst) throws LabJackException {
		synchronized (thisClass) {
			this.apiPulseOutStart(this.getSerialNumber(), lowFirst ? 0 : 1, bitSelect, numPulses, this.pulseB1, this.pulseC1, this.pulseB2, this.pulseC2);
		}
	}

	public void updatePulseAsynch(int dChannel, int numPulses) throws LabJackException {
		this.updatePulseAsynch(numPulses, 1 << dChannel, true);
	}

	public void updatePulseAsynch(boolean[] dChannels, int numPulses) throws LabJackException {
		this.updatePulseAsynch(numPulses, this.getBitSelect(dChannels), true);
	}

	public void updatePulseAsynch(boolean[] dChannels, int numPulses, boolean lowFirst) throws LabJackException {
		this.updatePulseAsynch(numPulses, this.getBitSelect(dChannels), lowFirst);
	}

	public void updatePulseAsynchFinish(int timeoutMS) throws LabJackException {
		synchronized (thisClass) {
			this.apiPulseOutFinish(this.getSerialNumber(), timeoutMS);
		}
	}

	public float updateStreamStart(long[] channels, float scanRate, boolean readCount) throws LabJackException {
		float actualScanRate;
		synchronized (thisClass) {
			actualScanRate = this.apiAIStreamStart(this.getSerialNumber(), 0, 0, 1, channels.length, channels, this.getGains(channels), scanRate, 0, readCount?1:0);
		}
		return actualScanRate;
	}
	
	public float updateStreamStart(long[] channels, float scanRate) throws LabJackException {
		return this.updateStreamStart(channels, scanRate, false);
	}
	
	public void updateStreamClear() throws LabJackException {
		synchronized (thisClass) {
			this.apiAIStreamClear(this.getLocalId());
		}
	}
	
	public void updateScanFromStream(int scans) throws LabJackException {
		this.updateScanFromStream(scans, 10);
	}
	
   public void updateScanFromStream(int scans, int timeout) throws LabJackException {
		LabJackScan newScan = new LabJackScan();
   	synchronized (thisClass) {
         this.apiAIStreamRead(this.getLocalId(), scans, timeout, newScan);
	   }
	   this.scan = newScan;
   }
}
